﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <Commands/CommandLoop.h>
namespace nnt
{
    namespace abuse
    {

        bool CommandStartLoop::parseIterator(const ScriptArg& arg, const CommandVector& commands)
        {
            maxItr = ParseIteratorValue(arg.argValue, commands);
            deleteMaxItr = false;

            return maxItr != nullptr;
        }

        bool CommandStartLoop::parseName(const ScriptArg& arg, const CommandVector& commands)
        {
            for(BaseCommand* command: commands)
            {
                if(command->GetCommandType() == COMMAND_START_LOOP)
                {
                    CommandStartLoop* loopCommand = (CommandStartLoop*) command;
                    if(loopCommand->loopFinishedIndex == 0 && loopCommand->itrName == arg.argValue)
                        return false;
                }
            }
            itrName = arg.argValue;
            return true;
        }

        CommandStartLoop::CommandStartLoop() : BaseCommand(), loopFinishedIndex(0), curItr(0), maxItr(0), itrName(""), deleteMaxItr(false) {};
        CommandStartLoop::~CommandStartLoop()
        {
            if(deleteMaxItr)
                Platform::Free(maxItr);
        };

        bool CommandStartLoop::Execute(unsigned& nextCommand)
        {
            ++curItr;
            if(*maxItr == 0xFFFFFFFF)
                ++nextCommand;
            else if(curItr > *maxItr)
            {
                curItr = 0;
                nextCommand = loopFinishedIndex;
            }
            else
            {
                ++nextCommand;
            }
            return true;
        }

        ParseResult CommandStartLoop::ParseArgs(const ArgVector& args, CommandVector& commands)
        {
            if(args.size() > 2)
                return ParseResult::PARSE_INVALID_ARG;

            if(args.size() == 0)
            {
                maxItr = (unsigned*) Platform::Allocate(sizeof(unsigned));
                deleteMaxItr = true;
                *maxItr = 0xFFFFFFFF;
            }
            else if(args.size() == 1)
            {
                //This is an anonymous, forever loop
                if(args[0].argValue[0] == '{')
                {
                    bool parsed = parseIterator(args[0], commands);
                    if(parsed)
                        return ParseResult::PARSE_SUCCESS;
                    else
                        return ParseResult::PARSE_INVALID_ARG;
                }

                //This is either a named loop, or does not loop forever
                maxItr = (unsigned*)Platform::Allocate(sizeof(unsigned));
                deleteMaxItr = true;
                //Loop either has a name or max value set, but not both
                if(!ParseUnsigned(maxItr, args[0].argValue) )
                {
                    //argValue was not an int - this must be a named forever loop
                    *maxItr = 0xFFFFFFFF;
                    bool nameParsed = parseName(args[0], commands);
                    if(!nameParsed)
                        return ParseResult::PARSE_INVALID_ARG;
                }
            }
            else
            {
                bool nameParsed = parseName(args[0], commands);

                if(!nameParsed)
                    return ParseResult::PARSE_INVALID_ARG;

                if(args[1].argValue[0] == '{')
                {
                    bool parsed = parseIterator(args[1], commands);
                    if(parsed)
                        return ParseResult::PARSE_SUCCESS;
                    else
                        return ParseResult::PARSE_INVALID_ARG;
                }

                maxItr = (unsigned*)Platform::Allocate(sizeof(unsigned));
                deleteMaxItr = true;
                //Loop either has a name or max value set, but not both
                if(!ParseUnsigned(maxItr, args[1].argValue) )
                {
                    return ParseResult::PARSE_INVALID_ARG;
                }

            }
            return ParseResult::PARSE_SUCCESS;
        }

        bool CommandEndLoop::Execute(unsigned& nextCommand)
        {
            nextCommand = startLoopIndex;
            return true;
        }

        ParseResult CommandEndLoop::ParseArgs(const ArgVector& args, CommandVector& commandList)
        {
            if(args.size() != 0)
                return ParseResult::PARSE_INVALID_ARG;

            int i = (int)commandList.size() - 1;

            for(; i >= 0; --i)
            {
                BaseCommand* command = commandList[i];
                if(command->GetCommandType() == COMMAND_START_LOOP)
                {
                    CommandStartLoop* startLoop = (CommandStartLoop*)command;
                    if(startLoop->loopFinishedIndex != 0)
                        continue;
                    else
                    {
                        startLoop->loopFinishedIndex = (unsigned)(commandList.size() + 1);
                        startLoopIndex = i;
                        return ParseResult::PARSE_SUCCESS;
                    }
                }
            }
            return ParseResult::PARSE_UNMATCHED_COMMAND;
        }
    }
}
