﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <list>
#include <Log.h>
#include <AbuseString.h>

namespace nnt
{
    namespace abuse
    {
        enum InitStatus
        {
            INIT_ERROR = -1,
            INIT_OK = 0,
        };

        enum StartStatus
        {
            START_ERROR = -1,
            START_OK = 0,
            START_RESOURCE_DENIED = 1,
        };

        enum RunStatus
        {
            RUN_KILL = -2,
            RUN_ERROR = -1,
            RUN_OK = 0,
            RUN_YIELD = 1,
        };

        enum StopStatus
        {
            STOP_ERROR = -1,
            STOP_OK = 0,
        };

        enum ShutdownStatus
        {
            SHUTDOWN_ERROR = -1,
            SHUTDOWN_OK = 0,
        };

        struct ThreadInfo
        {
            ThreadInfo(nn::os::ThreadType* _thread, int _processor) : thread(_thread), processor(_processor) {};
            nn::os::ThreadType* thread;
            int processor;
        };

        typedef std::list<ThreadInfo, PlatformAllocator<ThreadInfo>> ListThread;
        class BaseTask
        {
            friend class Abuse;
        public:
            /*
                These are the functions that need to be implemented on a per-task basis
            */
            virtual ~BaseTask();
            virtual InitStatus Initialize(const String& params) = 0;
            virtual StartStatus Start() = 0;
            virtual RunStatus Run() = 0;
            virtual StopStatus Stop() = 0;
            virtual ShutdownStatus Shutdown() = 0;

            //Optional - default returns ""
            virtual const char* GetParamOptions();


            /*
                These are helper/utiliy functions that are available for each task.
            */

            //m_typename is the name of the task type (i.e. HelloTask)
            //name is the name of this instance of the task.
            //i.e. CREATE TASK=HelloTask NAME=hello1
            //       CREATE TASK=HelloTask NAME=hello2
            //would create 2 HelloTasks, names hello1 and hello2.
            BaseTask(const String& typeName, const String& instanceName);
            int GetId();
            const String& GetTypeName();
            const String& GetName();

            void LogVerbose(const char* format, ...);
            void LogInfo(const char* format, ...);
            void LogWarning(const char* format, ...);
            void LogError(const char* format, ...);

            void FlushLog(bool crashing);
            Log* GetLog();

            bool IsInitialized();
            void SetInitialized();

            bool IsRunning();
            void SetRunning(bool running);

            void SetCanRun(bool canRun);
            bool CanRun();

            bool IsSuspended();
            void SetSuspended(bool suspended);

            bool IsShutdown();
            void SetShutdown();

            bool IsKilled();
            void SetKilled();

            void SetPriority(int priority);
            int GetBasePriority();
            int GetCurrentPriority();
            void RaisePriority();
            void ResetPriority();

            void SetProcessor(int);
            int GetProcessor();

            const String& GetParams();
            void  SetParams(const String& params);

            int Serialize(char* buffer, int bufferStart, int bufferSize);

            //If processor is not provided, this thread will run on the processor with the fewest running threads
            bool CreateThread(nn::os::ThreadType* thread, void (*threadFunc)(void*), void* arg, void* stack, int stackSize, int priority);
            bool CreateThread(nn::os::ThreadType* thread, void (*threadFunc)(void*), void* arg, void* stack, int stackSize, int priority, int processor);
            bool DestroyThread(nn::os::ThreadType* thread);

            static bool sOpenFiles(StringVector& fileNames, std::vector<File, PlatformAllocator<File>>& files, OpenOptions options);
            bool WaitNotRunning();

            template <typename Task>
            static BaseTask* CreateTask(const String& typeName, const String& instanceName)
            {
                Task* task = (Task*)Platform::Allocate(sizeof(Task));
                ::new (task) Task(typeName, instanceName);
                return task;
            }

        private:
            static int s_nextId;
            static bool s_initialized;

            ListThread m_threads;
            String m_name;
            String m_typename;
            String m_params;
            Log* m_log;
            nn::os::Tick m_startTime;
            nn::os::Tick m_runTime;
            int m_basePriority;
            int m_currentPriority;
            int m_id;
            int m_processor;
            bool m_isRunning;
            bool m_canRun;
            bool m_isSuspended;
            bool m_isShutdown;
            bool m_isKilled;
            bool m_isInitialized;
        };
    }
}
