﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <vector>
#include <nn/nn_Result.h>
#include <AbuseString.h>
#include <nn/os.h>
#include <nn/socket/socket_Api.h>
namespace nnt
{
    namespace abuse
    {
        template<typename> class PlatformAllocator;
        typedef unsigned File;
        typedef std::vector<String, PlatformAllocator<String> > StringVector;
        static const unsigned FILE_INVALID_HANDLE = 0xFFFFFFFF;
        static const int NUM_CORES = 4;

        struct OpenOptions
        {
            OpenOptions() : read(false), write(false), create(false) {}
            bool read;
            bool write;
            bool create;
        };

        class Platform
        {
        public:
            //
            //      SYSTEM INIT/SHUTDOWN
            //
            static void Initialize();
            static void Finalize();

            //
            //      FILE IO
            //

            //If OpenOptions.create is set, fileName and any intermediate directories will be created if they don't already exist.
            //File paths are relative to Platform::WorkingDirectory
            static const char* GetWorkingDirectory();
            static void        SetWorkingDirectory(const char* directory);
            static const char* GetScriptDirectory();
            static void        SetScriptDirectory(const char* directory);
            static const char* GetLogDirectory();
            static void        SetLogDirectory(const char* directory);

            static void GetFilesInDir(StringVector& vec, const char* directoryName, bool includeSubDirs);
            static File FileOpen(const char* fileName, OpenOptions options, bool absolutePath=false);
            static void FileClose(File file);
            static void FileWrite(File file, const char* buffer, int writeSize, bool append=true);
            static void FileRead(File file, int64_t offset, char* buffer, int bufferSize);
            static void FileDelete(File file);
            static int64_t GetFileSize(File file);
            static const int GetMaxPathLength();

            //
            //Sockets
            //
            //Enough of a wrapper to do a simple echo server in both socket and sokcet0
            static long GetAddrAny();
            static int InitializeSocketLib();
            static int FinalizeSocketLib();
            static int socket(nn::socket::Family family, nn::socket::Type type, nn::socket::Protocol protocol);

           // static int inet_pton(uint8_t family, const char* addressString, void* addressBuffer);
            static int connect(int descriptor, const nn::socket::SockAddrIn* addr, int sizeBytes);
            static nn::socket::Errno GetLastError();
            static int send(int descriptor, const void* buffer, size_t bufferByteSize, nn::socket::MsgFlag flags);
            static int recv(int descriptor, void* buffer, size_t bufferByteSize, nn::socket::MsgFlag flags);
            static int sendto(int descriptor, void* buffer,size_t bufferByteSize, nn::socket::MsgFlag flags,const nn::socket::SockAddrIn* to, int toByteSize);
            static int recvfrom(int descriptor, void* buffer,size_t bufferByteSize, nn::socket::MsgFlag flags, nn::socket::SockAddrIn* from, int* fromByteSize);

            static int shutdown(int descriptor, nn::socket::ShutdownMethod how);
            static int close(int descriptor);
            static int poll(nn::socket::PollFd* descriptors, size_t fdCount, int timeout);
            static int bind(int descriptor, const nn::socket::SockAddr* addr, int addrSize);
            static int listen(int descriptor, int backlogCount);
            static int accept(int descriptor, nn::socket::SockAddr* addr, int* addrSize);
            static int SetNonBlocking(int descriptor, bool nonblocking);
            static uint16_t htons(uint16_t hostshort);
            static uint32_t htonl(uint32_t hostlong);
            static uint32_t ntohl(uint32_t netlong);
            static uint64_t ntohll(uint64_t hostlonglong);
            static uint64_t htonll(uint64_t netlonglong);

            static uint16_t PutNetworkBuffer(char* buffer, uint16_t start, unsigned data);
            static uint16_t PutNetworkBuffer(char* buffer, uint16_t start, uint64_t data);

            //
            //      MEMORY
            //
            static void*  AllocateAligned(size_t size, int alignment);
            static void*  Allocate(size_t size);
            static void*  Reallocate(void * ptr,size_t size);
            static char*  StringDuplicate(const char* string);
            static void*  ClearAllocate(size_t num,size_t size);
            static void   Free(void* ptr);
            static void   Free(void* ptr, size_t size);
            static size_t GetHeapSize();

            //
            //      UTILITY
            //
            static int64_t GetStartTime();
            static int64_t GetSystemRunningTimeMilli();
            static int64_t GetPlatformTick();
            static int64_t GetPlatformTickFrequency();
            static double GetPlatformTimeSinceStart();
            static const char* GetPlatformName();
            static const int GetNumCores();
        };
    }
}

#ifdef WIN32
#include <PlatformAllocatorWin32.h>
#else
#include <PlatformAllocatorHorizon.h>
#endif

