﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <list>
#include <nn/os.h>
#include <nn/os/os_UserExceptionHandler.h>
#include <Tasks/BaseTask.h>
#include <AbuseString.h>
#include <Commands/CommandCommon.h>
namespace nnt
{
    namespace abuse
    {
        enum AbuseResource
        {
            RESOURCE_HEAP_SIZE = 0,
            RESOURCE_TLS_SLOTS,

            NUM_RESOURCES,
        };

        class BaseCommand;
        class ShellServer;
        class AbuseDriver;
        const size_t DRIVER_STACK_SIZE = 4096 * 8;
        const size_t CPU_BUSY_STACK_SIZE = 4096;

        class ScopedMutex
        {
        public:
            NN_IMPLICIT ScopedMutex(nn::os::MutexType* mutex);
            ~ScopedMutex();

        private:
            nn::os::MutexType* m_mutex;
        };

        typedef BaseTask* (*TaskCreateFunc) (const String&, const String&);

        struct RegisteredTask
        {
            RegisteredTask(const String& _typename, TaskCreateFunc createFunc)
                : typeName(_typename), Create(createFunc){}

            const String typeName;
            TaskCreateFunc Create;
        };

        struct ActiveTask
        {
            NN_IMPLICIT ActiveTask(BaseTask* newTask)
                : task(newTask){}
            BaseTask* task;
        };

        struct TaskCreateArgs
        {
            TaskCreateArgs() : typeName(""), instanceName(""), params(""), priority(nn::os::DefaultThreadPriority), processor(-1), load(0),
                verbosity(LogVerbosity::VerbosityUseGlobal), useTimestamps(SettingUseGlobal), forwardToConsole(SettingUseGlobal), startSuspended(false) {}
            String typeName;
            String instanceName;
            String params;

            //Thread info
            int priority;
            int processor;
            int load;

            //log info
            LogVerbosity verbosity;
            LogSetting useTimestamps;
            LogSetting forwardToConsole;

            bool startSuspended;
        };

        typedef std::list<RegisteredTask, PlatformAllocator<RegisteredTask>> ListRegisteredTask;
        typedef std::list<ActiveTask, PlatformAllocator<ActiveTask>> ListActiveTask;

        // args to pass into the cpu busy core thread function
        struct CpuBusyThreadValues {
            unsigned cpuBusyMs = 10;
            unsigned cpuBusyThreadDutyCyclePercent = 0;
        };

        void CpuBusyThreadFunc(void *args);

        class Abuse
        {
        public:
            static void Initialize(const String& configPath);
            static void Finalize();

            static nn::os::Tick GetStartTime();
            static int GetLeastUsedProcessor();

            static void ThreadRunningOnProcessor(int processor);
            static void ThreadStoppingOnProcessor(int processor);

            static bool CreateTask(const TaskCreateArgs& createArgs);
            static bool KillTask(int id);
            static bool KillTask(const String& name);
            static void KillAllTasks();
            static int  CleanupShutdownTasks();
            static bool SuspendTask(const String& name);
            static void SuspendAllTasks();
            static bool ResumeTask(const String& name);
            static void ResumeAllTasks();
            static bool RegisterTask(const String& _typename, TaskCreateFunc createFunction);

            static const char* GetStartingScriptName();

            static int RandRange(int min, int max);

            static bool     IsShellEnabled();
            static uint16_t GetListenPort();

            static void GeneralMutexLock();
            static void GeneralMutexUnlock();
            static void GeneralMutexTryLock();
            static ScopedMutex GeneralMutexScopedLock();

            static void StopRunning();
            static bool IsRunning();

            static unsigned GetTimeoutDuration();

            static void Run();

            static bool AcquireResource(BaseTask* task, AbuseResource resource);
            static void ReleaseResource(BaseTask* task, AbuseResource resource);

            static ListActiveTask* GetActiveTasksList();

        private:
            static void driverWorkerThread(void* baseTask);
            static void killTask(ActiveTask& activeTask);
            static bool setSuspended(const String& name, bool suspended);
            static void exceptionHandler(nn::os::UserExceptionInfo* info);
            static void addTask(BaseTask* task, bool acquireLock);
            static void updateScripts();
            static void updateShell();
            static void clearScriptCommands();

            static Log*                 s_log;
            static String              s_activeScript;
            static BaseCommand*        s_shellCommand;
            static CommandVector       s_commands;
            static ShellServer*        s_shell;
            static unsigned            s_commandIndex;
            static AbuseDriver*        s_taskDrivers;
            static int64_t*            s_heartbeats;
            static int*                s_tasksOnProcessor;
            static BaseTask*           s_resourceOwner[NUM_RESOURCES];
            static void*               s_exceptionStack;
            static nn::os::UserExceptionInfo* s_exceptionInfo;
            static unsigned            s_timeoutMillis;
            static nn::os::MutexType   s_generalMutex;
            static nn::os::MutexType   s_taskMutex;
            static nn::os::MutexType   s_coreMutex;
            static nn::os::MutexType   s_resourceMutex;
            static nn::os::MutexType   s_threadMutex;
            static nn::os::BarrierType s_driverBarrier;
            static nn::os::Tick        s_startTime;
            static ListRegisteredTask  s_registeredTasks;
            static ListActiveTask      s_activeTasks;
            static char*               s_scriptName;
            static int                 s_numRunningDrivers;
            static unsigned            s_taskRuntimeMillis;
            static uint16_t            s_listenPort;
            static bool                s_isShellEnabled;
            static bool                s_isRunning;
            static bool                s_resourceOwned[NUM_RESOURCES];
            static unsigned            s_cpuBusyCore;
            static int                 s_cpuBusyThreadPriority;
            static unsigned            s_cpuBusyMs;
            static unsigned            s_cpuBusyThreadDutyCyclePercent;
        };
    }
}
