﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestSdmmc.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <cstring>

#ifdef SUPPORT_SDCARD_TEST

namespace
{
    const uint64_t SpaceAddr = 0x0;

#ifdef SUPPORT_MMC_TEST
    const nn::sdmmc1::Port TestPort = nn::sdmmc1::Port_Mmc0;    // sdmmc4a
#else
    const nn::sdmmc1::Port TestPort = nn::sdmmc1::Port_SdCard0; // sdmmc3a
#endif

    const uint32_t TestSectorIndex = 0;
    const uint32_t TestSize = 32 * 1024;
    const uint32_t TestNumSectors = TestSize / nn::sdmmc1::SectorSize;
    NN_ALIGNAS(4096) uint8_t s_Buffer[TestSize];
}
extern "C" void nnMain();

TEST(sMMU, Test0)
{
    nn::Result result;
    nn::svc::Handle handle;
    uint64_t deviceAddress = 0;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, 0x80000000);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 130-10
    // 複数のデバイスが１つのアドレススペースを共有することが出来る
    result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);

#ifdef SUPPORT_MMC_TEST
    result = nn::svc::AttachDeviceAddressSpace(DeviceName_Mmc, handle);
    ASSERT_RESULT_SUCCESS(result);
#endif // SUPPORT_MMC_TEST

    // TEST 132-45
    // devicePermission は MemoryPermission_ReadWrite を受け付ける
    result = nn::svc::MapDeviceAddressSpaceByForce(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uint64_t>(s_Buffer), sizeof(s_Buffer), deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    Sdmmc sdmmc(s_Buffer, sizeof(s_Buffer), deviceAddress, TestPort);
    if (sdmmc.Activate())
    {
        std::memset(s_Buffer, 0, sizeof(s_Buffer));
        result = nn::sdmmc1::Read(s_Buffer, sizeof(s_Buffer), TestPort, TestSectorIndex, TestNumSectors);
        ASSERT_RESULT_SUCCESS(result);

#if 0
        for (int i = 0; i < sizeof(s_Buffer); i += 16)
        {
            NN_LOG("%08x: %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x\n", i,
                    s_Buffer[i + 0], s_Buffer[i + 1], s_Buffer[i + 2], s_Buffer[i + 3], s_Buffer[i + 4], s_Buffer[i + 5],
                    s_Buffer[i + 6], s_Buffer[i + 7], s_Buffer[i + 8], s_Buffer[i + 9], s_Buffer[i + 10], s_Buffer[i + 11],
                    s_Buffer[i + 12], s_Buffer[i + 13], s_Buffer[i + 14], s_Buffer[i + 15]);
        }
#endif

    }

#ifdef SUPPORT_MMC_TEST
    result = nn::svc::DetachDeviceAddressSpace(DeviceName_Mmc, handle);
    ASSERT_RESULT_SUCCESS(result);
#endif // SUPPORT_MMC_TEST

    result = nn::svc::DetachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(sMMU, Test1)
{
    nn::Result result;
    nn::svc::Handle handle;
    uint64_t deviceAddress = 0;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, 0x80000000);
    ASSERT_RESULT_SUCCESS(result);

#ifdef SUPPORT_MMC_TEST
    result = nn::svc::AttachDeviceAddressSpace(DeviceName_Mmc, handle);
    ASSERT_RESULT_SUCCESS(result);
#endif // SUPPORT_MMC_TEST

    result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetMemoryAttribute(reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer),
            nn::svc::MemoryAttribute_Uncached, nn::svc::MemoryAttribute_Uncached);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapDeviceAddressSpaceByForce(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

#ifdef SUPPORT_MMC_TEST
    result = nn::svc::DetachDeviceAddressSpace(DeviceName_Mmc, handle);
    ASSERT_RESULT_SUCCESS(result);
#endif // SUPPORT_MMC_TEST

    result = nn::svc::DetachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetMemoryAttribute(reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer),
            nn::svc::MemoryAttribute_Uncached, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}


TEST(sMMU, Test2)
{
    nn::Result result;
    nn::svc::Handle handle;
    uint64_t deviceAddress = 0;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, 0x80000000);
    ASSERT_RESULT_SUCCESS(result);

#ifdef SUPPORT_MMC_TEST
    result = nn::svc::AttachDeviceAddressSpace(DeviceName_Mmc, handle);
    ASSERT_RESULT_SUCCESS(result);
#endif // SUPPORT_MMC_TEST

    result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 132-43
    // devicePermission は MemoryPermission_Write を受け付ける
    result = nn::svc::MapDeviceAddressSpaceByForce(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), deviceAddress,
            nn::svc::MemoryPermission_Write);
    ASSERT_RESULT_SUCCESS(result);

    Sdmmc sdmmc(s_Buffer, sizeof(s_Buffer), deviceAddress, TestPort);
    if (sdmmc.Activate())
    {
        std::memset(s_Buffer, 0, sizeof(s_Buffer));
        result = nn::sdmmc1::Read(s_Buffer, sizeof(s_Buffer), TestPort, TestSectorIndex, TestNumSectors);
        ASSERT_RESULT_SUCCESS(result);
    }

#ifdef SUPPORT_MMC_TEST
    result = nn::svc::DetachDeviceAddressSpace(DeviceName_Mmc, handle);
    ASSERT_RESULT_SUCCESS(result);
#endif // SUPPORT_MMC_TEST

    result = nn::svc::DetachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(sMMU, Test3)
{
    nn::Result result;
    nn::svc::Handle handle;
    uint64_t deviceAddress = 0;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, 0x80000000);
    ASSERT_RESULT_SUCCESS(result);

#ifdef SUPPORT_MMC_TEST
    result = nn::svc::AttachDeviceAddressSpace(DeviceName_Mmc, handle);
    ASSERT_RESULT_SUCCESS(result);
#endif // SUPPORT_MMC_TEST

    result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 132-42
    // devicePermission は MemoryPermission_Read を受け付ける
    result = nn::svc::MapDeviceAddressSpaceByForce(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), deviceAddress,
            nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);

    Sdmmc sdmmc(s_Buffer, sizeof(s_Buffer), deviceAddress, TestPort);
    if (sdmmc.Activate())
    {
        NN_LOG("sMMU エラー割り込みが発生します for %p\n", s_Buffer);
        std::memset(s_Buffer, 0, sizeof(s_Buffer));
        result = nn::sdmmc1::Read(s_Buffer, sizeof(s_Buffer), TestPort, TestSectorIndex, TestNumSectors);
        ASSERT_RESULT_SUCCESS(result);
    }

#ifdef SUPPORT_MMC_TEST
    result = nn::svc::DetachDeviceAddressSpace(DeviceName_Mmc, handle);
    ASSERT_RESULT_SUCCESS(result);
#endif // SUPPORT_MMC_TEST

    result = nn::svc::DetachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(sMMU, Test4)
{
    nn::Result result;
    nn::svc::Handle handle;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, 0x80000000);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 130-8
    // 既にアタッチしているデバイスを再度アタッチすることは出来ない
    result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultBusy());

    // TEST 131-10
    // 既にデタッチしているハンドルをデタッチすることは出来ない
    result = nn::svc::DetachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::DetachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());

    result = nn::svc::UnmapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), reinterpret_cast<uint64_t>(s_Buffer));
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

#ifdef SVC_AGING_TEST
// TEST 129-17
// DetachAddressSpace を呼ばなくても、ハンドルをクローズすれば、リソースリークは発生しない
TEST(sMMU, Test5)
{
    nn::Result result;
    nn::svc::Handle handle;
    uint64_t deviceAddress = 0;

    for (int i = 0; i < 2u * 1024 * 1024 * 1024 / 4096; i++)
    {
        result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, 0x80000000);
        ASSERT_RESULT_SUCCESS(result);

#ifdef SUPPORT_MMC_TEST
        result = nn::svc::AttachDeviceAddressSpace(DeviceName_Mmc, handle);
#else
        result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
#endif // SUPPORT_MMC_TEST
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::MapDeviceAddressSpaceByForce(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), deviceAddress,
                nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::UnmapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                reinterpret_cast<uintptr_t>(s_Buffer), sizeof(s_Buffer), deviceAddress);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
}
#endif


TEST(sMMU, Test6)
{
    uintptr_t addr;
    size_t size;

    nn::Result result;
    nn::svc::Handle handle;
    uint64_t deviceAddress = 0;

    size = 0x01000000;
    uintptr_t origHeapAddr;
    result = nn::svc::SetHeapSize(&origHeapAddr, size);
    ASSERT_RESULT_SUCCESS(result);
    addr = (origHeapAddr + 0x00400000 - 1) & ~(0x00400000 - 1);
    size -= addr - origHeapAddr;

    nn::svc::PhysicalMemoryInfo physInfo = { 0 };
    result = nn::svc::QueryPhysicalAddress(&physInfo, addr);
    ASSERT_RESULT_SUCCESS(result);
    // 4MBアラインされるはず
    ASSERT_TRUE(((physInfo.physicalAddress + (addr - physInfo.virtualAddress)) & (0x00400000 - 1)) == 0);

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, 0xf0000000);
    ASSERT_RESULT_SUCCESS(result);

#ifdef SUPPORT_MMC_TEST
    result = nn::svc::AttachDeviceAddressSpace(DeviceName_Mmc, handle);
    ASSERT_RESULT_SUCCESS(result);
#endif // SUPPORT_MMC_TEST

    result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapDeviceAddressSpaceByForce(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr + 0x1000, size - 0x2000, deviceAddress + 0x1000,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    {
        Sdmmc sdmmc(reinterpret_cast<void*>(addr + 0x1000), size - 0x2000, deviceAddress + 0x1000, TestPort);
        if (sdmmc.Activate())
        {
            std::memset(reinterpret_cast<void*>(addr), 0, size);
            result = nn::sdmmc1::Read(reinterpret_cast<void*>(addr + 0x1000), size - 0x2000, TestPort, TestSectorIndex, TestNumSectors);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

#ifdef SUPPORT_MMC_TEST
    result = nn::svc::DetachDeviceAddressSpace(DeviceName_Mmc, handle);
    ASSERT_RESULT_SUCCESS(result);
#endif // SUPPORT_MMC_TEST

    result = nn::svc::DetachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr + 0x1000, size - 0x2000, deviceAddress + 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    size = 0;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);
}

#ifdef SVC_AGING_TEST
TEST(sMMU, Test7)
{
    uintptr_t addr;
    size_t size;

    nn::Result result;
    nn::svc::Handle handle;
    uint64_t deviceAddress = 0;

    size = 0x01000000;
    uintptr_t origHeapAddr;
    result = nn::svc::SetHeapSize(&origHeapAddr, size);
    ASSERT_RESULT_SUCCESS(result);
    addr = (origHeapAddr + 0x00400000 - 1) & ~(0x00400000 - 1);
    size -= addr - origHeapAddr;

    nn::svc::PhysicalMemoryInfo physInfo = { 0 };
    result = nn::svc::QueryPhysicalAddress(&physInfo, addr);
    ASSERT_RESULT_SUCCESS(result);
    // 4MBアラインされるはず
    ASSERT_TRUE(((physInfo.physicalAddress + (addr - physInfo.virtualAddress)) & (0x00400000 - 1)) == 0);

    for (int i = 0; i < 2u * 1024 * 1024 * 1024 / 4096; i++)
    {
        result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, 0xf0000000);
        ASSERT_RESULT_SUCCESS(result);

#ifdef SUPPORT_MMC_TEST
        result = nn::svc::AttachDeviceAddressSpace(DeviceName_Mmc, handle);
        ASSERT_RESULT_SUCCESS(result);
#endif // SUPPORT_MMC_TEST

        result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::MapDeviceAddressSpaceByForce(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr + 0x1000, size - 0x2000, deviceAddress + 0x1000,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

#ifdef SUPPORT_MMC_TEST
        result = nn::svc::DetachDeviceAddressSpace(DeviceName_Mmc, handle);
        ASSERT_RESULT_SUCCESS(result);
#endif // SUPPORT_MMC_TEST

        result = nn::svc::DetachDeviceAddressSpace(DeviceName_SdCard, handle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::UnmapDeviceAddressSpace(handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr + 0x1000, size - 0x2000, deviceAddress + 0x1000);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }

    size = 0;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);
}
#endif

#endif // SUPPORT_SDCARD_TEST

