﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tick.h>
#include <nn/svc/svc_HardwareParamsSelect.h>

extern "C" void nnMain();

namespace {
nn::Bit32 g_Value;
char g_Stack[DefaultStackSize] __attribute__((aligned(0x1000)));
int32_t g_Sequence;

void SignalThread(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;

    g_Sequence = 1;

    nn::svc::SleepThread(100 * 1000 * 1000);

    g_Sequence = 2;

    nn::svc::SignalProcessWideKey(0, 1);
}

} // namespace

TEST(WaitProcessWideKeyAtomic, AddressTest)
{
    nn::Result result;
    size_t align = 4;
    uintptr_t key = 0xdeadbeaf;
    size_t size = 0x1000;

    TestHeap heap(HeapAlign);
    uintptr_t addr = heap.GetAddress();
    nn::Bit32* ptr = reinterpret_cast<nn::Bit32*>(addr);

    ASSERT_TRUE(addr % align == 0);

    // TEST 138-1
    // 4 byte アライメントされていると成功する
    for (uintptr_t i = 0; i < 0x100; i += align)
    {
        ptr = reinterpret_cast<nn::Bit32*>(addr + i);
        *ptr = 1;
        result = nn::svc::WaitProcessWideKeyAtomic(addr + i, key, 1, 1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        ASSERT_TRUE(*ptr == 0);
    }

    // TEST 138-2
    // 4 byte アライメントされていないと失敗する
    for (uintptr_t i = 1; i < align; i++)
    {
        ptr = reinterpret_cast<nn::Bit32*>(addr + i);
        *ptr = 1;
        result = nn::svc::WaitProcessWideKeyAtomic(addr + i, key, 1, 1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
        ASSERT_TRUE(*ptr == 1);
    }

    ptr = reinterpret_cast<nn::Bit32*>(addr);
    *ptr = 1;

    // TEST 138-3
    // MemoryPermission_None の領域は受け付けない
    {
        TestMemoryPermission perm(addr, size, nn::svc::MemoryPermission_None);
        result = nn::svc::WaitProcessWideKeyAtomic(addr, key, 1, 1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }
    ASSERT_TRUE(*ptr == 1);

    // TEST 138-10
    // MemoryPermission_Read の領域は受け付けない
    {
        TestMemoryPermission perm(addr, size, nn::svc::MemoryPermission_Read);
        result = nn::svc::WaitProcessWideKeyAtomic(addr, key, 1, 1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }
    ASSERT_TRUE(*ptr == 1);

    // TEST 138-13
    // MemoryPermission_ReadWrite の領域を受け付ける
    {
        TestMemoryPermission perm(addr, size, nn::svc::MemoryPermission_ReadWrite);
        result = nn::svc::WaitProcessWideKeyAtomic(addr, key, 1, 1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        ASSERT_TRUE(*ptr == 0);
        *ptr = 1;
    }

    // TEST 138-14
    // MemoryPermission_ReadExecute の領域は受け付けない
    {
        addr = reinterpret_cast<uintptr_t>(nnMain);
        addr = (addr + (align - 1)) & ~(align - 1);
        result = nn::svc::WaitProcessWideKeyAtomic(addr, key, 1, 1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }
    ASSERT_TRUE(*ptr == 1);
}

TEST(WaitProcessWideKeyAtomic, KeyTest)
{
    nn::Result result;
    uintptr_t addr = reinterpret_cast<uintptr_t>(&g_Value);
    uintptr_t ValidKey[] = {
        0, 1, 0x1000, 0xFFFF, 0x10000, 0xFFFFFFFF, static_cast<uintptr_t>(-1)
    };

    // TEST 138-9
    // uintptr_t の値を受け付ける
    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(ValidKey) / sizeof(uintptr_t)); i++)
    {
        g_Value = 1;
        result = nn::svc::WaitProcessWideKeyAtomic(addr, ValidKey[i], 0, 1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        ASSERT_TRUE(g_Value == 0);
    }
}

TEST(WaitProcessWideKeyAtomic, NewValueTest)
{
    nn::Result result;
    uintptr_t addr = reinterpret_cast<uintptr_t>(&g_Value);
    nn::Bit32 ValidValue[] = {0, 1, 0x1000, 0xFFFF, 0x10000, 0xFFFFFFFF};

    // TEST 138-10
    // Bit32 の値を受け付ける
    for (int32_t i = 0; i < static_cast<int32_t>(sizeof(ValidValue) / sizeof(nn::Bit32)); i++)
    {
        g_Value = 1;
        result = nn::svc::WaitProcessWideKeyAtomic(addr, 0, ValidValue[i], 1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        ASSERT_TRUE(g_Value == 0);
    }
}

TEST(WaitProcessWideKeyAtomic, TimeTest)
{
    nn::Result result;
    uintptr_t addr = reinterpret_cast<uintptr_t>(&g_Value);

    // TEST 138-11
    // 正の値を与えると、与えた分以上の時間を過ぎるとタイムアウトする
    for (int64_t ns = 1; ns < 10 * 1000ll * 1000ll * 1000ll; ns = ns * 2)
    {
        g_Value = 1;

        volatile int64_t before = nn::svc::GetSystemTick();
        result = nn::svc::WaitProcessWideKeyAtomic(addr, 0, 0, ns);
        volatile int64_t after = nn::svc::GetSystemTick();

        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        ASSERT_TRUE(g_Value == 0);
        ASSERT_TRUE(((after - before) / (NN_HW_TICKS_PER_SECOND / (1000 * 1000)) * 1000) >= ns);
    }

    // TEST 138-12
    // 負の値を与えると、タイムアウトをしない
    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(SignalThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
        uintptr_t param = 0;
        int32_t priority = TestLowestThreadPriority;
        int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

        g_Sequence = 0;
        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        g_Value = 0;
        result = nn::svc::WaitProcessWideKeyAtomic(addr, 0, 2, -1);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(g_Sequence == 2);
        ASSERT_TRUE(g_Value == 2);
    }

    // TEST 138-13
    // 0 を与えるとタイムアウトする
    {
        g_Value = 1;
        result = nn::svc::WaitProcessWideKeyAtomic(addr, 0, 0, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        ASSERT_TRUE(g_Value == 0);
    }
}
