﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/TargetConfigs/build_Compiler.h>
#include <cstring>

extern "C" void nnMain();

namespace {

} // namespace

TEST(UnmapProcessMemory, AddrTest)
{
    nn::Result result;

    TestProcess process(1);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    uintptr_t toAddr = g_FreeAreaBegin;
    uintptr_t fromAddr = process.GetCodeAddress();

    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 117-1
    // toAddr のアラインメントが4KBに揃っていると成功する
    // TEST 117-10
    // fromAddr のアラインメントが4KBに揃っていると成功する
    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 117-2
    // toAddr のアラインメントが4KBに揃っていないと失敗する
    for (uintptr_t i = toAddr + 1; i < toAddr + 0x1000; i++)
    {
        result = nn::svc::UnmapProcessMemory(i, processHandle, fromAddr, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }

    // TEST 117-11
    // fromAddr のアラインメントが4KBに揃っていないと失敗する
    for (uintptr_t i = fromAddr + 1; i < fromAddr + 0x1000; i++)
    {
        result = nn::svc::UnmapProcessMemory(toAddr, processHandle, i, size);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }

    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(UnmapProcessMemory, HandleTest)
{
    nn::Result result;

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    uintptr_t toAddr = g_FreeAreaBegin;
    uintptr_t fromAddr = process.GetCodeAddress();
    AssignExitCode(processHandle, process.GetCodeAddress(), size);

    // TEST 117-3
    // 実行前のプロセスのハンドルを受け付ける
    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartProcess(
            processHandle, TestLowestThreadPriority, g_ProcessIdealCore, DefaultStackSize);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 117-4
    // 実行中のプロセスのハンドルを受け付ける
    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    WaitProcess(processHandle);

    // TEST 117-5
    // 実行後のプロセスのハンドルを受け付ける
    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 117-6
    // プロセスの擬似ハンドルを受け付けない
    result = nn::svc::UnmapProcessMemory(
            toAddr, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 117-7
    // INVALID_HANDLE_VALUE を受け付けない
    result = nn::svc::UnmapProcessMemory(
            toAddr, nn::svc::INVALID_HANDLE_VALUE, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 117-8
    // スレッドの擬似ハンドルを受け付けない
    result = nn::svc::UnmapProcessMemory(
            toAddr, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 117-9
    // Close したプロセスのハンドルを受け付けない
    result = nn::svc::MapProcessMemory(
            toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(UnmapProcessMemory, SizeTest)
{
    nn::Result result;

    TestProcess process(1);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    uintptr_t toAddr = g_FreeAreaBegin;
    uintptr_t fromAddr = process.GetCodeAddress();

    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 117-12
    // サイズのアラインメントが4KBに揃っていると成功する
    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 117-13
    // サイズのアラインメントが4KBに揃っていないと失敗する
    for (size_t i = size + 1; i < size + 0x1000; i++)
    {
        result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, i);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
    }

    // TEST 117-14
    // サイズは 0 を受け付けない
    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(UnmapProcessMemory, ToAddrAreaTest)
{
    nn::Result result;

    TestProcess process(1);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    uintptr_t toAddr = g_FreeAreaBegin;
    uintptr_t fromAddr = process.GetCodeAddress();

    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 117-58
    // toAddr/size で示す領域が MemoryState_SharedCode だと成功する
    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 117-15
    // toAddr/size で示す領域が MemoryState_Free だと失敗する
    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 117-16
    // toAddr/size で示す領域が MemoryState_Normal だと失敗する
    //　廃止

    // TEST 117-17
    // toAddr/size で示す領域が MemoryState_Shared だと失敗する
    //　廃止

    // TEST 117-18
    // toAddr/size で示す領域が MemoryState_Aliased だと失敗する
    //　廃止

    // TEST 117-19
    // toAddr/size で示す領域が MemoryState_Alias だと失敗する
    //　廃止

    // TEST 117-20
    // toAddr/size で示す領域が MemoryState_Ipc だと失敗する
    //　廃止

    // TEST 117-21
    // toAddr/size で示す領域が MemoryState_Locked だと失敗する
    //　廃止

    // TEST 117-22
    // toAddr/size で示す領域が MemoryState_Io だと失敗する
    // 廃止

    // TEST 117-23
    // toAddr/size で示す領域が MemoryState_Static だと失敗する
    // 廃止

    // TEST 117-24
    // toAddr/size で示す領域が MemoryState_Code だと失敗する
    // 廃止

    // TEST 117-25
    // toAddr/size で示す領域が MemoryState_AliasedCode だと失敗する
    // 廃止

    // TEST 117-26
    // toAddr/size で示す領域が MemoryState_AliasCode だと失敗する
    // 廃止

    // TEST 117-27
    // toAddr/size で示す領域が MemoryState_Stack だと失敗する
    // 廃止

    // TEST 117-28
    // toAddr/size で示す領域が MemoryState_ThreadLocal だと失敗する
    // 廃止

    // TEST 117-29
    // toAddr/size で示す領域が MemoryState_Inaccessible だと失敗する
    // 廃止

    // TEST 117-30
    // toAddr/size で示す領域 のメモリ状態が一様でないと失敗する (前方)

    // TEST 117-31
    // toAddr/size で示す領域 のメモリ状態が一様でないと失敗する (後方)
    {
        result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size * 2);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
    }

    // TEST 117-32
    // オーバーフローする組み合わせを指定できない
    size = static_cast<size_t>(-0x1000);
    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // 終了処理
    size = 0x1000;
    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(UnmapProcessMemory, FromAddrAreaTest)
{
    nn::Result result;

    TestProcess process(1);
    nn::svc::Handle processHandle = process.GetHandle();

    size_t size = 0x1000;
    uintptr_t toAddr = g_FreeAreaBegin;
    uintptr_t fromAddr = process.GetCodeAddress();

    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 117-33
    // fromAddr/size で示す領域が MemoryState_Free だと失敗する
    // 廃止

    // TEST 117-34
    // fromAddr/size で示す領域が MemoryState_Normal だと失敗する
    // 廃止

    // TEST 117-35
    // fromAddr/size で示す領域が MemoryState_Shared だと失敗する
    // 廃止

    // TEST 117-36
    // fromAddr/size で示す領域が MemoryState_Aliased だと失敗する
    // 廃止

    // TEST 117-37
    // fromAddr/size で示す領域が MemoryState_Ipc だと失敗する
    // 廃止

    // TEST 117-38
    // fromAddr/size で示す領域が MemoryState_Locked だと失敗する
    // 廃止

    // TEST 117-39
    // fromAddr/size で示す領域が MemoryState_Io だと失敗する
    // 廃止

    // TEST 117-40
    // fromAddr/size で示す領域が MemoryState_Static だと失敗する
    // 廃止

    // TEST 117-41
    // fromAddr/size で示す領域が MemoryState_Code だと成功する
    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 117-42
    // fromAddr/size で示す領域が MemoryState_AliasedCode だと失敗する
    // 廃止

    // TEST 117-43
    // fromAddr/size で示す領域が MemoryState_AliasCode だと失敗する
    // 廃止

    // TEST 117-44
    // fromAddr/size で示す領域が MemoryState_Stack だと失敗する
    // 廃止

    // TEST 117-45
    // fromAddr/size で示す領域が MemoryState_ThreadLocal だと失敗する
    // 廃止

    // TEST 117-46
    // fromAddr/size で示す領域が MemoryState_Inaccessible だと失敗する
    // 廃止

    // TEST 117-47
    // fromAddr/size で示す領域 のメモリ状態が一様でないと失敗する (前方)
    // 廃止

    // TEST 117-48
    // fromAddr/size で示す領域 のメモリ状態が一様でないと失敗する (後方)
    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size * 2);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // TEST 117-49
    // オーバーフローする組み合わせを指定できない
    size = static_cast<size_t>(-0x1000);
    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    // 終了処理
    size = 0x1000;
    result = nn::svc::UnmapProcessMemory(toAddr, processHandle, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);
}

