﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Thread.h>
#include <cstring>

#if defined NN_BUILD_CONFIG_COMPILER_GCC
#pragma GCC optimize("O0")
#endif
extern "C" int TestSetJmp(uint64_t*);
extern "C" void TestLongJmp(uint64_t*);
#if defined NN_BUILD_CONFIG_CPU_ARM64
const size_t JmpBufSize = 30;
extern "C" void SetFpuContext(uint64_t*, uint32_t*, uint32_t*);
extern "C" void GetFpuContext(uint64_t*, uint32_t*, uint32_t*);
#elif defined NN_BUILD_CONFIG_CPU_ARM
const size_t JmpBufSize = 14;
extern "C" void SetFpuContext(uint64_t*, uint32_t*);
extern "C" void GetFpuContext(uint64_t*, uint32_t*);
#else
#error NN_BUILD_CONFIG_CPU_
#endif
extern int32_t g_ProcessIdealCore;

namespace {
char g_Buffer[DefaultStackSize] __attribute__((aligned(0x1000)));

const int NumTest = 100000;
bool g_LoopFlag;

#if defined NN_BUILD_CONFIG_CPU_ARM64
struct TestFpuData
{
    uint64_t fpr[64];
    uint32_t fpcr;
    uint32_t fpsr;
};

void InitTestFpuData(TestFpuData* baseData)
{
    static uint64_t s_Value = 0;
    int fprNum = static_cast<int>(sizeof(baseData->fpr) / sizeof(uint64_t));

    for (int j = 0; j < fprNum; j++)
    {
        baseData->fpr[j] = (s_Value << 56) | (s_Value << 48) | (s_Value << 40) | (s_Value << 32) | (s_Value << 24) | (s_Value << 16) | (s_Value << 8) | s_Value;
        s_Value++;
        s_Value &= 0xFF;
    }
    baseData->fpcr = (s_Value << 24) | (s_Value << 16) | (s_Value << 8) | s_Value;
    s_Value++;
    s_Value &= 0xFF;
    baseData->fpsr = (s_Value << 24) | (s_Value << 16) | (s_Value << 8) | s_Value;
    s_Value++;
    s_Value &= 0xFF;
}

void CallGetFpu(TestFpuData* data)
{
    GetFpuContext(data->fpr, &data->fpcr, &data->fpsr);
}

void CallSetFpu(TestFpuData* data)
{
    SetFpuContext(data->fpr, &data->fpcr, &data->fpsr);
}

void CmpTestFpuData(TestFpuData* baseData, TestFpuData* cmpData)
{
    ASSERT_TRUE(::std::memcmp(baseData->fpr, cmpData->fpr, sizeof(baseData->fpr)) == 0);
    ASSERT_TRUE(baseData->fpcr == cmpData->fpcr);
    ASSERT_TRUE(baseData->fpsr == cmpData->fpsr);
}

#elif defined NN_BUILD_CONFIG_CPU_ARM
struct TestFpuData
{
    uint64_t fpr[NN_BUILD_CONFIG_FPU_NUM_DOUBLE_REGISTERS];
    uint32_t fpscr;
};

void InitTestFpuData(TestFpuData* baseData)
{
    static uint64_t s_Value = 0;
    int fprNum = static_cast<int>(sizeof(baseData->fpr) / sizeof(uint64_t));

    for (int j = 0; j < fprNum; j++)
    {
        baseData->fpr[j] = (s_Value << 56) | (s_Value << 48) | (s_Value << 40) | (s_Value << 32) | (s_Value << 24) | (s_Value << 16) | (s_Value << 8) | s_Value;
        s_Value++;
        s_Value &= 0xFF;
    }
    baseData->fpscr = (s_Value << 24) | (s_Value << 16) | (s_Value << 8) | s_Value;
    s_Value++;
    s_Value &= 0xFF;
}

void CallGetFpu(TestFpuData* data)
{
    GetFpuContext(data->fpr, &data->fpscr);
}

void CallSetFpu(TestFpuData* data)
{
    SetFpuContext(data->fpr, &data->fpscr);
}

void CmpTestFpuData(TestFpuData* baseData, TestFpuData* cmpData)
{
    ASSERT_TRUE(::std::memcmp(baseData->fpr, cmpData->fpr, sizeof(baseData->fpr)) == 0);
    ASSERT_TRUE(baseData->fpscr == cmpData->fpscr);
}

#else
#error NN_BUILD_CONFIG_CPU_
#endif

void GetSetFpu()
{
    TestFpuData baseData;
    TestFpuData cmpData = { { 0 } };
    uint64_t jmpbuf[JmpBufSize];

    InitTestFpuData(&baseData);

    if (TestSetJmp(jmpbuf) == 0)
    {
        CallSetFpu(&baseData);
        // ステータスコントロールレジスタにセットした値がそのまま適用されるとは限らないので、
        // 一旦適用された値を取得しておく
        CallGetFpu(&baseData);
        CallGetFpu(&cmpData);
        TestLongJmp(jmpbuf);
    }

    CmpTestFpuData(&baseData, &cmpData);
}

void TestFunction()
{
    AutoThreadExit autoExit;

    while (g_LoopFlag)
    {
        GetSetFpu();
    }
}
}

// 自分のコアを切り替える
TEST(ThreadCoreMask, Test0)
{
    TestThreadLeak leakTest;
    nnHandle handle;
    handle.value = 0xFFFF8000;

    for (int n = 0; n < NumTest; n++)
    {
        for (int32_t i = 0; i < NumCore; i++)
        {
            TestFpuData baseData;
            TestFpuData cmpData = { { 0 } };
            uint64_t jmpbuf[JmpBufSize];
            InitTestFpuData(&baseData);

            if (TestSetJmp(jmpbuf) == 0)
            {
                // FPU のセット
                CallSetFpu(&baseData);
                // ステータスコントロールレジスタにセットした値がそのまま適用されるとは限らないので、
                // 一旦適用された値を取得しておく
                CallGetFpu(&baseData);

                {
                    nn::Bit64 mask = 1 << i;
                    nn::Result result = nn::svc::SetThreadCoreMask(static_cast<nn::svc::Handle>(handle), i, mask);
                    ASSERT_RESULT_SUCCESS(result);
                }

                // FPU の確認
                CallGetFpu(&cmpData);
                CmpTestFpuData(&baseData, &cmpData);

                {
                    nn::Bit64 mask = 0;
                    int32_t tmp;
                    // TEST 12-1
                    // 優先コアが取得できる
                    nn::Result result = nn::svc::GetThreadCoreMask(&tmp, &mask, static_cast<nn::svc::Handle>(handle));
                    ASSERT_RESULT_SUCCESS(result);

                    int32_t coreNo = nn::svc::GetCurrentProcessorNumber();
                    ASSERT_TRUE(coreNo == tmp);
                }

                TestLongJmp(jmpbuf);
            }
        }
    }

    {
        nn::Bit64 mask = 1 << g_ProcessIdealCore;
        nn::Result result = nn::svc::SetThreadCoreMask(static_cast<nn::svc::Handle>(handle), g_ProcessIdealCore, mask);
        ASSERT_RESULT_SUCCESS(result);
    }
}

// FPUをSetやGetしている途中で、コアを切り替える。
TEST(ThreadCoreMask, Test1)
{
    TestThreadLeak leakTest;
    int coreNo = 0;
    nnHandle myHandle;
    myHandle.value = 0xFFFF8000;
    nn::Result result;

    {
        // TEST 13-1
        // 自スレッドのidealCore とaffinityMask を変更することが出来る
        nn::Bit64 mask = 1 << coreNo;
        nn::Result result = nn::svc::SetThreadCoreMask(static_cast<nn::svc::Handle>(myHandle), coreNo, mask);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(nn::svc::GetCurrentProcessorNumber() == coreNo);
    }

    g_LoopFlag = true;
    nn::svc::Handle handle;
    result = nn::svc::CreateThread(&handle, reinterpret_cast<uintptr_t>(TestFunction), 0, reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer)), 24, 1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::StartThread(handle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 13-2
    // 実行中のスレッドに対して、idealCore とaffinityMask を変更することが出来る
    for (int n = 0; n < NumTest; n++)
    {
        for (int i = 1; i < NumCore; i++)
        {
            {
                nn::Bit64 mask = (1 << i);
                nn::Result result = nn::svc::SetThreadCoreMask(handle, i, mask);
                ASSERT_RESULT_SUCCESS(result);
            }
            nn::svc::SleepThread(100000);
            {
                nn::Bit64 mask = 0;
                int32_t tmp;
                nn::Result result = nn::svc::GetThreadCoreMask(&tmp, &mask, handle);
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(i == tmp);
            }
        }
    }
    g_LoopFlag = false;

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    {
        nn::Bit64 mask = 1 << g_ProcessIdealCore;
        nn::Result result = nn::svc::SetThreadCoreMask(static_cast<nn::svc::Handle>(myHandle), g_ProcessIdealCore, mask);
        ASSERT_RESULT_SUCCESS(result);
    }
}

