﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Kernel.h>
#include <nn/svc/svc_Result.h>

namespace {
char g_Buffer[DefaultStackSize] __attribute__((aligned(0x1000)));
uintptr_t g_SavedParam0;
}

struct TestThreadData
{
    uintptr_t sp;
    uintptr_t pc;
    int32_t priority;
    int32_t idealCore;
};

extern "C" void TestThreadEntry0(uintptr_t arg);
extern "C" void TestThreadFunction0(TestThreadData* pData, uintptr_t sp, uintptr_t pc);
extern int32_t g_ProcessIdealCore;

void TestThreadFunction0(TestThreadData* pData, uintptr_t sp, uintptr_t pc)
{
    nn::Result result;
    ASSERT_TRUE(reinterpret_cast<uintptr_t>(pData) == g_SavedParam0);
    ASSERT_TRUE(sp == pData->sp);
    ASSERT_TRUE(pc == pData->pc);

    nnHandle handle;
    handle.value = 0xFFFF8000;

    {
        nn::Bit64 id;
        result = nn::svc::GetThreadId(&id, static_cast<nn::svc::Handle>(handle));
        ASSERT_RESULT_SUCCESS(result);
        g_SavedParam0 = id;
    }

    {
        int32_t tmp;
        result = nn::svc::GetThreadPriority(&tmp, static_cast<nn::svc::Handle>(handle));
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(tmp == pData->priority);
    }

    {
        nn::Bit64 mask;
        int32_t tmp;
        int32_t coreNo = nn::svc::GetCurrentProcessorNumber();
        result = nn::svc::GetThreadCoreMask(&tmp, &mask, static_cast<nn::svc::Handle>(handle));
        ASSERT_RESULT_SUCCESS(result);
        if (pData->idealCore == nn::svc::IdealCoreUseProcessValue)
        {
            ASSERT_TRUE(tmp == g_ProcessIdealCore);
        }
        else
        {
            ASSERT_TRUE(tmp == pData->idealCore);
        }
        ASSERT_TRUE(coreNo == tmp);
        ASSERT_TRUE(mask == (1ULL << coreNo));
    }

    nn::svc::ExitThread();
}
namespace {
void Test0(int32_t idealCore)
{
    TestThreadData testData;
    uintptr_t sp;
    uintptr_t pc;
    int32_t priority;
    nn::Result result;
    nn::svc::Handle handle;
    uintptr_t param;

    for (priority = TestHighestThreadPriority; priority <= TestLowestThreadPriority; priority++)
    {
        sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
        pc = reinterpret_cast<uintptr_t>(TestThreadEntry0);
        param = reinterpret_cast<uintptr_t>(&testData);

        testData.sp = sp;
        testData.pc = pc;
        testData.priority = priority;
        testData.idealCore = idealCore;

        g_SavedParam0 = param;
        result = nn::svc::CreateThread(&handle, pc, param, sp, priority, idealCore);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::StartThread(handle);
        ASSERT_RESULT_SUCCESS(result);

        {
            int32_t tmp;
            result = nn::svc::GetThreadPriority(&tmp, handle);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(tmp == priority);
        }

        {
            nn::Bit64 mask;
            int32_t tmp;
            result = nn::svc::GetThreadCoreMask(&tmp, &mask, handle);
            ASSERT_RESULT_SUCCESS(result);
            if (idealCore == nn::svc::IdealCoreUseProcessValue)
            {
                ASSERT_TRUE(tmp == g_ProcessIdealCore);
            }
            else
            {
                ASSERT_TRUE(tmp == idealCore);
            }
            ASSERT_TRUE(mask == (1ULL << tmp));
        }

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        {
            nn::Bit64 id;
            result = nn::svc::GetThreadId(&id, handle);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(g_SavedParam0 == id);
        }

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
}
}

/* TEST 7-1, 7-6, 7-8, 7-9 */
/* CreateThread が正常に動作することを確認する */
TEST(CreateThreadTest, Test0)
{
    TestThreadLeak leakTest;
    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        Test0(idealCore);
    }
    Test0(nn::svc::IdealCoreUseProcessValue);
}

/* TEST 7-7, 7-11, 7-12 */
/* CreateThread が正しくエラーを返すかを確認する */
TEST(CreateThreadTest, Test1)
{
    TestThreadLeak leakTest;
    uintptr_t sp;
    uintptr_t pc;
    int32_t priority;
    nn::Result result;
    nn::svc::Handle handle;
    int32_t idealCore;

    sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
    pc = reinterpret_cast<uintptr_t>(TestThreadEntry0);
    idealCore = 0;
    priority = -1;
    result = nn::svc::CreateThread(&handle, pc, 0, sp, priority, idealCore);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPriority());

    priority = 64;
    result = nn::svc::CreateThread(&handle, pc, 0, sp, priority, idealCore);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPriority());

    priority = 32;
    idealCore = nn::svc::IdealCoreNoUpdate;
    result = nn::svc::CreateThread(&handle, pc, 0, sp, priority, idealCore);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCoreNumber());

    priority = 32;
    idealCore = nn::svc::IdealCoreDontCare;
    result = nn::svc::CreateThread(&handle, pc, 0, sp, priority, idealCore);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCoreNumber());
}



