﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <cstring>

namespace {

char g_Stack[DefaultStackSize] __attribute__((aligned(0x1000)));

void SendHandleThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle* clientSession = reinterpret_cast<nn::svc::Handle*>(arg);

    nn::Bit32 flag[DefaultCapabilityFlagNum];
    SetDefaultCapability(flag, DefaultCapabilityFlagNum);
    TestProcess process(1, 0, flag, DefaultCapabilityFlagNum);

    nn::svc::Handle handle = process.GetHandle();
    AssignExitCode(handle, process.GetCodeAddress(), process.GetCodeAreaSize());

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 1, 0, 0, 0, 0, 0, 0);
    nn::svc::ipc::MessageBuffer::SpecialHeader specialHeader(false, 1, 0);
    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(specialHeader);
    offset = ipcMsg.SetHandle(offset, handle);

    result = nn::svc::SendSyncRequest(*clientSession);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

} // namespace

TEST(StartProcess, SendHandleViaIpc)
{
    TestProcessLeak leakTest;
    nn::Result result;

    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;

    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);

    uintptr_t pc = reinterpret_cast<uintptr_t>(SendHandleThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + DefaultStackSize;
    uintptr_t param = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = 0;

    TestThread thread(pc, param, sp, priority, idealCore);
    thread.Start();

    // IPC でハンドラを受信
    nn::svc::Handle handle;
    {
        int32_t index;
        nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        ipcMsg.SetNull();

        result = nn::svc::ReplyAndReceive(
                &index, &serverSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
        nn::svc::ipc::MessageBuffer::SpecialHeader specialHeader(ipcMsg, ipcHeader);
        ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
        ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
        ASSERT_TRUE(specialHeader.GetCopyHandleNum() == 1);
        int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, specialHeader);
        handle = ipcMsg.GetHandle(offset);

        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceive(
                &index, &serverSession, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }

    // TEST 122-17
    // IPC で渡された INITIALIZING 状態のプロセスを受け付ける
    result = nn::svc::StartProcess(handle, priority, idealCore, DefaultStackSize);
    ASSERT_RESULT_SUCCESS(result);

    // 終了処理
    WaitProcess(handle);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
    thread.Wait();
}

TEST(StartProcess, ReCallTest)
{
    TestProcessLeak leakTest;
    nn::Result result;

    nn::Bit32 flag[DefaultCapabilityFlagNum];
    SetDefaultCapability(flag, DefaultCapabilityFlagNum);

    TestProcess process(1, 0, flag, DefaultCapabilityFlagNum);
    nn::svc::Handle handle = process.GetHandle();
    AssignExitCode(handle, process.GetCodeAddress(), process.GetCodeAreaSize());

    int32_t priority = TestLowestThreadPriority;
    int32_t coreNo = 0;

    result = nn::svc::StartProcess(handle, priority, coreNo, SIZE_MAX);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfMemory());

    result = nn::svc::StartProcess(handle, priority, coreNo, DefaultStackSize);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(StartProcess, SignalTest)
{
    TestProcessLeak leakTest;
    nn::Result result;

    nn::Bit32 flag[DefaultCapabilityFlagNum];
    SetDefaultCapability(flag, DefaultCapabilityFlagNum);

    TestProcess process(1, 0, flag, DefaultCapabilityFlagNum);
    nn::svc::Handle handle = process.GetHandle();

    // 作成時はシグナルされない
    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    AssignExitCode(handle, process.GetCodeAddress(), process.GetCodeAreaSize());

    int32_t priority = TestLowestThreadPriority;
    int32_t coreNo = 0;

    result = nn::svc::StartProcess(handle, priority, coreNo, SIZE_MAX);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfMemory());

    // スタート時はシグナルされる
    result = nn::svc::StartProcess(handle, priority, coreNo, DefaultStackSize);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ResetSignal(handle);
    ASSERT_RESULT_SUCCESS(result);

    // 終了時はシグナルされる
    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    int64_t info;
    result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);

    ASSERT_TRUE(static_cast<nn::svc::ProcessState>(info) == nn::svc::ProcessState_Terminated);
}
