﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Thread.h>
#include <nn/svc/svc_Tick.h>
#include <nn/svc/svc_HardwareParamsSelect.h>

namespace {

volatile int32_t g_SavedParam0;

struct TestThreadData
{
    int64_t waitTime;
    int32_t cmpValue;
    nn::svc::Handle writableEvent0;
    nn::svc::Handle readableEvent0;
    nn::svc::Handle writableEvent1;
    nn::svc::Handle readableEvent1;
};

void TestSleepThread0(uintptr_t arg)
{
    nn::Result result;
    struct TestThreadData *data = reinterpret_cast<struct TestThreadData*>(arg);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &data->readableEvent0, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::ClearEvent(data->readableEvent0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SignalEvent(data->writableEvent1);
    ASSERT_RESULT_SUCCESS(result);

    g_SavedParam0 = 1;
    nn::svc::SleepThread(data->waitTime);
    ASSERT_TRUE(g_SavedParam0 == data->cmpValue);

    nn::svc::ExitThread();
}

void TestSleepThread1(uintptr_t arg)
{
    nn::Result result;
    struct TestThreadData *data = reinterpret_cast<struct TestThreadData*>(arg);
    int32_t index;

    result = nn::svc::SignalEvent(data->writableEvent0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::WaitSynchronization(&index, &data->readableEvent1, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::ClearEvent(data->readableEvent1);
    ASSERT_RESULT_SUCCESS(result);

    g_SavedParam0 = 2;

    result = nn::svc::WaitSynchronization(&index, &data->readableEvent0, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::ExitThread();
}

void CreateTestThreads(uintptr_t heapPtr, struct TestThreadData *data, bool isSamePriority, bool doSwitch, int64_t waitTime)
{
    nn::svc::Handle handle[2];
    nn::Result result;
    uintptr_t sp;
    uintptr_t pc;
    uintptr_t param;
    int32_t idealCore;
    int32_t priority;
    int32_t index;

    param = reinterpret_cast<uintptr_t>(data);
    data->waitTime = waitTime;
    if(doSwitch)
    {
        data->cmpValue = 2;
    }
    else
    {
        data->cmpValue = 1;
    }

    for (idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t i = TestHighestThreadPriority;
                i <= TestLowestThreadPriority; i++)
        {
            if (isSamePriority && i == GetTimeSlicePriority(idealCore))
            {
                continue;
            }
            if (!isSamePriority && i == TestLowestThreadPriority)
            {
                break;
            }
            priority = i;
            sp = reinterpret_cast<uintptr_t>(heapPtr + 0x4000);
            pc = reinterpret_cast<uintptr_t>(TestSleepThread0);
            result = nn::svc::CreateThread(&handle[0], pc, param, sp, priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            sp = reinterpret_cast<uintptr_t>(heapPtr + 0x2000);
            pc = reinterpret_cast<uintptr_t>(TestSleepThread1);
            if(!isSamePriority)
            {
                priority = i + 1;
            }
            result = nn::svc::CreateThread(&handle[1], pc, param, sp, priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            g_SavedParam0 = 1;
            result = nn::svc::StartThread(handle[1]);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::StartThread(handle[0]);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::WaitSynchronization(&index, &handle[0], 1, -1);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::CloseHandle(handle[0]);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::SignalEvent(data->writableEvent0);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::WaitSynchronization(&index, &handle[1], 1, -1);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::CloseHandle(handle[1]);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
}
} // namespace

/* TEST 9-2, 9-3, 9-4 */
/* SleepThread の引数による振る舞いの変化を確認する */
TEST(SleepThread, TestSwitchThreads)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle writableEvent0;
    nn::svc::Handle readableEvent0;
    nn::svc::Handle writableEvent1;
    nn::svc::Handle readableEvent1;
    struct TestThreadData data;
    uintptr_t heapPtr;
    size_t heapSize;

    heapSize = (HeapAlign >= 0x4000) ? HeapAlign : ((0x4000 + (HeapAlign - 1)) & ~(HeapAlign - 1));
    result = nn::svc::SetHeapSize(&heapPtr, heapSize);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CreateEvent(&writableEvent0, &readableEvent0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CreateEvent(&writableEvent1, &readableEvent1);
    ASSERT_RESULT_SUCCESS(result);
    data.readableEvent0 = readableEvent0;
    data.writableEvent0 = writableEvent0;
    data.readableEvent1 = readableEvent1;
    data.writableEvent1 = writableEvent1;

    // TEST 9-2
    // -1 を与えた時には何もしない
    CreateTestThreads(heapPtr, &data, true, false, -1);
    // TEST 9-3
    // 0 を与えた時に、同じ優先度のスレッドがあれば切り替わる
    CreateTestThreads(heapPtr, &data, true, true, 0);
    // TEST 9-4
    // 0 を与えた時に、低い優先度のスレッドには切り替わらない
    CreateTestThreads(heapPtr, &data, false, false, 0);

    result = nn::svc::CloseHandle(readableEvent0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(writableEvent0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(readableEvent1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(writableEvent1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetHeapSize(&heapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);
}

