﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Thread.h>
#include <nn/svc/svc_Tick.h>
#include <nn/svc/svc_HardwareParamsSelect.h>

namespace {

const size_t StackSize = 0x2000;
const int32_t NumPriority = TestLowestThreadPriority - TestHighestThreadPriority + 1;
volatile int32_t g_SyncStart;
const int64_t SleepTime = 1000 * 1000 * 10; // 10 micro sec
const int64_t PerMicroSec = nn::svc::Tick::TICKS_PER_SECOND / (1000 * 1000); // micro sec

/*
   TEST 11-7
   自分の優先度を取得し、引数に与えられた優先度と等しいかを確認する
 */
void TestPriorityFunction0(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    int32_t priority;

    while (g_SyncStart == 0)
    {
        nn::svc::SleepThread(SleepTime);
    }

    result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(priority == static_cast<int>(arg));
}

/*
   TEST 11-19
   高い優先度のスレッドの処理
   優先度を変えることによって、TestPriorityFunction2 に切り替わるかを確認する
 */
void TestPriorityFunction1(uintptr_t priority)
{
    AutoThreadExit autoExit;
    nn::Result result;

    // TestPriorityFunction2 の起動を待つ
    while (g_SyncStart == 0)
    {
        nn::svc::SleepThread(SleepTime);
    }

    g_SyncStart = 3;
    result = nn::svc::SetThreadPriority(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, priority);
    ASSERT_RESULT_SUCCESS(result);

    // TestPriorityFunction2 に切り替わっているはず
    ASSERT_TRUE(g_SyncStart != 3);
}

/*
   TEST 11-19
   低い優先度のスレッドの処理
 */
void TestPriorityFunction2(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
    nn::Result result;

    g_SyncStart = 1;

    // スレッドはこのループの間に起きるはず
    while (g_SyncStart != 3)
    {
    }
    ASSERT_TRUE(g_SyncStart == 3);

    // ここから先はTestPriorityFunction1 で優先度が変更になったせいで、切り替わったはずの処理
    g_SyncStart = 4;
}

/*
   TEST 11-20
   優先度を変更するスレッドの処理
   高い優先度に変更しても、他コアのスケジューリングに影響を与えない
 */
void TestPriorityFunction3(uintptr_t priority)
{
    AutoThreadExit autoExit;
    nn::Result result;
    int32_t tmp;

    // TestPriorityFunction4 の起動を待つ
    while (g_SyncStart == 0)
    {
        nn::svc::SleepThread(SleepTime);
    }

    result = nn::svc::SetThreadPriority(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, priority);
    ASSERT_RESULT_SUCCESS(result);
    tmp = 0;
    g_SyncStart = tmp;

    volatile int64_t begin = nn::svc::GetSystemTick();
    int64_t tick = 0;

    // この間に他コアで動いている TestPriorityFunction4 で
    // g_SyncStart がインクリメントされているはず
    // システムプロセスが動くかもしれないので、1秒待つ
    while (tick < 1000000)
    {
        volatile int64_t current = nn::svc::GetSystemTick();
        tick = (current - begin) / PerMicroSec;
        if (tmp != g_SyncStart)
        {
            break;
        }
    }

    ASSERT_TRUE(tmp != g_SyncStart);
}

void TestPriorityFunction4(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
    nn::Result result;

    // TestPriorityFunction3 の起動を待つ
    g_SyncStart = 1;
    while (g_SyncStart == 1)
    {
        nn::svc::SleepThread(SleepTime);
    }

    g_SyncStart = 1;
}

/*
   TEST 11-19
   同じコアに高い優先度と低い優先度のスレッドを作成する。
   優先度の差は1
 */
void CreateTestThreads0(uintptr_t heapPtr, size_t heapSize,
    int32_t priority, int32_t idealCore)
{
    const uintptr_t PcHigh = reinterpret_cast<uintptr_t>(TestPriorityFunction1);
    const uintptr_t PcLow = reinterpret_cast<uintptr_t>(TestPriorityFunction2);
    uintptr_t sp = heapPtr + heapSize;
    nn::svc::Handle *handle = reinterpret_cast<nn::svc::Handle*>(heapPtr);
    nn::Result result;

    ASSERT_TRUE(heapSize >= (sizeof(nn::svc::Handle) + StackSize) * 2);
    ASSERT_TRUE(priority + 2 <= TestLowestThreadPriority);
    result = nn::svc::CreateThread(&handle[0],
            PcHigh, priority + 2, sp,
            priority, idealCore);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CreateThread(&handle[1],
            PcLow, 0, sp - StackSize,
            priority + 1, idealCore);
    ASSERT_RESULT_SUCCESS(result);

    g_SyncStart = 0;

    result = nn::svc::StartThread(handle[0]);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::StartThread(handle[1]);
    ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &handle[0], 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::WaitSynchronization(&index, &handle[1], 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle[0]);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle[1]);
    ASSERT_RESULT_SUCCESS(result);
}

/*
   TEST 11-20
   違うコアに同じ優先度のスレッドを作成する
 */
void CreateTestThreads1(uintptr_t heapPtr, size_t heapSize,
    int32_t priority, int32_t selfCore, int32_t otherCore)
{
    const uintptr_t PcHigh = reinterpret_cast<uintptr_t>(TestPriorityFunction3);
    const uintptr_t PcLow = reinterpret_cast<uintptr_t>(TestPriorityFunction4);
    uintptr_t sp = heapPtr + heapSize;
    nn::svc::Handle *handle = reinterpret_cast<nn::svc::Handle*>(heapPtr);
    nn::Result result;

    ASSERT_TRUE(priority - 1 >= TestHighestThreadPriority);
    ASSERT_TRUE(heapSize >= (sizeof(nn::svc::Handle) + StackSize) * 2);
    ASSERT_TRUE(selfCore != otherCore);
    result = nn::svc::CreateThread(&handle[0],
            PcHigh, priority - 1, sp,
            priority, selfCore);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CreateThread(&handle[1],
            PcLow, 0, sp - StackSize,
            priority, otherCore);
    ASSERT_RESULT_SUCCESS(result);

    g_SyncStart = 0;

    result = nn::svc::StartThread(handle[0]);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::StartThread(handle[1]);
    ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &handle[0], 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::WaitSynchronization(&index, &handle[1], 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle[0]);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle[1]);
    ASSERT_RESULT_SUCCESS(result);
}

} // namespace {

/* TEST 11-7 */
/* 複数のスレッドがあった時に、SetThreadPriority が正しく動作することを確認する */
TEST(SetThreadPriority, CallAtMultipleThreadsTest)
{
    TestThreadLeak leakTest;
    nn::svc::Handle *handles;
    nn::Result result;
    uintptr_t heapPtr;
    size_t heapSize;
    size_t wantSize = StackSize * NumPriority + sizeof(nn::svc::Handle) * NumPriority;
    int32_t idealCore;
    int32_t priority;
    int32_t curPriority;

    // 生成者のスレッドを最高優先度にしておく
    result = nn::svc::GetThreadPriority(&curPriority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SetThreadPriority(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD,
            TestHighestThreadPriority);
    ASSERT_RESULT_SUCCESS(result);

    heapSize = (HeapAlign >= wantSize) ?
        HeapAlign : (wantSize + HeapAlign - 1) & ~(HeapAlign - 1);
    result = nn::svc::SetHeapSize(&heapPtr, heapSize);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SetMemoryPermission(
            heapPtr, heapSize, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    handles = reinterpret_cast<nn::svc::Handle*>(heapPtr);

    for (idealCore = 0; idealCore < NumCore; idealCore++)
    {
        g_SyncStart = 0;
        for (priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority; priority++)
        {
            uintptr_t sp = reinterpret_cast<uintptr_t>(
                    heapPtr + heapSize - StackSize * priority);
            uintptr_t pc = reinterpret_cast<uintptr_t>(TestPriorityFunction0);
            int32_t nextPriority = (priority + 1) % NumPriority
                + TestHighestThreadPriority;
            nn::svc::Handle handle;

            result = nn::svc::CreateThread(&handles[priority],
                    pc, nextPriority, sp, priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::StartThread(handles[priority]);
            ASSERT_RESULT_SUCCESS(result);

            nn::Result result = nn::svc::SetThreadPriority(handles[priority], nextPriority);
            ASSERT_RESULT_SUCCESS(result);
        }

        g_SyncStart = 1;
        for (priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority; priority++)
        {
            int32_t index;
            result = nn::svc::WaitSynchronization(&index, &handles[priority], 1, -1);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::CloseHandle(handles[priority]);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

    result = nn::svc::SetHeapSize(&heapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);

    // 生成者のスレッドを元の優先度に戻す
    result = nn::svc::SetThreadPriority(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD,
            curPriority);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(SetThreadPriority, SwitchThreadsTest)
{
    TestThreadLeak leakTest;
    nn::Result result;
    uintptr_t heapPtr;
    nn::svc::Handle handle;
    int32_t curPriority;
    size_t heapSize = StackSize * 2
        + sizeof(nn::svc::Handle) * 2;

    heapSize = (heapSize + HeapAlign - 1) & ~(HeapAlign - 1);
    result = nn::svc::SetHeapSize(&heapPtr, heapSize);
    ASSERT_RESULT_SUCCESS(result);

    // 生成者のスレッドを最高優先度にしておく
    result = nn::svc::GetThreadPriority(&curPriority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SetThreadPriority(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD,
            TestHighestThreadPriority);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 11-19
    // SetThreadPriority によって優先度が変わったことによってスレッドが切り替わるかを確認
    for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
    {
        for (int32_t priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority - 2; priority++)
        {
            CreateTestThreads0(heapPtr, heapSize, priority, idealCore);
        }
    }

    // TEST 11-20
    // SetThreadPriority による優先度の変更が他コアで動くスレッドに影響を与えないかを確認
    for (int32_t priority = TestHighestThreadPriority + 1;
            priority <= TestLowestThreadPriority; priority++)
    {
        for (int32_t selfCore = 0; selfCore < NumCore; selfCore++)
        {
            for (int32_t otherCore = 0; otherCore < NumCore; otherCore++)
            {
                if (selfCore == otherCore)
                {
                    continue;
                }
                CreateTestThreads1(heapPtr, heapSize, priority,
                        selfCore, otherCore);
            }
        }
    }

    // 生成者のスレッドを元の優先度に戻す
    result = nn::svc::SetThreadPriority(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD,
            curPriority);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetHeapSize(&heapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);
}
