﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include "util_MemoryState.h"
#include "util_TestMemory.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/TargetConfigs/build_Compiler.h>
#include <cstring>

extern "C" void nnMain();

#if defined NN_BUILD_CONFIG_COMPILER_CLANG
#pragma clang diagnostic ignored "-Wtautological-compare"
#endif

namespace {

char g_Buffer[0x1000] __attribute__((aligned(0x1000)));

void CheckSuccessMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;

    nn::svc::MemoryPermission perm = nn::svc::MemoryPermission_ReadExecute;

    for (int i = 0; i < numState; i++)
    {
        ASSERT_TRUE(array[i]->GetSize() > 0 && ((array[i]->GetSize() & 0xfff) == 0));

        array[i]->Initialize();
        uint64_t size = 0x1000;
        uint64_t addr = array[i]->GetAddress();
        nn::svc::Handle processHandle = nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS;

        result = nn::svc::SetProcessMemoryPermission(processHandle, addr, size, perm);
        ASSERT_RESULT_SUCCESS(result);

        CheckMemory(
                addr, array[i]->GetState(), perm, 0);

        result = nn::svc::SetProcessMemoryPermission(
                processHandle, addr, size, array[i]->GetPermission());
        ASSERT_RESULT_SUCCESS(result);

        array[i]->CheckDefaultState();
        array[i]->Close();
    }
}

void CheckInvalidMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;

    nn::svc::MemoryPermission perm = nn::svc::MemoryPermission_ReadExecute;

    for (int i = 0; i < numState; i++)
    {
        if (!(array[i]->GetSize() > 0 && ((array[i]->GetSize() & 0xfff) == 0)))
        {
            array[i]->Close();
            continue;
        }

        array[i]->Initialize();
        uint64_t size = 0x1000;
        uint64_t addr =  array[i]->GetAddress();
        nn::svc::Handle processHandle = nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS;

        result = nn::svc::SetProcessMemoryPermission(processHandle, addr, size, perm);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        array[i]->CheckDefaultState();

        array[i]->Close();
    }
}



} // namespace

TEST(SetProcessMemoryPermission, HandleTest)
{
    nn::Result result;

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum);
    nn::svc::Handle processHandle = process.GetHandle();

    uint64_t addr = process.GetCodeAddress();
    uint64_t size = 0x1000;

    // TEST 115-1
    // 実行前のプロセスのハンドルを受け付ける
    result = nn::svc::SetProcessMemoryPermission(
            processHandle, addr, size, nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_SUCCESS(result);

    AssignExitCode(processHandle, addr, size);

    result = nn::svc::StartProcess(
            processHandle, TestLowestThreadPriority, g_ProcessIdealCore, DefaultStackSize);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 115-2
    // 実行中のプロセスのハンドルを受け付ける
    result = nn::svc::SetProcessMemoryPermission(
            processHandle, addr, size, nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::TerminateProcess(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 115-3
    // 実行後のプロセスのハンドルを受け付ける
    result = nn::svc::SetProcessMemoryPermission(
            processHandle, addr, size, nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 115-4
    // プロセスの擬似ハンドルを受け付ける
    addr = reinterpret_cast<uint64_t>(nnMain);
    nn::svc::MemoryInfo info;
    GetMemoryInfo(&info, addr);
    ASSERT_TRUE(info.state == nn::svc::MemoryState_Code);
    ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadExecute);
    ASSERT_TRUE((info.baseAddress & 0xfff) == 0);
    ASSERT_TRUE(info.size >= 0x1000);
    addr = info.baseAddress;
    result = nn::svc::SetProcessMemoryPermission(
            nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, addr, size,
            nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 115-3
    // INVALID_HANDLE_VALUE を受け付けない
    result = nn::svc::SetProcessMemoryPermission(
            nn::svc::INVALID_HANDLE_VALUE, addr, size,
            nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 115-4
    // スレッドの擬似ハンドルを受け付けない
    result = nn::svc::SetProcessMemoryPermission(
            nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, addr, size,
            nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 115-5
    // Close したプロセスのハンドルを受け付けない
    addr = process.GetCodeAddress();
    result = nn::svc::SetProcessMemoryPermission(
            processHandle, addr, size, nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(SetProcessMemoryPermission, AddressTest)
{
    nn::Result result;
    uint64_t addr = reinterpret_cast<uint64_t>(nnMain);
    uint64_t size = 0x1000;

    nn::svc::MemoryInfo blockInfo;
    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Code);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadExecute);
    ASSERT_TRUE((blockInfo.baseAddress & 0xfff) == 0);
    ASSERT_TRUE(blockInfo.size >= 0x1000);
    addr = blockInfo.baseAddress;

    // TEST 115-8
    // addr のアラインメントが4KBに揃っていると成功する
    for (uint64_t i = blockInfo.baseAddress;
            i < blockInfo.baseAddress + blockInfo.size;
            i += 0x1000)
    {
        result = nn::svc::SetProcessMemoryPermission(
                nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, i, size,
                nn::svc::MemoryPermission_ReadExecute);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 115-9
    // addr のアラインメントが4KBに揃っていないと失敗する
    for (uint64_t i = addr + 1; i < addr + 0x1000; i++)
    {
        result = nn::svc::SetProcessMemoryPermission(
                nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, i, size,
                nn::svc::MemoryPermission_ReadExecute);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }
}

TEST(SetProcessMemoryPermission, SizeTest)
{
    nn::Result result;
    uint64_t addr = reinterpret_cast<uint64_t>(nnMain);

    nn::svc::MemoryInfo blockInfo;
    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Code);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadExecute);
    ASSERT_TRUE((blockInfo.baseAddress & 0xfff) == 0);
    ASSERT_TRUE(blockInfo.size >= 0x1000);
    addr = blockInfo.baseAddress;

    // TEST 115-10
    // size のアラインメントが4KBに揃っていると成功する
    for (uint64_t i = 0x1000; i < blockInfo.size; i += 0x1000)
    {
        result = nn::svc::SetProcessMemoryPermission(
                nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, blockInfo.baseAddress, i,
                nn::svc::MemoryPermission_ReadExecute);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 115-11
    // size のアラインメントが4KBに揃っていないと失敗する
    for (uint64_t i = 1; i < 0x1000; i++)
    {
        result = nn::svc::SetProcessMemoryPermission(
                nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, addr, i,
                nn::svc::MemoryPermission_ReadExecute);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
    }

    // TEST 115-12
    // size は 0 を受け付けない
    result = nn::svc::SetProcessMemoryPermission(
            nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, addr, 0,
            nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());
}

TEST(SetProcessMemoryPermission, MemoryAreaTest)
{
    nn::Result result;
    uint64_t memSize = 0x1000;
    uint64_t addr;
    uint64_t size;

    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_AliasCode,
        nn::svc::MemoryState_Code,
    };
    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);

    // 他プロセス
    {
        TestMemoryStateProcess stateProcess;
        nn::svc::Handle processHandle = stateProcess.GetHandle();
        nn::svc::MemoryState state;
        nn::svc::MemoryPermission perm = nn::svc::MemoryPermission_Read;

        // 許可されたメモリ状態
        for (int i = 0; i < NumSuccessState; i++)
        {
            SCOPED_TRACE("Success Memory Area");
            state = successState[i];
            stateProcess.GetMemoryArea(&addr, &size, state);
            ASSERT_TRUE(size >= memSize);

            nn::svc::MemoryInfo blockInfo;
            nn::svc::PageInfo pageInfo;
            result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, addr);
            nn::svc::MemoryPermission curPerm = blockInfo.permission;
            ASSERT_TRUE(blockInfo.state == state);

            result = nn::svc::SetProcessMemoryPermission(
                    processHandle, addr, memSize, perm);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, addr);

            CheckProcessMemory(
                    processHandle,
                    addr,
                    state,
                    perm,
                    0);

            result = nn::svc::SetProcessMemoryPermission(
                    processHandle, addr, memSize, curPerm);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, addr);

            CheckProcessMemory(
                    processHandle,
                    addr,
                    state,
                    curPerm,
                    0);

            stateProcess.FreeMemoryArea();
        }

        // 許可されていないメモリ状態
        for (int i = 0; i < NumTestMemoryState; i++)
        {
            SCOPED_TRACE("Invalid Memory Area");
            nn::svc::MemoryState state = AllMemoryState[i];
            bool isSuccess = false;
            for (int j = 0; j < NumSuccessState; j++)
            {
                if (state == successState[j])
                {
                    isSuccess = true;
                    break;
                }
            }
            if (isSuccess)
            {
                continue;
            }

            stateProcess.GetMemoryArea(&addr, &size, state);
            if (size < memSize)
            {
                continue;
            }

            nn::svc::MemoryInfo blockInfo;
            nn::svc::PageInfo pageInfo;
            result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, addr);
            nn::svc::MemoryPermission curPerm = blockInfo.permission;

            result = nn::svc::SetProcessMemoryPermission(
                    processHandle, addr, memSize, perm);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

            CheckProcessMemory(
                    processHandle,
                    addr,
                    state,
                    curPerm,
                    0);

            stateProcess.FreeMemoryArea();
        }

        // AliasCode を用いた異常系チェック
        {
            uint64_t aliasCodeAddr;
            stateProcess.GetMemoryArea(&aliasCodeAddr, &size, nn::svc::MemoryState_AliasCode);
            ASSERT_TRUE(size >= memSize * 2);

            nn::svc::MemoryInfo info;
            nn::svc::PageInfo pageInfo;
            result = nn::svc::QueryProcessMemory(&info, &pageInfo, stateProcess.GetHandle(), aliasCodeAddr);
            ASSERT_RESULT_SUCCESS(result);

            // TEST 115-27
            // メモリ状態が一様でないと失敗する(前方)
            addr = aliasCodeAddr - memSize;
            result = nn::svc::SetProcessMemoryPermission(
                    processHandle, addr, memSize * 2,
                    nn::svc::MemoryPermission_ReadExecute);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

            // TEST 115-28
            // メモリ状態が一様でないと失敗する(後方)
            addr = aliasCodeAddr + size - memSize;
            result = nn::svc::SetProcessMemoryPermission(
                    processHandle, addr, memSize * 2,
                    nn::svc::MemoryPermission_ReadExecute);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

            // TEST 115-58
            // オーバーフローする組み合わせを指定できない
            addr = aliasCodeAddr + memSize;
            size = static_cast<uint64_t>(-memSize);
            result = nn::svc::SetProcessMemoryPermission(
                    processHandle, addr, size,
                    nn::svc::MemoryPermission_ReadExecute);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

            // メモリパーミッションが一様でないと失敗する
            result = nn::svc::SetProcessMemoryPermission(
                    processHandle, aliasCodeAddr, memSize,
                    nn::svc::MemoryPermission_None);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::SetProcessMemoryPermission(
                    processHandle, aliasCodeAddr + memSize, memSize,
                    nn::svc::MemoryPermission_Read);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::SetProcessMemoryPermission(
                    processHandle, aliasCodeAddr, memSize * 2,
                    nn::svc::MemoryPermission_ReadExecute);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

            result = nn::svc::SetProcessMemoryPermission(
                    processHandle, aliasCodeAddr, memSize,
                    info.permission);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::SetProcessMemoryPermission(
                    processHandle, aliasCodeAddr + memSize, memSize,
                    info.permission);
            ASSERT_RESULT_SUCCESS(result);

            stateProcess.FreeMemoryArea();
        }
    }

    // 自プロセス
    {
        TestMemoryInfo* header;
        GenerateMemoryStateList(
                &header, reinterpret_cast<uint64_t>(g_Buffer), sizeof(g_Buffer));

        // 許可されたメモリ状態の領域を受け付ける
        TestMemoryInfo* successArea[NumSuccessState];
        header->GetTestListWithStates(successArea, successState, NumSuccessState);

        CheckSuccessMemoryArea(successArea, NumSuccessState);

        // 許可されていないメモリ状態の領域を受け付けない
        const int NumInvalid = NumTestMemoryState - NumSuccessState;
        TestMemoryInfo* invalidArea[NumInvalid];
        header->GetTestListExceptStates(invalidArea, NumInvalid, successState, NumSuccessState);

        CheckInvalidMemoryArea(invalidArea, NumInvalid);

        // AliasCode を用いた異常系チェック
        {
            uint64_t aliasCodeAddr;
            TestAliasCodeMemoryState aliasCode;
            aliasCode.Initialize();
            aliasCodeAddr = aliasCode.GetAddress();
            nn::svc::Handle processHandle = nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS;

            // TEST 115-27
            // メモリ状態が一様でないと失敗する(前方)
            addr = aliasCodeAddr - memSize;
            result = nn::svc::SetProcessMemoryPermission(
                    processHandle, addr, memSize * 2,
                    nn::svc::MemoryPermission_ReadExecute);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

            // TEST 115-28
            // メモリ状態が一様でないと失敗する(後方)
            addr = aliasCodeAddr + aliasCode.GetSize() - memSize;
            result = nn::svc::SetProcessMemoryPermission(
                    processHandle, addr, memSize * 2,
                    nn::svc::MemoryPermission_ReadExecute);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

            // TEST 115-58
            // オーバーフローする組み合わせを指定できない
            addr = aliasCodeAddr + memSize;
            size = static_cast<uint64_t>(-memSize);
            result = nn::svc::SetProcessMemoryPermission(
                    processHandle, addr, size,
                    nn::svc::MemoryPermission_ReadExecute);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }
    }
} // NOLINT (readability/fn_size)

