﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"

namespace {

const int32_t BeginIndex = 0; // 先頭
const int32_t MiddleIndex = (HeapAlign  / sizeof(int32_t)) / 2; // 真ん中: (HeapAlign / 4) / 2
const int32_t EndIndex = (HeapAlign / sizeof(int32_t)) - 1; // 終端: (HeapAlign / 4) - 1
const int32_t BeginValue = 1;
const int32_t MiddleValue = 2;
const int32_t EndValue = 3;

void WriteData(uintptr_t addr, size_t size)
{
    ASSERT_TRUE((HeapAlign & 3U) == 0);

    int32_t *data = reinterpret_cast<int32_t*>(addr);
    int32_t diff = HeapAlign / sizeof(int32_t);
    // size は必ずHeapAlign の倍数
    // 時間が掛かるので、各HeapAlign内の先頭、真ん中、終端のデータを確認する
    for (int32_t i = 0; i < static_cast<int32_t>(size / HeapAlign); i++, data += diff)
    {
        data[BeginIndex] = BeginValue;
        data[MiddleIndex] = MiddleValue;
        data[EndIndex] = EndValue;
    }
}

void ReadDataTest(uintptr_t addr, size_t size)
{
    ASSERT_TRUE((HeapAlign & 3U) == 0);

    if (size == 0)
    {
        return;
    }

    int32_t *data = reinterpret_cast<int32_t*>(addr);
    // size は必ずHeapAlign の倍数
    // 時間が掛かるので、各HeapAlign内の先頭、真ん中、終端のデータを確認する
    for (int32_t i = 0; i < static_cast<int32_t>(size / HeapAlign); i++, data += HeapAlign >> 2)
    {
        ASSERT_TRUE(data[BeginIndex] == BeginValue);
        ASSERT_TRUE(data[MiddleIndex] == MiddleValue);
        ASSERT_TRUE(data[EndIndex] == EndValue);
    }
}
} // namespace {

/* TEST 1-5 */
/* ヒープ領域を拡張・縮小したときに、以前のデータが残っていることを確認する */
TEST(SetHeapSizeTest, RemindDataTest)
{
    uintptr_t addr;
    size_t size;
    size_t maxAllocation;
    nn::svc::MemoryInfo info;
    nn::Result result;
    int i;

    size = 0;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    size = HeapAlign;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    WriteData(addr, size);

    // ヒープを拡張する
    maxAllocation = size;
    for (i = 2;; i++)
    {
        uintptr_t backAddr = addr + size;
        GetMemoryInfo(&info, backAddr);

        uintptr_t tmp;
        size = HeapAlign * i;
        result = nn::svc::SetHeapSize(&tmp, size);
        if (result.IsFailure())
        {
            break;
        }
        ASSERT_TRUE(tmp == addr);
        ASSERT_TRUE(info.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(info.attribute == 0);

        GetMemoryInfo(&info, backAddr);
        ASSERT_TRUE(info.baseAddress == addr);
        ASSERT_TRUE(info.size == size);
        ASSERT_TRUE(info.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(info.attribute == 0);

        // 前回書いたデータ分
        ReadDataTest(tmp, maxAllocation);

        // 新たに追加するデータ
        WriteData(addr + maxAllocation, HeapAlign);
        maxAllocation = size;
    }

    size = 0;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    size = HeapAlign;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&info, addr);
    ASSERT_TRUE(info.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(info.attribute == 0);
    ASSERT_TRUE(info.baseAddress == addr);
    ASSERT_TRUE(info.size == size);

    // ヒープを縮小する
    for (size = maxAllocation; size > 0; size -= HeapAlign)
    {
        uintptr_t tmp;
        result = nn::svc::SetHeapSize(&tmp, size);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&info, tmp);
        if (size == maxAllocation)
        {
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Normal);
            ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadWrite);
            ASSERT_TRUE(info.baseAddress == addr);
            ASSERT_TRUE(info.size == size);
            WriteData(tmp, size);
        }
        else
        {
            GetMemoryInfo(&info, tmp + size);
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Free);
            ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_None);
            ASSERT_TRUE(info.attribute == 0);
        }

        ReadDataTest(tmp, size);
    }

    size = 0;
    result = nn::svc::SetHeapSize(&addr, size);
    ASSERT_RESULT_SUCCESS(result);
}

#ifdef INVALID_POINTER_TEST
/* TEST 1-9 */
/* 不正な値をアドレスに渡すと失敗する */
TEST(SetHeapSizeTest, InvalidPointerTest)
{
    nn::Result result;
    // TEST 1-9
    // NULL を渡すと失敗する
    result = nn::svc::SetHeapSize(NULL, HeapAlign);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
}
#endif

