﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/ipc/svc_SessionMessage.h>

namespace {

const int32_t NumPriority = TestLowestThreadPriority - TestHighestThreadPriority + 1;
char g_ClientBuffer[0x1000] __attribute__((aligned(0x1000)));
char g_ServerBuffer[0x1000] __attribute__((aligned(0x1000)));
char g_TestBuffer[0x1000] __attribute__((aligned(0x1000)));
char g_Stack[2][DefaultStackSize] __attribute__((aligned(0x1000)));
int g_Sequence;
const int64_t SleepTime = 100 * 1000 * 1000;
const int BeginThread = 0;
const int RunningThread = 1;
const int WaitThread = 2;
const int EndThread = 3;

struct TestData
{
    nn::svc::Handle clientSession;
    nn::svc::Handle writableEvent;
};

struct UserBufferThreadData
{
    nn::svc::Handle clientSession;
    nn::Bit32* userBuffer;
    size_t bufferSize;
};

void SendThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle *handle = reinterpret_cast<nn::svc::Handle*>(arg);

    while (g_Sequence != EndThread)
    {
        g_Sequence = RunningThread;
        SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
        // IsFailure になることもあるので、result のチェックはしない
        result = nn::svc::SendSyncRequest(*handle);
    }
}

void OnlyReplyTestServer(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle *handle = reinterpret_cast<nn::svc::Handle*>(arg);
    nn::svc::Handle handles[1];
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(g_ServerBuffer);
    size_t msgSize = sizeof(g_ServerBuffer);
    int32_t index;

    handles[0] = *handle;

    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize,
                handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);

    g_Sequence = RunningThread;

    // Cancel を待つ
    while (g_Sequence == RunningThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    // IPC の結果となるデータを用意しておく
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 0, 0, 0));

    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize,
                handles, 1, *handle, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultCancelled());
}

void OnlyReplyTestClient(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle *handle = reinterpret_cast<nn::svc::Handle*>(arg);
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();

    SetOnlyIpcTag(nullptr, DefaultIpcTag_Send);
    result = nn::svc::SendSyncRequest(*handle);
    ASSERT_RESULT_SUCCESS(result);

    // 受信確認
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
        ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
        ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
        ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
        ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
        ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
        ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
        ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
        ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);
    }
}

void TestCancelAndCloseSessionOrderThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle* handles = reinterpret_cast<nn::svc::Handle*>(arg);
    uintptr_t msgBufferAddr = reinterpret_cast<uintptr_t>(g_ServerBuffer);
    size_t msgSize = sizeof(g_ServerBuffer);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(msgBufferAddr);
    int32_t index;

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Receive);
    g_Sequence = RunningThread;
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 2, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Send);

    // Cancel とセッションのクローズを待つ
    while(g_Sequence != EndThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Reply);
    nn::svc::Handle tmp = handles[1];
    handles[1] = handles[0];
    handles[0] = tmp;
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 1, handles[1], -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultCancelled());
}

void TestSignalAndCloseSessionOrderThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;

    nn::svc::Handle* handles = reinterpret_cast<nn::svc::Handle*>(arg);
    uintptr_t msgBufferAddr = reinterpret_cast<uintptr_t>(g_ServerBuffer);
    size_t msgSize = sizeof(g_ServerBuffer);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(msgBufferAddr);
    int32_t index;

    g_Sequence = RunningThread;
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 2, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Send);

    // Signal とClose を待つ
    while(g_Sequence != EndThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Reply);
    // セッションのクローズが優先される
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 2, handles[0], -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());

    // 念のため、要求を受信できるか確認する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Receive);
    nn::svc::Handle tmp = handles[1];
    handles[1] = handles[0];
    handles[0] = tmp;
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Send);

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Reply);
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 0, handles[0], 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}

void TestCancelAndSignalOrderThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;

    nn::svc::Handle *handle = reinterpret_cast<nn::svc::Handle*>(arg);
    nn::svc::Handle handles[1];
    uintptr_t msgBufferAddr = reinterpret_cast<uintptr_t>(g_ServerBuffer);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(msgBufferAddr);
    size_t msgSize = sizeof(g_ServerBuffer);
    int32_t index;

    handles[0] = *handle;

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Receive);
    g_Sequence = RunningThread;
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Send);

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Reply);
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 1, *handle, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultCancelled());
}

void TestCloseAndTimeoutOrderThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;

    nn::svc::Handle *handles = reinterpret_cast<nn::svc::Handle*>(arg);
    uintptr_t msgBufferAddr = reinterpret_cast<uintptr_t>(g_ServerBuffer);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(msgBufferAddr);
    size_t msgSize = sizeof(g_ServerBuffer);
    int32_t index;

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Receive);
    g_Sequence = RunningThread;
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 2, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Send);

    // セッションがクローズされるのを待つ
    while(g_Sequence != EndThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Reply);
    nn::svc::Handle replyTarget = handles[0];
    handles[0] = handles[1];
    handles[1] = replyTarget;
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 1, replyTarget, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
}

void TestSignalAndTimeoutOrderThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;

    nn::svc::Handle *handles = reinterpret_cast<nn::svc::Handle*>(arg);
    uintptr_t msgBufferAddr = reinterpret_cast<uintptr_t>(g_ServerBuffer);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(msgBufferAddr);
    size_t msgSize = sizeof(g_ServerBuffer);
    int32_t index;

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Receive);
    g_Sequence = RunningThread;
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 2, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Send);

    // タイムアウトと同期オブジェクトのシグナルを待つ
    while(g_Sequence != EndThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Reply);
    nn::svc::Handle replyTarget = handles[0];
    handles[0] = handles[1];
    handles[1] = replyTarget;

    // 同期オブジェクトのシグナルが優先される
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 1, replyTarget, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Send);

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Reply);
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 0, handles[0], 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}

void TestCancelAndTimeoutOrderThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;

    nn::svc::Handle *handles = reinterpret_cast<nn::svc::Handle*>(arg);
    uintptr_t msgBufferAddr = reinterpret_cast<uintptr_t>(g_ServerBuffer);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(msgBufferAddr);
    size_t msgSize = sizeof(g_ServerBuffer);
    int32_t index;

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Receive);
    g_Sequence = RunningThread;
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 2, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);
    ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Send);

    // キャンセルされるのを待つ
    while(g_Sequence != EndThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    nn::svc::Handle replyTarget = handles[0];
    handles[0] = handles[1];
    handles[1] = replyTarget;

    // タイムアウトが優先される
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Reply);
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 1, replyTarget, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // Cancel を確認する
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 0, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultCancelled());
}

void TestOrderClientThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle eventHandle;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    TestData *data = reinterpret_cast<TestData*>(arg);

    pMsgBuffer = reinterpret_cast<nn::Bit32*>(g_ClientBuffer);
    msgSize = sizeof(g_ClientBuffer);

    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);

    while (g_Sequence != WaitThread)
    {
        nn::svc::SleepThread(SleepTime);
    }

    result = nn::svc::SendAsyncRequestWithUserBuffer(
                &eventHandle, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize,
                data->clientSession);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose eventCloser(eventHandle);

    result = nn::svc::SignalEvent(data->writableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // 結果がテストによって変わるので、result のチェックはしない
    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, -1);
}

void SendSyncRequestThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle* handle = reinterpret_cast<nn::svc::Handle*>(arg);
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();

    // ReceiveMultipleRequest のテストのため、クライアントのセッションを閉じる
    AutoHandleClose cSessionCloser(*handle);

    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 0, 0, 0));

        result = nn::svc::SendSyncRequest(*handle);
        ASSERT_RESULT_SUCCESS(result);
    }

    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
        ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
        ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
        ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
        ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
        ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
        ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
        ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
        ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);
    }
}

void SendSyncRequestWithUserBufferThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    UserBufferThreadData* data = reinterpret_cast<UserBufferThreadData*>(arg);

    // ReceiveMultipleRequest のテストのため、クライアントのセッションを閉じる
    AutoHandleClose cSessionCloser(data->clientSession);

    {
        nn::svc::ipc::MessageBuffer ipcMsg(data->userBuffer);
        ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 0, 0, 0));

        result = nn::svc::SendSyncRequestWithUserBuffer(
                    reinterpret_cast<uintptr_t>(data->userBuffer), data->bufferSize,
                    data->clientSession);
        ASSERT_RESULT_SUCCESS(result);
    }

    {
        nn::svc::ipc::MessageBuffer ipcMsg(data->userBuffer);
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
        ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
        ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
        ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
        ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
        ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
        ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
        ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
        ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);
    }
}

void SendAsyncRequestWithUserBufferThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle eventHandle;
    UserBufferThreadData* data = reinterpret_cast<UserBufferThreadData*>(arg);

    // ReceiveMultipleRequest のテストのため、クライアントのセッションを閉じる
    AutoHandleClose cSessionCloser(data->clientSession);

    {
        nn::svc::ipc::MessageBuffer ipcMsg(data->userBuffer);
        ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 0, 0, 0));

        result = nn::svc::SendAsyncRequestWithUserBuffer(
                    &eventHandle, reinterpret_cast<uintptr_t>(data->userBuffer),
                    data->bufferSize, data->clientSession);
    }

    // メモリが足りずに失敗することがある
    if (result.IsSuccess())
    {
        AutoHandleClose eventCloser(eventHandle);
        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
    }
    else
    {
        // ハンドラを作らないSync系で通信を行う
        result = nn::svc::SendSyncRequestWithUserBuffer(
                    reinterpret_cast<uintptr_t>(data->userBuffer),
                    data->bufferSize, data->clientSession);
    }

    {
        nn::svc::ipc::MessageBuffer ipcMsg(data->userBuffer);
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
        ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
        ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
        ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
        ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
        ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
        ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
        ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
        ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);
    }
}

void RemoveHandleFromArray(nn::svc::Handle *array, size_t size, int32_t index)
{
    for (int i = index; i < static_cast<int>(size - 1); i++)
    {
        array[i] = array[i + 1];
    }
    array[size - 1] = nn::svc::INVALID_HANDLE_VALUE;
}

} // namespace

/* TEST 68-243 */
/*
   事前にCancel されていると、replyTarget には結果を送信するけど、
   同期オブジェクトのシグナル待ちは行わない
 */
TEST(ReplyAndReceiveWithUserBuffer, OnlyReplyTest)
{
    TestSessionLeak leakTest;
    nn::Result result;

    // セッションの設定
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);

    // スレッドの設定
    uintptr_t pc[2];
    uintptr_t sp[2];
    uintptr_t serverParam = reinterpret_cast<uintptr_t>(&serverSession);
    uintptr_t clientParam = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

    pc[0] = reinterpret_cast<uintptr_t>(OnlyReplyTestServer);
    pc[1] = reinterpret_cast<uintptr_t>(OnlyReplyTestClient);
    sp[0] = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
    sp[1] = reinterpret_cast<uintptr_t>(g_Stack[1]) + sizeof(g_Stack[1]);

    TestThread serverThread(pc[0], serverParam, sp[0], priority, idealCore);
    TestThread clientThread(pc[1], clientParam, sp[1], priority, idealCore);

    g_Sequence = BeginThread;

    serverThread.Start();
    clientThread.Start();

    // ReplyAndReceiveWithUserBuffer で待っているスレッドをCancel する
    while (g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }
    ASSERT_TRUE(g_Sequence == RunningThread);

    result = nn::svc::CancelSynchronization(serverThread.GetHandle());
    ASSERT_RESULT_SUCCESS(result);

    // 終了処理
    g_Sequence = EndThread;

    serverThread.Wait();
    clientThread.Wait();
}

/* TEST 68-244 */
/*
   セッションのクローズとCancelの条件が整っていた場合、
   セッションのクローズが優先される
 */
TEST(ReplyAndReceiveWithUserBuffer, CancelAndSessionCloseOrderTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    int32_t index;

    // セッションの設定
    nn::svc::Handle serverSessions[2];
    nn::svc::Handle clientSessions[2];
    result = nn::svc::CreateSession(&serverSessions[0], &clientSessions[0], false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser0(serverSessions[0]);
    AutoHandleClose cSessionCloser0(clientSessions[0]);

    result = nn::svc::CreateSession(&serverSessions[1], &clientSessions[1], false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser1(serverSessions[1]);
    AutoHandleClose cSessionCloser1(clientSessions[1]);

    // イベントの設定
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;
    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose wEventCloser(writableEvent);
    AutoHandleClose rEventCloser(readableEvent);

    // スレッドの設定
    uintptr_t pc[2];
    uintptr_t sp[2];
    uintptr_t param[2];
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestData data;

    pc[0] = reinterpret_cast<uintptr_t>(TestCancelAndCloseSessionOrderThread);
    pc[1] = reinterpret_cast<uintptr_t>(TestOrderClientThread);
    sp[0] = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
    sp[1] = reinterpret_cast<uintptr_t>(g_Stack[1]) + sizeof(g_Stack[1]);
    param[0] = reinterpret_cast<uintptr_t>(serverSessions);
    param[1] = reinterpret_cast<uintptr_t>(&data);

    data.clientSession = clientSessions[0];
    data.writableEvent = writableEvent;

    g_Sequence = BeginThread;

    TestThread serverThread(pc[0], param[0], sp[0], priority, idealCore);
    TestThread clientThread(pc[1], param[1], sp[1], priority, idealCore);

    serverThread.Start();
    clientThread.Start();

    // ReplyAndReceive が呼ばれるのを待つ
    while (g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }
    ASSERT_TRUE(g_Sequence == RunningThread);

    // SendAsyncRequestWithUserBuffer が呼ばれるのを待つ
    g_Sequence = WaitThread;
    result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::ClearEvent(readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // 一旦 ReplyAndReceive を抜けさせるために処理を受け渡す
    nn::svc::SleepThread(SleepTime);

    // Cancel とセッションの終了なら、セッションの終了が優先される
    result = nn::svc::CancelSynchronization(serverThread.GetHandle());
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(clientSessions[0]);
    ASSERT_RESULT_SUCCESS(result);

    // 終了処理
    g_Sequence = EndThread;

    serverThread.Wait();
    clientThread.Wait();
}

/* TEST 68-245 */
/*
   セッションのクローズと同期オブジェクトのシグナルが整っていた場合、
   セッションのクローズが優先される
 */
TEST(ReplyAndReceiveWithUserBuffer, SignalAndCloseSessionOrderTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    int32_t index;

    // セッションの設定
    nn::svc::Handle serverSessions[2];
    nn::svc::Handle clientSessions[2];
    result = nn::svc::CreateSession(&serverSessions[0], &clientSessions[0], false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser0(serverSessions[0]);
    AutoHandleClose cSessionCloser0(clientSessions[0]);

    result = nn::svc::CreateSession(&serverSessions[1], &clientSessions[1], false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser1(serverSessions[1]);
    AutoHandleClose cSessionCloser1(clientSessions[1]);

    // イベントの設定
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;
    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose wEventCloser(writableEvent);
    AutoHandleClose rEventCloser(readableEvent);

    // スレッドの設定
    uintptr_t pc[2];
    uintptr_t sp[2];
    uintptr_t param[2];
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestData data;

    pc[0] = reinterpret_cast<uintptr_t>(TestSignalAndCloseSessionOrderThread);
    pc[1] = reinterpret_cast<uintptr_t>(TestOrderClientThread);
    sp[0] = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
    sp[1] = reinterpret_cast<uintptr_t>(g_Stack[1]) + sizeof(g_Stack[1]);
    param[0] = reinterpret_cast<uintptr_t>(serverSessions);
    param[1] = reinterpret_cast<uintptr_t>(&data);

    data.clientSession = clientSessions[0];
    data.writableEvent = writableEvent;

    g_Sequence = BeginThread;

    TestThread serverThread(pc[0], param[0], sp[0], priority, idealCore);
    TestThread clientThread(pc[1], param[1], sp[1], priority, idealCore);

    serverThread.Start();
    clientThread.Start();

    // ReplyAndReceive が呼ばれるのを待つ
    while (g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }
    ASSERT_TRUE(g_Sequence == RunningThread);

    // SendAsyncRequestWithUserBuffer が呼ばれるのを待つ
    g_Sequence = WaitThread;
    result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::ClearEvent(readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // 一旦 ReplyAndReceive が抜けるように、処理を受け渡す
    nn::svc::SleepThread(SleepTime);

    // 要求の受信とセッションの終了なら、セッションの終了が優先される
    result = nn::svc::CloseHandle(clientSessions[0]);
    ASSERT_RESULT_SUCCESS(result);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(g_TestBuffer);
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    nn::svc::Handle eventHandle;
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer),
            clientSessions[1]);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose eventCloser(eventHandle);

    // 終了処理
    g_Sequence = EndThread;

    serverThread.Wait();
    clientThread.Wait();
}

/* TEST 68-246 */
/*
   Cancel の条件と同期オブジェクトのシグナルが整っていた場合、
   同期オブジェクトのシグナルが優先される
 */
TEST(ReplyAndReceiveWithUserBuffer, CancelAndSignalOrderTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    int32_t index;

    // セッションの設定
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser0(serverSession);
    AutoHandleClose cSessionCloser0(clientSession);

    // イベントの設定
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;
    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose wEventCloser(writableEvent);
    AutoHandleClose rEventCloser(readableEvent);

    // スレッドの設定
    uintptr_t pc[2];
    uintptr_t sp[2];
    uintptr_t param[2];
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestData data;

    pc[0] = reinterpret_cast<uintptr_t>(TestCancelAndSignalOrderThread);
    pc[1] = reinterpret_cast<uintptr_t>(TestOrderClientThread);
    sp[0] = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
    sp[1] = reinterpret_cast<uintptr_t>(g_Stack[1]) + sizeof(g_Stack[1]);
    param[0] = reinterpret_cast<uintptr_t>(&serverSession);
    param[1] = reinterpret_cast<uintptr_t>(&data);

    data.clientSession = clientSession;
    data.writableEvent = writableEvent;

    g_Sequence = BeginThread;

    TestThread serverThread(pc[0], param[0], sp[0], priority, idealCore);
    TestThread clientThread(pc[1], param[1], sp[1], priority, idealCore);

    serverThread.Start();
    clientThread.Start();

    // ReplyAndReceive が呼ばれるのを待つ
    while (g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }
    ASSERT_TRUE(g_Sequence == RunningThread);

    // SendAsyncRequestWithUserBuffer が呼ばれるのを待つ
    g_Sequence = WaitThread;
    result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::ClearEvent(readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // 要求の受信とキャンセルなら、要求の受信が優先される
    result = nn::svc::CancelSynchronization(serverThread.GetHandle());
    ASSERT_RESULT_SUCCESS(result);

    // 終了処理
    g_Sequence = EndThread;

    serverThread.Wait();
    clientThread.Wait();
}

/* TEST 68-296 */
/*
   セッション閉じとタイムアウトが同時に成り立っていたら、
   セッション閉じが優先される
 */
TEST(ReplyAndReceiveWithUserBuffer, CloseAndTimeoutOrderTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    int32_t index;

    // セッションの設定
    nn::svc::Handle serverSessions[2];
    nn::svc::Handle clientSessions[2];
    result = nn::svc::CreateSession(&serverSessions[0], &clientSessions[0], false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser0(serverSessions[0]);
    AutoHandleClose cSessionCloser0(clientSessions[0]);

    result = nn::svc::CreateSession(&serverSessions[1], &clientSessions[1], false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser1(serverSessions[1]);
    AutoHandleClose cSessionCloser1(clientSessions[1]);

    // イベントの設定
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;
    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose wEventCloser(writableEvent);
    AutoHandleClose rEventCloser(readableEvent);

    // スレッドの設定
    uintptr_t pc[2];
    uintptr_t sp[2];
    uintptr_t param[2];
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestData data;

    pc[0] = reinterpret_cast<uintptr_t>(TestCloseAndTimeoutOrderThread);
    pc[1] = reinterpret_cast<uintptr_t>(TestOrderClientThread);
    sp[0] = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
    sp[1] = reinterpret_cast<uintptr_t>(g_Stack[1]) + sizeof(g_Stack[1]);
    param[0] = reinterpret_cast<uintptr_t>(serverSessions);
    param[1] = reinterpret_cast<uintptr_t>(&data);

    data.clientSession = clientSessions[0];
    data.writableEvent = writableEvent;

    g_Sequence = BeginThread;

    TestThread serverThread(pc[0], param[0], sp[0], priority, idealCore);
    TestThread clientThread(pc[1], param[1], sp[1], priority, idealCore);

    serverThread.Start();
    clientThread.Start();

    // ReplyAndReceive が呼ばれるのを待つ
    while (g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }
    ASSERT_TRUE(g_Sequence == RunningThread);

    // SendAsyncRequestWithUserBuffer が呼ばれるのを待つ
    g_Sequence = WaitThread;
    result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::ClearEvent(readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // 一旦 ReplyAndReceive が抜けるように、処理を受け渡す
    nn::svc::SleepThread(SleepTime);

    // セッションの終了とタイムアウトなら、セッションの終了が優先される
    result = nn::svc::CloseHandle(clientSessions[0]);
    ASSERT_RESULT_SUCCESS(result);
    nn::svc::Handle eventHandle;

    // 終了処理
    g_Sequence = EndThread;

    serverThread.Wait();
    clientThread.Wait();
}

/* TEST 68-297 */
/*
   同期オブジェクトのシグナル待機とタイムアウトが同時に成り立っていたら、
   同期オブジェクトのシグナルが優先される
 */
TEST(ReplyAndReceiveWithUserBuffer, SignalAndTimeoutOrderTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    int32_t index;

    // セッションの設定
    nn::svc::Handle serverSessions[2];
    nn::svc::Handle clientSessions[2];
    result = nn::svc::CreateSession(&serverSessions[0], &clientSessions[0], false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser0(serverSessions[0]);
    AutoHandleClose cSessionCloser0(clientSessions[0]);

    result = nn::svc::CreateSession(&serverSessions[1], &clientSessions[1], false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser1(serverSessions[1]);
    AutoHandleClose cSessionCloser1(clientSessions[1]);

    // イベントの設定
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;
    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose wEventCloser(writableEvent);
    AutoHandleClose rEventCloser(readableEvent);

    // スレッドの設定
    uintptr_t pc[2];
    uintptr_t sp[2];
    uintptr_t param[2];
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestData data;

    pc[0] = reinterpret_cast<uintptr_t>(TestSignalAndTimeoutOrderThread);
    pc[1] = reinterpret_cast<uintptr_t>(TestOrderClientThread);
    sp[0] = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
    sp[1] = reinterpret_cast<uintptr_t>(g_Stack[1]) + sizeof(g_Stack[1]);
    param[0] = reinterpret_cast<uintptr_t>(serverSessions);
    param[1] = reinterpret_cast<uintptr_t>(&data);

    data.clientSession = clientSessions[0];
    data.writableEvent = writableEvent;

    g_Sequence = BeginThread;

    TestThread serverThread(pc[0], param[0], sp[0], priority, idealCore);
    TestThread clientThread(pc[1], param[1], sp[1], priority, idealCore);

    serverThread.Start();
    clientThread.Start();

    // ReplyAndReceive が呼ばれるのを待つ
    while (g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }
    ASSERT_TRUE(g_Sequence == RunningThread);

    // SendAsyncRequestWithUserBuffer が呼ばれるのを待つ
    g_Sequence = WaitThread;
    result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::ClearEvent(readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // 要求の受信とタイムアウトなら、要求の受信が優先される
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(g_TestBuffer);
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    nn::svc::Handle eventHandle;
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &eventHandle, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer),
            clientSessions[1]);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose eventCloser(eventHandle);

    // 終了処理
    g_Sequence = EndThread;

    result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    serverThread.Wait();
    clientThread.Wait();
}

/* TEST 68-298 */
/*
   CancelSynchronization によるキャンセルとタイムアウトが同時に成り立っていたら、
   タイムアウトが優先される
 */
TEST(ReplyAndReceiveWithUserBuffer, CancelAndTimeoutOrderTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    int32_t index;

    // セッションの設定
    nn::svc::Handle serverSessions[2];
    nn::svc::Handle clientSessions[2];
    result = nn::svc::CreateSession(&serverSessions[0], &clientSessions[0], false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser0(serverSessions[0]);
    AutoHandleClose cSessionCloser0(clientSessions[0]);

    result = nn::svc::CreateSession(&serverSessions[1], &clientSessions[1], false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser1(serverSessions[1]);
    AutoHandleClose cSessionCloser1(clientSessions[1]);

    // イベントの設定
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;
    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose wEventCloser(writableEvent);
    AutoHandleClose rEventCloser(readableEvent);

    // スレッドの設定
    uintptr_t pc[2];
    uintptr_t sp[2];
    uintptr_t param[2];
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    TestData data;

    pc[0] = reinterpret_cast<uintptr_t>(TestCancelAndTimeoutOrderThread);
    pc[1] = reinterpret_cast<uintptr_t>(TestOrderClientThread);
    sp[0] = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
    sp[1] = reinterpret_cast<uintptr_t>(g_Stack[1]) + sizeof(g_Stack[1]);
    param[0] = reinterpret_cast<uintptr_t>(serverSessions);
    param[1] = reinterpret_cast<uintptr_t>(&data);

    data.clientSession = clientSessions[0];
    data.writableEvent = writableEvent;

    g_Sequence = BeginThread;

    TestThread serverThread(pc[0], param[0], sp[0], priority, idealCore);
    TestThread clientThread(pc[1], param[1], sp[1], priority, idealCore);

    serverThread.Start();
    clientThread.Start();

    // ReplyAndReceive が呼ばれるのを待つ
    while (g_Sequence == BeginThread)
    {
        nn::svc::SleepThread(SleepTime);
    }
    ASSERT_TRUE(g_Sequence == RunningThread);

    // SendAsyncRequestWithUserBuffer が呼ばれるのを待つ
    g_Sequence = WaitThread;
    result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::ClearEvent(readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // Cancel とタイムアウトなら、要求の受信が優先される
    result = nn::svc::CancelSynchronization(serverThread.GetHandle());
    ASSERT_RESULT_SUCCESS(result);

    // 終了処理
    g_Sequence = EndThread;

    serverThread.Wait();
    clientThread.Wait();
}


/* TEST 68-247 */
/*
   SendSyncRequest に対して返信できる
 */
TEST(ReplyAndReceiveWithUserBuffer, UseSendSyncRequest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle handles[1];
    int32_t index;

    // セッションの設定
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);

    handles[0] = serverSession;
    uintptr_t msgBufferAddr = reinterpret_cast<uintptr_t>(g_ServerBuffer);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(msgBufferAddr);
    size_t msgSize = sizeof(g_ServerBuffer);

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendSyncRequestThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
    uintptr_t clientParam = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

    TestThread thread(pc, clientParam, sp, priority, idealCore);
    thread.Start();

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());

    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, msgBufferAddr, msgSize,
            handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
    ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
    ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
    ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
    ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
    ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
    ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
    ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

    ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));

    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, msgBufferAddr, msgSize,
            handles, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    thread.Wait();
}

/* TEST 68-248 */
/*
   SendSyncRequestWithUserBuffer に対して返信できる
 */
TEST(ReplyAndReceiveWithUserBuffer, UseSendSyncRequestWithUserBuffer)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle handles[1];
    int32_t index;

    // セッションの設定
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);

    handles[0] = serverSession;
    uintptr_t msgBufferAddr = reinterpret_cast<uintptr_t>(g_ServerBuffer);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(msgBufferAddr);
    size_t msgSize = sizeof(g_ServerBuffer);

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendSyncRequestWithUserBufferThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    UserBufferThreadData data;
    uintptr_t clientParam = reinterpret_cast<uintptr_t>(&data);
    data.clientSession = clientSession;
    data.userBuffer = reinterpret_cast<nn::Bit32*>(g_ClientBuffer);
    data.bufferSize = sizeof(g_ClientBuffer);

    TestThread thread(pc, clientParam, sp, priority, idealCore);
    thread.Start();

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());

    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, msgBufferAddr, msgSize,
            handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
    ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
    ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
    ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
    ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
    ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
    ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
    ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

    ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));

    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, msgBufferAddr, msgSize,
            handles, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    thread.Wait();
}

/* TEST 68-249 */
/*
   SendAsyncRequestWithUserBuffer に対して返信できる
 */
TEST(ReplyAndReceiveWithUserBuffer, UseSendAsyncRequestWithUserBuffer)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle handles[1];
    int32_t index;

    // セッションの設定
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);

    handles[0] = serverSession;
    uintptr_t msgBufferAddr = reinterpret_cast<uintptr_t>(g_ServerBuffer);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(msgBufferAddr);
    size_t msgSize = sizeof(g_ServerBuffer);

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendAsyncRequestWithUserBufferThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;
    UserBufferThreadData data;
    uintptr_t clientParam = reinterpret_cast<uintptr_t>(&data);
    data.clientSession = clientSession;
    data.userBuffer = reinterpret_cast<nn::Bit32*>(g_ClientBuffer);
    data.bufferSize = sizeof(g_ClientBuffer);

    TestThread thread(pc, clientParam, sp, priority, idealCore);
    thread.Start();

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());

    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, msgBufferAddr, msgSize,
            handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
    ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
    ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
    ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
    ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
    ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
    ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
    ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
    ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

    ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));

    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, msgBufferAddr, msgSize,
            handles, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    thread.Wait();
}

/* TEST 68-250 */
/*
   結果を送信するのと、要求を受信するのを同時に行うことが出来る
 */
TEST(ReplyAndReceiveWithUserBuffer, ReceiveAndReplyTest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    nn::svc::Handle handles[1];
    int32_t index;

    // セッションの設定
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;
    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);
    AutoHandleClose cSessionCloser(clientSession);

    handles[0] = serverSession;
    uintptr_t msgBufferAddr = reinterpret_cast<uintptr_t>(g_ServerBuffer);
    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(msgBufferAddr);
    size_t msgSize = sizeof(g_ServerBuffer);

    // スレッドの設定
    uintptr_t pc = reinterpret_cast<uintptr_t>(SendThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
    uintptr_t clientParam = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

    g_Sequence = BeginThread;

    TestThread thread(pc, clientParam, sp, priority, idealCore);
    thread.Start();

    // 一度受信しておく
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Receive);
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, msgBufferAddr, msgSize,
            handles, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);

    // 何度も送受信を繰り返す
    for (int i = 0; i < 10; i++)
    {
        SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Reply);
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, msgBufferAddr, msgSize,
                handles, 1, serverSession, -1);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(GetIpcTag(pMsgBuffer), DefaultIpcTag_Send);
    }

    // 終了処理
    g_Sequence = EndThread;
    result = nn::svc::ReplyAndReceiveWithUserBuffer(
            &index, msgBufferAddr, msgSize,
            handles, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    thread.Wait();
}

/* TEST 68-251 */
/*
   複数のセッションを待つことが出来る
 */
TEST(ReplyAndReceiveWithUserBuffer, ReceiveMultipleRequest)
{
    TestSessionLeak leakTest;
    nn::Result result;
    int32_t index;

    // ヒープの設定
    size_t stackSize = 0x1000;
    size_t bufferSize = 0x1000;
    size_t wantSize =
        (sizeof(nn::svc::Handle) * 2 + sizeof(UserBufferThreadData) + bufferSize + stackSize)
        * nn::svc::ArgumentHandleCountMax;
    TestHeap heap(wantSize);
    uintptr_t heapPtr = heap.GetAddress();
    size_t heapSize = heap.GetSize();

    // IPC のユーザーバッファは4KB アライメントになっている必要があるので、先頭に持ってくる
    char* userBuffers = reinterpret_cast<char*>(heapPtr);
    nn::svc::Handle* threadHandles =
        reinterpret_cast<nn::svc::Handle*>(
                heapPtr + bufferSize
                * nn::svc::ArgumentHandleCountMax);
    nn::svc::Handle* serverSessions =
        reinterpret_cast<nn::svc::Handle*>(
                heapPtr + (bufferSize + sizeof(nn::svc::Handle))
                * nn::svc::ArgumentHandleCountMax);
    UserBufferThreadData* userData =
        reinterpret_cast<UserBufferThreadData*>(
                heapPtr + (bufferSize + sizeof(nn::svc::Handle)  * 2)
                * nn::svc::ArgumentHandleCountMax);

    // スレッドを作れるだけ作る
    int32_t threadNum;
    const int funcNum = 3; // スレッドが利用する関数の数
    uintptr_t pc[funcNum] = {
        reinterpret_cast<uintptr_t>(SendSyncRequestThread),
        reinterpret_cast<uintptr_t>(SendSyncRequestWithUserBufferThread),
        reinterpret_cast<uintptr_t>(SendAsyncRequestWithUserBufferThread),
    };

    for (threadNum = 0; threadNum < nn::svc::ArgumentHandleCountMax; threadNum++)
    {
        UserBufferThreadData *data = &userData[threadNum];
        result = nn::svc::CreateSession(
                &serverSessions[threadNum], &data->clientSession, false, 0);
        if (result.IsFailure())
        {
            break;
        }

        uintptr_t sp = heapPtr + heapSize - stackSize * threadNum;
        int32_t priority = threadNum % NumPriority;
        int32_t idealCore = threadNum % NumCore;
        uintptr_t param = reinterpret_cast<uintptr_t>(data);

        if (threadNum % funcNum != 0)
        {
            data->userBuffer = reinterpret_cast<nn::Bit32*>(
                    &userBuffers[bufferSize * threadNum]);
            data->bufferSize = bufferSize;
        }

        result = nn::svc::CreateThread(
                &threadHandles[threadNum], pc[threadNum % funcNum], param,
                sp, priority, idealCore);

        if (result.IsFailure())
        {
            result = nn::svc::CloseHandle(serverSessions[threadNum]);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::CloseHandle(userData[threadNum].clientSession);
            ASSERT_RESULT_SUCCESS(result);
            break;
        }
        result = nn::svc::StartThread(threadHandles[threadNum]);
        ASSERT_RESULT_SUCCESS(result);
    }

    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(g_ServerBuffer);
    size_t msgSize = sizeof(g_ServerBuffer);

    // 複数のセッションから要求を受け付ける
    nn::svc::Handle targetHandle = nn::svc::INVALID_HANDLE_VALUE;
    index = -1;
    for (int32_t i = 0; i < threadNum; i++)
    {
        SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Receive);
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize,
                serverSessions, threadNum - i, targetHandle, -1);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index >= 0 && index < threadNum);

        if (targetHandle != nn::svc::INVALID_HANDLE_VALUE)
        {
            result = nn::svc::CloseHandle(targetHandle);
            ASSERT_RESULT_SUCCESS(result);
        }

        // 受信確認
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);

            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);
        }

        // 送信データ作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
        }

        targetHandle = serverSessions[index];
        RemoveHandleFromArray(serverSessions, threadNum - i, index);
    }

    // 残っているスレッドに結果を送信する
    if (targetHandle != nn::svc::INVALID_HANDLE_VALUE)
    {
        SetOnlyIpcTag(pMsgBuffer, 0x0002);
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                &index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize,
                serverSessions, 0, targetHandle, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        result = nn::svc::CloseHandle(targetHandle);
        ASSERT_RESULT_SUCCESS(result);
    }

    // 終了処理
    for (int32_t i = 0; i < threadNum; i++)
    {
        result = nn::svc::WaitSynchronization(&index, &threadHandles[i], 1, -1);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(threadHandles[i]);
        ASSERT_RESULT_SUCCESS(result);
    }
} // NOLINT (readability/fn_size)

