﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_TestIoMapping.h"
#include "test_QueryIoMapping.h"
#include "test_Common.h"
#include "util_TestLoader.h"
#include "util_TestProcess.h"
#include <nn/svc/ipc/svc_SessionMessage.h>

extern char BinQueryIoMapping_begin[];
extern char BinQueryIoMapping_end[];

namespace {
class NamedPortManager
{
    public:
        NamedPortManager(const char* name, int32_t maxSession)
            : m_PortName(name)
            , m_MaxSession(maxSession)
        {
            nn::Result result;
            result = nn::svc::ManageNamedPort(&m_Handle, m_PortName, 0);
            NN_ASSERT(result.IsSuccess() || result <= nn::svc::ResultNotFound());

            NN_LOG("Register named port: '%s'\n", m_PortName);
            result = nn::svc::ManageNamedPort(&m_Handle, m_PortName, m_MaxSession);
            if (result.IsFailure())
            {
                DebugResult(result);
            }
            NN_ASSERT_RESULT_SUCCESS(result);
        }

        ~NamedPortManager()
        {
            nn::Result result;
            result = nn::svc::CloseHandle(m_Handle);
            NN_ASSERT_RESULT_SUCCESS(result);

            nn::svc::Handle tmpHandle;
            result = nn::svc::ManageNamedPort(&tmpHandle, m_PortName, 0);
            NN_ASSERT_RESULT_SUCCESS(result);
            NN_ASSERT(tmpHandle == nn::svc::INVALID_HANDLE_VALUE);
        }

        const char* GetName() const
        {
            return m_PortName;
        }
        int32_t GetMaxSession() const
        {
            return m_MaxSession;
        }
        nn::svc::Handle GetHandle() const
        {
            return m_Handle;
        }

    private:
        const char* m_PortName;
        int32_t m_MaxSession;
        nn::svc::Handle m_Handle;
};

void MakeProcess(nn::svc::Handle* pOut, uint64_t testAddr, size_t testSize, bool isReadWrite)
{
    nn::Result result;
    int32_t index;

    // 名前付きポートの設定
    NamedPortManager portManager(PortName, 1);
    nn::svc::Handle namedPort = portManager.GetHandle();

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();

    uintptr_t begin = reinterpret_cast<uintptr_t>(BinQueryIoMapping_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinQueryIoMapping_end);
    TestLoader loader(BinQueryIoMapping_begin, end - begin);

    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    int32_t maxFlag = heap.GetSize() / sizeof(nn::Bit32);

    MakeNo6Flag(flags, maxFlag, testAddr, testSize, isReadWrite, true);
    loader.SetAdditionalCapability(flags, 2);

    nn::svc::Handle clientProcess;
    loader.SpawnProcess(&clientProcess);
    loader.StartProcess(clientProcess);

    result = nn::svc::WaitSynchronization(&index, &namedPort, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, namedPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);

    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        ipcMsg.SetNull();
        result = nn::svc::ReplyAndReceive(
                &index, &serverSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);

        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
        nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

        ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);

        ipcMsg.SetNull();
        int offset = ipcMsg.Set(
                nn::svc::ipc::MessageBuffer::MessageHeader(
                    0x0002, 0, 0, 0, 0, 0, 4, 0));
        nn::Bit32 data[4];
        data[0] = testAddr >> 32;
        data[1] = (testAddr << 32) >> 32;
        data[2] = testSize;
        data[3] = isReadWrite ? 1 : 0;

        offset = ipcMsg.SetRawArray(
                offset, reinterpret_cast<void*>(data), sizeof(nn::Bit32) * 4);

        result = nn::svc::ReplyAndReceive(&index, &serverSession, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }

    *pOut = clientProcess;
}

}

// TEST 125-29
// ホールになっている物理アドレスに対して、QueryIoMapping を行うことが出来る
TEST(QueryIoMapping, CheckHoleArea)
{
    nn::Result result;
    nn::svc::Handle processHandle;

    {
        MakeProcess(&processHandle, 0x60030000, 0x10000, true);
        AutoHandleClose processCloser(processHandle);
        WaitProcess(processHandle);
    }

    {
        MakeProcess(&processHandle, 0x60030000, 0x10000, false);
        AutoHandleClose processCloser(processHandle);
        WaitProcess(processHandle);
    }
}

// TEST 125-30
// 連続にマッピングされている領域の先頭アドレスを取得できる
TEST(QueryIoMapping, AlignTest)
{
    nn::Result result;

    nn::svc::Handle processHandle;

    // ページテーブルのブロックサイズを考慮して 64 M まで実施する
    size_t maxSize = 0x4000000; // 64M
    uint64_t ioAddress = GetIoAddress(maxSize);
    for (size_t size = 0x1000; size <= maxSize; size *= 2)
    {
        {
            MakeProcess(&processHandle, ioAddress, size, true);
            AutoHandleClose processCloser(processHandle);
            WaitProcess(processHandle);
        }

        {
            MakeProcess(&processHandle, ioAddress, size, false);
            AutoHandleClose processCloser(processHandle);
            WaitProcess(processHandle);
        }
    }
}

