﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Thread.h>
#include <cstring>

#if !defined(NN_SDK_BUILD_RELEASE)

TEST(OutputDebugString, StringTest)
{
    nn::Result result;

    // TEST 34-1
    // ascii 文字列が出力できる
    result = nn::svc::OutputDebugString("Hello, world\n", 13);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 34-2
    // UTF-8 文字列が出力できる
    result = nn::svc::OutputDebugString("こんにちは\n", 16);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 34-3
    // 途中にヌル文字が入っていても、続けて出力される
    result = nn::svc::OutputDebugString("Hello,\n\0 world\n", 15);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 34-4
    // NULL を渡されると失敗する
    result = nn::svc::OutputDebugString(NULL, 1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
}

TEST(OutputDebugString, SizeTest)
{
    nn::Result result;

    // TEST 34-5
    // 文字列よりも小さいサイズが指定されると、サイズ分しか文字列が出力されない
    result = nn::svc::OutputDebugString("Hello,\n world\n", 7);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(OutputDebugString, MemoryPermissionTest)
{
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    uintptr_t heapPtr;
    char* strPtr;

    // TEST 34-6
    // 読み込み権限がないメモリアドレスを指定する
    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, g_FreeAreaBegin);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Free);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);

    result = nn::svc::OutputDebugString(reinterpret_cast<char*>(g_FreeAreaBegin), 1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());

    // TEST 34-7
    // 途中で読み込み権限がないメモリ領域を指定する
    result = nn::svc::SetHeapSize(&heapPtr, HeapAlign);
    ASSERT_RESULT_SUCCESS(result);

    strPtr = reinterpret_cast<char*>(heapPtr + 0x1000 - 1);
    strncpy(strPtr, "Hello, world\n", 14);

    result = nn::svc::OutputDebugString(strPtr, 13);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetMemoryPermission(heapPtr + 0x1000, 0x1000, nn::svc::MemoryPermission_None);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::QueryMemory(&blockInfo, &pageInfo, heapPtr + 0x1000);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);

    result = nn::svc::OutputDebugString(strPtr, 13);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());

    result = nn::svc::SetMemoryPermission(
            heapPtr + 0x1000, 0x1000, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetHeapSize(&heapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);
}

#endif // !defined(NN_SDK_BUILD_RELEASE)

