﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "test_MemoryState.h"
#include "util_TestIpc.h"
#include "util_TestMemory.h"
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/init.h>

char g_Buffer[0x1000]  __attribute__((aligned(NN_ALIGNOF(std::max_align_t))));
int32_t g_ProcessIdealCore;
uintptr_t g_HeapAreaBegin;
uintptr_t g_HeapAreaEnd;
uintptr_t g_ReservedAreaBegin;
uintptr_t g_ReservedAreaEnd;
uintptr_t g_FreeAreaBegin;
uintptr_t g_FreeAreaEnd;

extern "C" void nndiagStartup()
{
}

extern "C" void nninitStartup()
{
}

void AllocateMemoryArea(uintptr_t heapAddr, size_t heapSize, nn::svc::Handle clientSession)
{
    uint64_t addr;
    uint64_t size;
    nn::svc::MemoryState request;
    ClientReceiveData(&request, sizeof(request), clientSession);
    TestMemoryState* state = nullptr;

    switch(request)
    {
    case nn::svc::MemoryState_Free:
        state = new(g_Buffer) TestFreeMemoryState();
        break;
    case nn::svc::MemoryState_Io:
        state = new(g_Buffer) TestIoMemoryState();
        break;
    case nn::svc::MemoryState_Static:
        state = new(g_Buffer) TestStaticMemoryState();
        break;
    case nn::svc::MemoryState_Code:
        state = new(g_Buffer) TestCodeMemoryState();
        break;
    case nn::svc::MemoryState_CodeData:
        state = new(g_Buffer) TestCodeDataMemoryState();
        break;
    case nn::svc::MemoryState_Normal:
        state = new(g_Buffer) TestNormalMemoryState();
        break;
    case nn::svc::MemoryState_Shared:
        state = new(g_Buffer) TestSharedMemoryState();
        break;
    case nn::svc::MemoryState_Alias:
        state = new(g_Buffer) TestAliasMemoryState();
        break;
    case nn::svc::MemoryState_AliasCode:
        state = new(g_Buffer) TestAliasCodeMemoryState();
        break;
    case nn::svc::MemoryState_AliasCodeData:
        state = new(g_Buffer) TestAliasCodeDataMemoryState();
        break;
    case nn::svc::MemoryState_Ipc:
        state = new(g_Buffer) TestIpcMemoryState();
        break;
    case nn::svc::MemoryState_NonSecureIpc:
        state = new(g_Buffer) TestNonSecureIpcMemoryState();
        break;
    case nn::svc::MemoryState_NonDeviceIpc:
        state = new(g_Buffer) TestNonDeviceIpcMemoryState();
        break;
    case nn::svc::MemoryState_Stack:
        state = new(g_Buffer) TestStackMemoryState();
        break;
    case nn::svc::MemoryState_ThreadLocal:
        state = new(g_Buffer) TestThreadLocalMemoryState();
        break;
    case nn::svc::MemoryState_Transfered:
        state = new(g_Buffer) TestTransferedMemoryState();
        break;
    case nn::svc::MemoryState_SharedTransfered:
        state = new(g_Buffer) TestSharedTransferedMemoryState();
        break;
    case nn::svc::MemoryState_SharedCode:
        state = new(g_Buffer) TestSharedCodeMemoryState();
        break;
    case nn::svc::MemoryState_Inaccessible:
        state = new(g_Buffer) TestInaccessibleMemoryState();
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    state->SetHeap(heapAddr, heapSize);
    state->Initialize();
    addr = state->GetAddress();
    size = state->GetSize();

    ClientSendData(clientSession, &addr, sizeof(addr));
    ClientSendData(clientSession, &size, sizeof(size));
}

void FreeMemoryArea()
{
    TestMemoryState* state = reinterpret_cast<TestMemoryState*>(g_Buffer);
    state->Close();
}

void AllocateNormalArea(uintptr_t* pHeapAddr, size_t* pHeapSize, nn::svc::Handle clientSession)
{
    NN_ASSERT(*pHeapAddr + *pHeapSize >= *pHeapAddr);

    uint64_t heapAddr = *pHeapAddr;
    uint64_t size = 0;

    ClientReceiveData(&size, sizeof(size), clientSession);
    NN_ASSERT(size <= *pHeapSize);
    ClientSendData(clientSession, &heapAddr, sizeof(heapAddr));
    *pHeapAddr += size;
    *pHeapSize -= size;
}

void SetUncached(nn::svc::Handle clientSession)
{
    uint64_t addr;
    uint64_t size;
    ClientReceiveData(&addr, sizeof(addr), clientSession);
    ClientReceiveData(&size, sizeof(size), clientSession);

    nn::Result result;
    result = nn::svc::SetMemoryAttribute(
            static_cast<uintptr_t>(addr), static_cast<size_t>(size), nn::svc::MemoryAttribute_Uncached, nn::svc::MemoryAttribute_Uncached);
    NN_ASSERT_RESULT_SUCCESS(result);

    // シグナルの代わり
    ClientSendData(clientSession, &addr, sizeof(addr));
}

void UnsetUncached(nn::svc::Handle clientSession)
{
    uint64_t addr;
    uint64_t size;
    ClientReceiveData(&addr, sizeof(addr), clientSession);
    ClientReceiveData(&size, sizeof(size), clientSession);

    nn::Result result;
    result = nn::svc::SetMemoryAttribute(static_cast<uintptr_t>(addr), static_cast<size_t>(size), nn::svc::MemoryAttribute_Uncached, 0);
    NN_ASSERT_RESULT_SUCCESS(result);

    // シグナルの代わり
    ClientSendData(clientSession, &addr, sizeof(addr));
}

extern "C" void nnMain()
{
    InitTestMemory();

    nn::Result result;

    nn::svc::Handle clientSession;

    {
        // Reserved 領域の取得
        nn::Bit64 tmpAddr;
        nn::Bit64 tmpSize;
        result = nn::svc::GetInfo(
                &tmpAddr, nn::svc::InfoType_ReservedRegionAddress,
                nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0);
        NN_ASSERT_RESULT_SUCCESS(result);
        g_ReservedAreaBegin = static_cast<uintptr_t>(tmpAddr);

        result = nn::svc::GetInfo(
                &tmpSize, nn::svc::InfoType_ReservedRegionSize,
                nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0);
        NN_ASSERT_RESULT_SUCCESS(result);
        NN_ASSERT(tmpSize != 0);
        g_ReservedAreaEnd = g_ReservedAreaBegin + tmpSize - 1;
    }

    result = nn::svc::ConnectToNamedPort(&clientSession, MemoryStateProcessName);
    NN_ASSERT_RESULT_SUCCESS(result);

    TestHeap heap(HeapAlign);
    uintptr_t heapAddr = heap.GetAddress();
    size_t heapSize = heap.GetSize();

    bool isEnd = false;
    while(!isEnd)
    {
        TestTag tag;
        ClientReceiveData(&tag, sizeof(tag), clientSession);
        switch(static_cast<TestTag>(tag))
        {
            case TestTag_Allocate:
                AllocateMemoryArea(heapAddr, heapSize, clientSession);
                break;
            case TestTag_Free:
                FreeMemoryArea();
                break;
            case TestTag_GetNormal:
                AllocateNormalArea(&heapAddr, &heapSize, clientSession);
                break;
            case TestTag_SetUncached:
                SetUncached(clientSession);
                break;
            case TestTag_UnsetUncached:
                UnsetUncached(clientSession);
                break;
            case TestTag_End:
                isEnd = true;
                break;
            default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

