﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_MemoryState.h"
#include "util_TestMemory.h"
#include <nn/svc/svc_Dd.h>
#include <cstring>

extern "C" void nnMain();
char g_Buffer[0x1000] __attribute__((aligned(0x1000)));

namespace {
void CheckSuccessMemoryArea(
        TestMemoryInfo** array, int numState, nn::svc::Handle handle,
        nn::svc::MemoryPermission perm)
{
    nn::Result result;

    for (int i = 0; i < numState; i++)
    {
        size_t memSize = 0x2000;

        ASSERT_TRUE(array[i]->GetSize() >= memSize && ((array[i]->GetSize() & 0xfff) == 0));
        array[i]->Initialize();

        uintptr_t addr = array[i]->GetAddress();
        result = nn::svc::MapSharedMemory(handle, addr, memSize, perm);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::UnmapSharedMemory(handle, addr, memSize);
        ASSERT_RESULT_SUCCESS(result);

        array[i]->Close();
    }
}

void CheckInvalidMemoryArea(
        TestMemoryInfo** array, int numState, nn::svc::Handle handle,
        nn::svc::MemoryPermission perm)
{
    nn::Result result;

    for (int i = 0; i < numState; i++)
    {
        size_t memSize = 0x2000;
        if (!(array[i]->GetSize() >= memSize && ((array[i]->GetSize() & 0xfff) == 0)))
        {
            array[i]->Close();
            continue;
        }

        array[i]->Initialize();

        uintptr_t addr = array[i]->GetAddress();
        result = nn::svc::MapSharedMemory(handle, addr, memSize, perm);
        if (IsInAslrRegion(addr))
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());
        }

        array[i]->Close();
    }
}
}



TEST(MapSharedMemory, InvalidHandleTest)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size = 0x1000;

    // TEST 17-15
    result = nn::svc::MapSharedMemory(nn::svc::INVALID_HANDLE_VALUE, g_FreeAreaBegin,
            size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 17-16
    result =  nn::svc::MapSharedMemory(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, g_FreeAreaBegin,
            size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 17-17
    result =  nn::svc::MapSharedMemory(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, g_FreeAreaBegin,
            size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

// TEST 17-1
// Free領域に対して、マップを行うことが出来る
TEST(MapSharedMemory, SuccessTest)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    nn::svc::MemoryInfo info;
    size_t size = 0x1000;

    nn::svc::MemoryPermission validPermission[] = {
        nn::svc::MemoryPermission_ReadWrite,
        nn::svc::MemoryPermission_Read,
    };

    for (
            int32_t i = 0;
            i < static_cast<int32_t>(sizeof(validPermission) / sizeof(nn::svc::MemoryPermission));
            i++)
    {
        result = nn::svc::CreateSharedMemory(
                &handle, size, validPermission[i], validPermission[i]);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&info, g_FreeAreaBegin);
        ASSERT_TRUE(info.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(info.attribute == 0);

        result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin, size, validPermission[i]);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&info, g_FreeAreaBegin);
        ASSERT_TRUE(info.state == nn::svc::MemoryState_Shared);
        ASSERT_TRUE(info.permission == validPermission[i]);
        ASSERT_TRUE(info.attribute == 0);

        result = nn::svc::UnmapSharedMemory(handle, g_FreeAreaBegin, size);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
}
TEST(MapSharedMemory, Rw_InvalidAddressTest)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size;

    size = 0x1000;
    result = nn::svc::CreateSharedMemory(
            &handle, size, nn::svc::MemoryPermission_ReadWrite, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 17-18
    // アライメントされていないアドレスを指定すると失敗する
    for (int i = 1; i < 0x1000; i++)
    {
        result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin + i,
                size, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }

    // 終了処理
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(MapSharedMemory, Ro_InvalidAddressTest)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size;

    size = 0x1000;
    result = nn::svc::CreateSharedMemory(
            &handle, size, nn::svc::MemoryPermission_Read, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 17-18
    // アライメントされていないアドレスを指定すると失敗する
    for (int i = 1; i < 0x1000; i++)
    {
        result = nn::svc::MapSharedMemory(handle, g_FreeAreaBegin + i,
                size, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());
    }

    // 終了処理
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(MapSharedMemory, Rw_InvalidMemoryStateTest)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size = 0x2000;

    // 前準備
    nn::svc::MemoryPermission perm = nn::svc::MemoryPermission_ReadWrite;
    result = nn::svc::CreateSharedMemory(&handle, size, perm, perm);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sharedCloser(handle);

    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_Free,
    };
    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));

    // 許可されたメモリ状態の領域を受け付ける
    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListWithStates(successArea, successState, NumSuccessState);

    CheckSuccessMemoryArea(successArea, NumSuccessState, handle, perm);

    // 許可されていないメモリ状態の領域を受け付けない
    const int NumInvalid = NumTestMemoryState - NumSuccessState;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successState, NumSuccessState);

    CheckInvalidMemoryArea(invalidArea, NumInvalid, handle, perm);

    {
        TestNormalMemoryState normal;
        normal.Initialize();
        uintptr_t addr;

        // メモリ状態が一様でないと失敗する
        // 前方
        addr = normal.GetAddress() - (size / 2);
        result = nn::svc::MapSharedMemory(handle, addr, size, perm);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() || result <= nn::svc::ResultInvalidRegion());

        // 後方
        addr = normal.GetAddress() + normal.GetSize() - (size / 2);
        result = nn::svc::MapSharedMemory(handle, addr, size, perm);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() || result <= nn::svc::ResultInvalidRegion());

        // TEST 17-42
        // オーバーフローするようなaddr/size の組み合わせは受け付けない
        addr = normal.GetAddress() + size;
        size = static_cast<size_t>(-size);
        result = nn::svc::MapSharedMemory(handle, addr, size, perm);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidSize() ||
                result <= nn::svc::ResultInvalidCurrentMemory() || result <= nn::svc::ResultInvalidRegion());
        normal.Close();
    }
}

TEST(MapSharedMemory, Ro_InvalidMemoryStateTest)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size = 0x2000;

    // 前準備
    nn::svc::MemoryPermission perm = nn::svc::MemoryPermission_Read;
    result = nn::svc::CreateSharedMemory(&handle, size, perm, perm);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sharedCloser(handle);

    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_Free,
    };
    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));

    // 許可されたメモリ状態の領域を受け付ける
    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListWithStates(successArea, successState, NumSuccessState);

    CheckSuccessMemoryArea(successArea, NumSuccessState, handle, perm);

    // 許可されていないメモリ状態の領域を受け付けない
    const int NumInvalid = NumTestMemoryState - NumSuccessState;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successState, NumSuccessState);

    CheckInvalidMemoryArea(invalidArea, NumInvalid, handle, perm);

    {
        TestNormalMemoryState normal;
        normal.Initialize();
        uintptr_t addr;

        // メモリ状態が一様でないと失敗する
        // 前方
        addr = normal.GetAddress() - (size / 2);
        result = nn::svc::MapSharedMemory(handle, addr, size, perm);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() || result <= nn::svc::ResultInvalidRegion());

        // 後方
        addr = normal.GetAddress() + normal.GetSize() - (size / 2);
        result = nn::svc::MapSharedMemory(handle, addr, size, perm);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() || result <= nn::svc::ResultInvalidRegion());

        // TEST 17-42
        // オーバーフローするようなaddr/size の組み合わせは受け付けない
        addr = normal.GetAddress() + size;
        size = static_cast<size_t>(-size);
        result = nn::svc::MapSharedMemory(handle, addr, size, perm);
        ASSERT_TRUE(result <= nn::svc::ResultInvalidSize() ||
                result <= nn::svc::ResultInvalidCurrentMemory() || result <= nn::svc::ResultInvalidRegion());

        normal.Close();
    }
}

TEST(MapSharedMemory, Rw_InvalidMemoryRegion)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size;

    size = 0x1000;
    result = nn::svc::CreateSharedMemory(
            &handle, size, nn::svc::MemoryPermission_ReadWrite, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sharedCloser(handle);

    // Reserved 領域
    {
        result = nn::svc::MapSharedMemory(handle, g_ReservedAreaBegin,
                size, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());
    }

    // Large 領域
    {
        uintptr_t addr;
        uint64_t largeAddr;

        bool check = GetLargeFreeArea(&largeAddr, size);
#if defined NN_BUILD_CONFIG_CPU_ARM_V7A
        ASSERT_TRUE(!check);
#else
        ASSERT_TRUE(check);
#endif
        if(check)
        {
            addr = static_cast<uintptr_t>(largeAddr);
            result = nn::svc::MapSharedMemory(handle, addr, size, nn::svc::MemoryPermission_ReadWrite);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::UnmapSharedMemory(handle, addr, size);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
}

TEST(MapSharedMemory, Ro_InvalidMemoryRegion)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size;

    size = 0x1000;
    result = nn::svc::CreateSharedMemory(
            &handle, size, nn::svc::MemoryPermission_Read, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sharedCloser(handle);

    // Reserved 領域
    {
        result = nn::svc::MapSharedMemory(handle, g_ReservedAreaBegin,
                size, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());
    }

    // Large 領域
    {
        uintptr_t addr;
        uint64_t largeAddr;

        bool check = GetLargeFreeArea(&largeAddr, size);
#if defined NN_BUILD_CONFIG_CPU_ARM_V7A
        ASSERT_TRUE(!check);
#else
        ASSERT_TRUE(check);
#endif
        if(check)
        {
            addr = static_cast<uintptr_t>(largeAddr);
            result = nn::svc::MapSharedMemory(handle, addr, size, nn::svc::MemoryPermission_Read);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::UnmapSharedMemory(handle, addr, size);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
}

TEST(MapSharedMemory, Rw_InvalidMemoryPermissionTest)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size;

    // 前準備
    size = 0x1000;
    result = nn::svc::CreateSharedMemory(
            &handle, size, nn::svc::MemoryPermission_ReadWrite, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);


    // TEST 17-34
    // MemoryPermission_None を指定できない
    result = nn::svc::MapSharedMemory(
            handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_None);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 17-35
    // MemoryPermission_Write を指定できない
    result = nn::svc::MapSharedMemory(
            handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_Write);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 17-36
    // MemoryPermission_Execute を指定できない
    result = nn::svc::MapSharedMemory(
            handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_Execute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 17-37
    // MemoryPermission_DontCare を指定できない
    result = nn::svc::MapSharedMemory(
            handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_DontCare);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 17-44
    // MemoryPermission_ReadExecute を指定できない
    result = nn::svc::MapSharedMemory(
            handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // 終了処理
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(MapSharedMemory, Ro_InvalidMemoryPermissionTest)
{
    TestSharedMemoryLeak leakTest;
    nn::svc::Handle handle;
    nn::Result result;
    size_t size;

    // 前準備
    size = 0x1000;
    result = nn::svc::CreateSharedMemory(
            &handle, size, nn::svc::MemoryPermission_Read, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);


    // TEST 17-38
    // MemoryPermission_None を指定できない
    result = nn::svc::MapSharedMemory(
            handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_None);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 17-39
    // MemoryPermission_Write を指定できない
    result = nn::svc::MapSharedMemory(
            handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_Write);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 17-40
    // MemoryPermission_Write を指定できない
    result = nn::svc::MapSharedMemory(
            handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_Execute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 17-41
    // MemoryPermission_DontCare を指定できない
    result = nn::svc::MapSharedMemory(
            handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_DontCare);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 17-42
    // MemoryPermission_ReadWrite を指定できない
    result = nn::svc::MapSharedMemory(
            handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 17-45
    // MemoryPermission_ReadExecute を指定できない
    result = nn::svc::MapSharedMemory(
            handle, g_FreeAreaBegin, size, nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // 終了処理
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

