﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include "util_MemoryState.h"
#include "util_TestMemory.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/TargetConfigs/build_Compiler.h>
#include <cstring>

extern "C" void nnMain();

namespace {
} // namespace


TEST(MapProcessCodeMemory, NormalCase)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    uint64_t size = 0x1000;
    uint64_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, size);

    CheckProcessMemory(
            processHandle, toAddr,
            nn::svc::MemoryState_Free, nn::svc::MemoryPermission_None, 0);
    CheckProcessMemory(
            processHandle, fromAddr,
            nn::svc::MemoryState_Normal, nn::svc::MemoryPermission_ReadWrite, 0);

    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 119-55
    // マップされた fromAddr の領域が MemoryPermission_None である
    // TEST 119-65
    // マップされた fromAddr の領域が MemoryAttribute_Locked である
    CheckProcessMemory(
            processHandle, fromAddr,
            nn::svc::MemoryState_Normal, nn::svc::MemoryPermission_None,
            nn::svc::MemoryAttribute_Locked);

    // TEST 119-57
    // マップされた toAddr の領域が MemoryPermission_None である
    // TEST 119-58
    // マップされた toAddr の領域が MemoryState_AliasCode である
    // TEST 119-66
    // マップされた toAddr の領域はメモリ属性がない
    CheckProcessMemory(
            processHandle, toAddr,
            nn::svc::MemoryState_AliasCode, nn::svc::MemoryPermission_None, 0);

    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    CheckProcessMemory(
            processHandle, toAddr,
            nn::svc::MemoryState_Free, nn::svc::MemoryPermission_None, 0);
    CheckProcessMemory(
            processHandle, fromAddr,
            nn::svc::MemoryState_Normal, nn::svc::MemoryPermission_ReadWrite, 0);
}

// TEST 119-59
// MemoryState_Normal の領域のどこであっても、マップすることが出来る
TEST(MapProcessCodeMemory, MapAnyWhereTest)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    uint64_t size = 0x1000;
    uint64_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, HeapAlign);

    CheckProcessMemory(
            processHandle, toAddr,
            nn::svc::MemoryState_Free, nn::svc::MemoryPermission_None, 0);
    CheckProcessMemory(
            processHandle, fromAddr,
            nn::svc::MemoryState_Normal, nn::svc::MemoryPermission_ReadWrite, 0);

    // 先頭
    result = nn::svc::MapProcessCodeMemory(
            processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapProcessCodeMemory(
            processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    // 中央
    result = nn::svc::MapProcessCodeMemory(
            processHandle, toAddr, fromAddr + HeapAlign / 2, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapProcessCodeMemory(
            processHandle, toAddr, fromAddr + HeapAlign / 2, size);
    ASSERT_RESULT_SUCCESS(result);

    // 終端
    result = nn::svc::MapProcessCodeMemory(
            processHandle, toAddr, fromAddr + HeapAlign - size, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::UnmapProcessCodeMemory(
            processHandle, toAddr, fromAddr + HeapAlign - size, size);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(MapProcessCodeMemory, MergeBlockTest)
{
    nn::Result result;

    TestMemoryStateProcess stateProcess;
    nn::svc::Handle processHandle = stateProcess.GetHandle();

    uint64_t size = 0x1000;
    uint64_t toAddr = g_FreeAreaBegin;
    uint64_t fromAddr;
    stateProcess.GetNormalArea(&fromAddr, size * 2);

    result = nn::svc::MapProcessCodeMemory(processHandle, toAddr, fromAddr, size);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::MapProcessCodeMemory(
            processHandle, toAddr + size, fromAddr + size, size);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 119-60
    // 連続した領域をマップすると、ブロックがマージされる
    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;
    result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, fromAddr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress == fromAddr);
    ASSERT_TRUE(blockInfo.size == size * 2);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);

    result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, processHandle, toAddr);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.baseAddress == toAddr);
    ASSERT_TRUE(blockInfo.size == size * 2);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_AliasCode);
    ASSERT_TRUE(blockInfo.attribute == 0);

    result = nn::svc::UnmapProcessCodeMemory(processHandle, toAddr, fromAddr, size * 2);
    ASSERT_RESULT_SUCCESS(result);
}

