﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include "util_MemoryState.h"
#include "util_TestMemory.h"
#include <nn/svc/svc_BaseId.autogen.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/ipc/svc_SessionMessage.h>

extern "C" void nnMain();

namespace {
char g_Buffer[2][0x2000] __attribute__((aligned(0x1000)));
char g_TestBuffer[0x1000];
const uint64_t SpaceAddr = 0;

const nn::svc::MemoryPermission EnablePermissionOnRead[] = {
    nn::svc::MemoryPermission_Read,
};
const nn::svc::MemoryPermission EnablePermissionOnReadWrite[] = {
    nn::svc::MemoryPermission_Read,
    nn::svc::MemoryPermission_Write,
    nn::svc::MemoryPermission_ReadWrite,
};
const nn::svc::MemoryPermission EnablePermissionOnReadExecute[] = {
    nn::svc::MemoryPermission_Read,
};

void CheckSuccessMemoryArea(TestMemoryInfo** array, int numState, nn::svc::Handle handle)
{
    nn::Result result;

    for (int i = 0; i < numState; i++)
    {
        ASSERT_TRUE(array[i]->GetSize() > 0 && ((array[i]->GetSize() & 0xfff) == 0));

        array[i]->Initialize();
        uintptr_t addr = array[i]->GetAddress();
        uintptr_t deviceAddress = 0x0;
        size_t outSize;
        const nn::svc::MemoryPermission* permArray;
        int numPerm = 0;

        switch(array[i]->GetPermission())
        {
        case nn::svc::MemoryPermission_Read:
            permArray = EnablePermissionOnRead;
            numPerm = sizeof(EnablePermissionOnRead) / sizeof(nn::svc::MemoryPermission);
            break;
        case nn::svc::MemoryPermission_ReadWrite:
            permArray = EnablePermissionOnReadWrite;
            numPerm = sizeof(EnablePermissionOnReadWrite) / sizeof(nn::svc::MemoryPermission);
            break;
        case nn::svc::MemoryPermission_ReadExecute:
            permArray = EnablePermissionOnReadExecute;
            numPerm = sizeof(EnablePermissionOnReadExecute) / sizeof(nn::svc::MemoryPermission);
            break;
        default: FAIL();
        }

        for (int j = 0; j < numPerm; j++)
        {
            result = nn::svc::MapDeviceAddressSpace( &outSize,
                    handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                    addr, array[i]->GetSize(), deviceAddress,
                    permArray[j]);
            ASSERT_RESULT_SUCCESS(result);

            CheckMemory(
                    addr,
                    array[i]->GetState(),
                    array[i]->GetPermission(),
                    nn::svc::MemoryAttribute_DeviceShared | array[i]->GetAttribute());

            result = nn::svc::UnmapDeviceAddressSpace(
                    handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                    addr, outSize, deviceAddress);
            ASSERT_RESULT_SUCCESS(result);

            array[i]->CheckDefaultState();
        }

        array[i]->Close();
    }
}

void CheckInvalidMemoryArea(TestMemoryInfo** array, int numState, nn::svc::Handle handle)
{
    nn::Result result;

    for (int i = 0; i < numState; i++)
    {
        if (!(array[i]->GetSize() > 0 && ((array[i]->GetSize() & 0xfff) == 0)))
        {
            array[i]->Close();
            continue;
        }

        array[i]->Initialize();
        uintptr_t addr =  array[i]->GetAddress();
        uintptr_t deviceAddress = 0x0;
        size_t outSize;

        result = nn::svc::MapDeviceAddressSpace( &outSize,
                    handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                    addr, array[i]->GetSize(), deviceAddress,
                    nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        array[i]->CheckDefaultState();

        array[i]->Close();
    }
}
} // namespace

TEST(MapDeviceAddressSpace, AddressSpaceHandleTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    uint64_t size = 0x80000000;
    uint64_t deviceAddress = 0;
    uint64_t align = 0x400000;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, size);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    TestHeap heap(align * 2);
    uintptr_t addr = RoundUp<uintptr_t>(heap.GetAddress(), align);
    size_t mapSize = 0x1000;
    size_t outSize;

    // TEST 132-1
    // CreateDeviceAddressSpace のハンドルを受け付ける
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 133-1
    // CreateDeviceAddressSpace のハンドルを受け付ける
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, outSize, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 132-2
    // INVALID_HANDLE_VALUE を受け付けない
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            nn::svc::INVALID_HANDLE_VALUE, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE(result);

    // TEST 132-3
    // スレッドの擬似ハンドルを受け付けない
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE(result);

    // TEST 132-4
    // プロセスの擬似ハンドルを受け付けない
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE(result);

    // TEST 132-5
    // Close したハンドラを受け付けない
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE(result);
}

TEST(MapDeviceAddressSpace, ProcessHandleTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t size = 0x80000000;
    uint64_t deviceAddress = 0;
    uint64_t align = 0x400000;
    nn::svc::Handle handle;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, size);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    TestHeap heap(align * 2);
    uintptr_t addr = RoundUp<uintptr_t>(heap.GetAddress(), align);
    size_t mapSize = 0x1000;
    size_t outSize;

    // TEST 132-6
    // CreateDeviceAddressSpace のハンドルを受け付けない
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, handle,
            addr, mapSize, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE(result);

    // Process
    {
        TestMemoryStateProcess stateProcess;
        nn::svc::Handle processHandle = stateProcess.GetHandle();

        size_t size = 0x1000;
        uint64_t addr;
        stateProcess.GetNormalArea(&addr, size);

        // TEST 132-7
        // プロセスのハンドルを受け付ける
        result = nn::svc::MapDeviceAddressSpace( &outSize,
                handle, processHandle,
                addr, size, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 133-7
        // CreateDeviceAddressSpace のハンドルを受け付けない
        result = nn::svc::UnmapDeviceAddressSpace(
                handle, handle,
                addr, outSize, deviceAddress);
        ASSERT_RESULT_FAILURE(result);

        // TEST 133-8
        // プロセスのハンドルを受け付ける
        result = nn::svc::UnmapDeviceAddressSpace(
                handle, processHandle,
                addr, outSize, 0);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::TerminateProcess(processHandle);
        ASSERT_RESULT_SUCCESS(result);

        stateProcess.SetDead();

        result = nn::svc::CloseHandle(processHandle);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 132-11
        // Close したハンドラを受け付けない
        result = nn::svc::MapDeviceAddressSpace( &outSize,
                handle, processHandle,
                addr, size, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE(result);

        // TEST 133-12
        // Close したハンドラを受け付けない
        result = nn::svc::UnmapDeviceAddressSpace(
                handle, processHandle,
                addr, outSize, deviceAddress);
        ASSERT_RESULT_FAILURE(result);
    }

    // TEST 132-10
    // プロセスの擬似ハンドルを受け付ける
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, mapSize, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 133-11
    // プロセスの擬似ハンドルを受け付ける
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, outSize, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 132-8
    // INVALID_HANDLE_VALUE を受け付けない
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::INVALID_HANDLE_VALUE,
            addr, mapSize, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE(result);

    // TEST 132-9
    // スレッドの擬似ハンドルを受け付けない
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD,
            addr, mapSize, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE(result);
}

TEST(MapDeviceAddressSpace, ProcessAddressTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    nn::svc::Handle handle;
    size_t align = 0x400000;
    uint64_t deviceAddress = 0;
    nn::svc::MemoryInfo blockInfo;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    uintptr_t heapAddr;
    size_t heapSize = align * 2;

    TestHeap heap(align * 2);
    ASSERT_TRUE(heapSize == heap.GetSize());
    heapAddr = (heap.GetAddress() + (align - 1)) & ~(align - 1);

    nn::svc::PhysicalMemoryInfo physInfo = { 0 };
    result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
    ASSERT_RESULT_SUCCESS(result);
    // 4MBアラインされるはず
    uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
    ASSERT_TRUE((physAddr & (align - 1)) == 0);

    heapSize -= heapAddr - heap.GetAddress();

    // TEST 132-12
    // 4KB にアライメントされている値を受け付ける
    uint64_t size = 0x1000;
    size_t outSize;
    for (uint64_t addr = 0; addr < heapSize; addr += 0x1000)
    {
        result = nn::svc::MapDeviceAddressSpace( &outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr + addr, size, deviceAddress + addr,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr + addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == (nn::svc::MemoryAttribute_DeviceShared));

        // TEST 133-13
        // MapDeviceAddressSpace と同じアドレスを受け付ける
        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr + addr, outSize, deviceAddress + addr);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr + addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }

    // TEST 132-13
    // 4KB にアライメントされていないと失敗する
    for (uint64_t addr = heapAddr + 1; addr < heapAddr + 0x1000; addr++)
    {
        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::MapDeviceAddressSpace( &outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr, size, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE(result);

        GetMemoryInfo(&blockInfo, addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }
}

TEST(MapDeviceAddressSpace, SizeTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    uint64_t deviceAddress = 0;
    nn::svc::Handle handle;
    size_t align = 0x400000;
    nn::svc::MemoryInfo blockInfo;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    uintptr_t heapAddr;
    size_t heapSize = align * 2;

    TestHeap heap(align * 2);
    ASSERT_TRUE(heapSize == heap.GetSize());
    heapAddr = (heap.GetAddress() + (align - 1)) & ~(align - 1);

    nn::svc::PhysicalMemoryInfo physInfo = { 0 };
    result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
    ASSERT_RESULT_SUCCESS(result);
    // 4MBアラインされるはず
    uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
    ASSERT_TRUE((physAddr & (align - 1)) == 0);

    heapSize -= heapAddr - heap.GetAddress();

    // TEST 132-12
    // 4KB にアライメントされている値を受け付ける
    size_t outSize;
    for (uint64_t size = 0x1000; size < heapSize; size += 0x1000)
    {
        result = nn::svc::MapDeviceAddressSpace( &outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, size, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

        // TEST 133-15
        // MapDeviceAddressSpace と同じサイズを受け付ける
        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, outSize, deviceAddress);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }

    // TEST 132-15
    // 4KB にアライメントされていないと失敗する
    for (uint64_t size = 1; size < 0x1000; size++)
    {
        result = nn::svc::MapDeviceAddressSpace( &outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, size, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }

    // TEST 132-16
    // 0 を受け付けない
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            heapAddr, 0, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

    GetMemoryInfo(&blockInfo, heapAddr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);
}

TEST(MapDeviceAddressSpace, DeviceAddressTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    size_t align = 0x400000;
    uint64_t spaceSize = align * 2;
    nn::svc::MemoryInfo blockInfo;
    size_t outSize;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    uintptr_t heapAddr;
    size_t heapSize = align * 2;

    TestHeap heap(spaceSize);
    ASSERT_TRUE(heapSize == heap.GetSize());
    heapAddr = (heap.GetAddress() + (align - 1)) & ~(align - 1);

    nn::svc::PhysicalMemoryInfo physInfo = { 0 };
    result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
    ASSERT_RESULT_SUCCESS(result);
    // 4MBアラインされるはず
    uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
    ASSERT_TRUE((physAddr & (align - 1)) == 0);

    heapSize -= heapAddr - heap.GetAddress();

    // TEST 132-17
    // 4KB にアライメントされている値を受け付ける
    uint64_t size = 0x1000;
    for (uint64_t addr = 0x1000; addr < heapSize; addr += 0x1000)
    {
        result = nn::svc::MapDeviceAddressSpace( &outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr + addr, size, addr,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr + addr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

        // TEST 133-18
        // MapDeviceAddressSpace と同じアドレスを受け付ける
        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr + addr, outSize, addr);
        ASSERT_RESULT_SUCCESS(result);
        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }

    // TEST 132-18
    // 4KB にアライメントされていないと失敗する
    for (uint64_t addr = 1; addr < 0x1000; addr++)
    {
        result = nn::svc::MapDeviceAddressSpace( &outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, size, addr,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());

        nn::svc::MemoryInfo blockInfo;
        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }

    // TEST 132-61
    // 0 を受け付ける
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            heapAddr, size, 0,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 133-42
    // 0 を受け付ける
    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            heapAddr, outSize, 0);
    ASSERT_RESULT_SUCCESS(result);

    // ProcessAddress と下位 22bit が揃っていなくても成功する
    for (uint64_t addr = 0x1000; addr < 0x400000; addr += 0x1000)
    {
        result = nn::svc::MapDeviceAddressSpace( &outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, size, addr,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::MemoryInfo blockInfo;
        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                heapAddr, outSize, addr);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }
}

TEST(MapDeviceAddressSpace, ProcessAreaTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;

    uint64_t spaceSize = 0x80000000;
    uint64_t deviceAddress = 0;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    uintptr_t addr;
    uint64_t size = 0x1000;
    size_t outSize;

    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_Normal,
        nn::svc::MemoryState_Stack,
        nn::svc::MemoryState_Alias,
        nn::svc::MemoryState_Code,
        nn::svc::MemoryState_CodeData,
        nn::svc::MemoryState_AliasCode,
#ifdef SUPPORT_ALIAS_CODE_DATA
        nn::svc::MemoryState_AliasCodeData,
#endif
        nn::svc::MemoryState_Ipc,
        nn::svc::MemoryState_NonSecureIpc,
        nn::svc::MemoryState_NonDeviceIpc,
        nn::svc::MemoryState_Transfered,
        nn::svc::MemoryState_SharedTransfered,
    };

    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    // 許可されたメモリ状態の領域を受け付ける
    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListWithStates(successArea, successState, NumSuccessState);

    CheckSuccessMemoryArea(successArea, NumSuccessState, handle);

    // 許可されていないメモリ状態の領域を受け付けない
    const int NumInvalid = NumTestMemoryState - NumSuccessState;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successState, NumSuccessState);

    CheckInvalidMemoryArea(invalidArea, NumInvalid, handle);

    uint32_t successAttribute[] = {
        0,
        nn::svc::MemoryAttribute_Uncached,
        nn::svc::MemoryAttribute_DeviceShared,
        nn::svc::MemoryAttribute_Uncached | nn::svc::MemoryAttribute_DeviceShared,
    };
    const int NumSuccessAttr = sizeof(successAttribute) / sizeof(nn::svc::MemoryAttribute);
    TestMemoryInfo* successAttrArea[NumSuccessAttr];

    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));
    header->GetTestListWithAttributes(successAttrArea, successAttribute, NumSuccessAttr);
    CheckSuccessMemoryArea(successAttrArea, NumSuccessAttr, handle);

    const int NumInvalidAttr = NumTestMemoryAttribute - NumSuccessAttr;
    TestMemoryInfo* invalidAttr[NumInvalidAttr];
    header->GetTestListExceptAttributes(
            invalidAttr, NumInvalidAttr, successAttribute, NumSuccessAttr);
    CheckInvalidMemoryArea(invalidAttr, NumInvalidAttr, handle);

    {
        // TEST 132-34
        // メモリ状態が一様でないと失敗する
        TestTransferedMemoryState transfer;
        transfer.Initialize();
        ASSERT_TRUE(size * 2 <= transfer.GetSize());

        // 前方
        addr = transfer.GetAddress() - size;

        result = nn::svc::MapDeviceAddressSpace( &outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr, size * 2, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        transfer.CheckDefaultState();

        // 後方
        addr = transfer.GetAddress() + transfer.GetSize() - size;

        result = nn::svc::MapDeviceAddressSpace( &outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr, size * 2, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        transfer.CheckDefaultState();

        // TEST 132-35
        // オーバーフローする組み合わせを指定できない
        addr = transfer.GetAddress() + size;
        size = reinterpret_cast<uint64_t>(-size);
        result = nn::svc::MapDeviceAddressSpace( &outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr, size, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        transfer.CheckDefaultState();

        addr = transfer.GetAddress();
        size = 0x1000;
        result = nn::svc::SetMemoryAttribute(
                addr, size,
                nn::svc::MemoryAttribute_Uncached,
                nn::svc::MemoryAttribute_Uncached);
        ASSERT_RESULT_SUCCESS(result);
        CheckMemory(
                addr,
                transfer.GetState(),
                transfer.GetPermission(),
                nn::svc::MemoryAttribute_Uncached);

        // TEST 132-67
        // MemoryAttribute_Locked 以外の属性が付いている場合は受け付ける
        result = nn::svc::MapDeviceAddressSpace( &outSize,
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr, size, deviceAddress,
                nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        CheckMemory(
                addr,
                transfer.GetState(),
                nn::svc::MemoryPermission_ReadWrite,
                (nn::svc::MemoryAttribute_Uncached |
                 nn::svc::MemoryAttribute_DeviceShared));

        result = nn::svc::UnmapDeviceAddressSpace(
                handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                addr, size, deviceAddress);
        ASSERT_RESULT_SUCCESS(result);

        CheckMemory(
                addr,
                transfer.GetState(),
                transfer.GetPermission(),
                nn::svc::MemoryAttribute_Uncached);

        result = nn::svc::SetMemoryAttribute(
                addr, size,
                nn::svc::MemoryAttribute_Uncached, 0);
        ASSERT_RESULT_SUCCESS(result);

        transfer.Close();
    }

    // Permission
    {
        TestNormalMemoryState normal;
        normal.Initialize();
        addr = normal.GetAddress();
        ASSERT_TRUE(size <= normal.GetSize());

        // Read
        {
            TestMemoryPermission perm(addr, size, nn::svc::MemoryPermission_Read);
            result = nn::svc::MapDeviceAddressSpace( &outSize,
                    handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                    addr, size, deviceAddress,
                    nn::svc::MemoryPermission_ReadWrite);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }

        // None
        {
            TestMemoryPermission perm(addr, size, nn::svc::MemoryPermission_None);
            result = nn::svc::MapDeviceAddressSpace( &outSize,
                    handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                    addr, size, deviceAddress,
                    nn::svc::MemoryPermission_ReadWrite);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }

        normal.Close();
    }
} // NOLINT(readability/fn_size)

TEST(MapDeviceAddressSpace, DeviceAreaTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::Handle handle;

    uint64_t spaceSize = 0x80000000;
    uint64_t deviceAddress = 0x0;
    size_t align = 0x400000;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    uintptr_t addr;
    TestHeap heap(align * 3);
    size_t outSize;

    // TEST 132-36
    // 未マップ状態だとマップできる
    addr = (heap.GetAddress() + align - 1) & ~(align - 1);
    deviceAddress = align;
    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, align, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);

    // TEST 132-37
    // 既にマップされている領域は指定できない
    addr += align;
    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);

    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, align, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);

    // TEST 132-38
    // メモリ状態が一様でないと失敗する
    addr -= align;
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, align * 2, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

    GetMemoryInfo(&blockInfo, addr + align);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);

    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, align, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);

    // TEST 132-39
    // オーバーフローする組み合わせを指定できない
    deviceAddress = 0xfffffffffffff000ul & ~(align - 1);
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, align, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(result <= nn::svc::ResultInvalidCurrentMemory() ||
            result <= nn::svc::ResultInvalidRegion());

    // TEST 132-40
    // CreateAddressSpace で作成したアドレス空間から出てしまう領域は指定できない
    deviceAddress = spaceSize;
    size_t size = 0x1000;
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidRegion());

}

TEST(MapDeviceAddressSpace, DevicePermissionTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    nn::svc::MemoryInfo blockInfo;
    nn::svc::Handle handle;

    uint64_t spaceSize = 0x80000000;
    uint64_t deviceAddress = 0;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    size_t align = 0x400000;
    TestHeap heap(align * 2);

    uintptr_t addr = (heap.GetAddress() + align - 1) & ~(align - 1);
    uint64_t size = 0x1000;
    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);

    size_t outSize;
    // TEST 132-48
    // MemoryPermission_None を受け付けない
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress,
            nn::svc::MemoryPermission_None);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());
    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);

    // TEST 132-49
    // MemoryPermission_Read を受け付ける
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress,
            nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);
    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);
    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);

    // TEST 132-50
    // MemoryPermission_Write を受け付ける
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress,
            nn::svc::MemoryPermission_Write);
    ASSERT_RESULT_SUCCESS(result);
    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);
    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);

    // TEST 132-51
    // MemoryPermission_Execute を受け付けない
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress,
            nn::svc::MemoryPermission_Execute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 132-52
    // MemoryPermission_ReadWrite を受け付ける
    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress,
            nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);
    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

    result = nn::svc::UnmapDeviceAddressSpace(
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress);
    ASSERT_RESULT_SUCCESS(result);
    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);

    // TEST 132-53
    // MemoryPermission_ReadExecute を受け付けない
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress,
            nn::svc::MemoryPermission_ReadExecute);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidNewMemoryPermission());

    // TEST 132-54
    // MemoryPermission_DontCare を受け付けない
    result = nn::svc::MapDeviceAddressSpace( &outSize,
            handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
            addr, size, deviceAddress,
            nn::svc::MemoryPermission_DontCare);
    ASSERT_RESULT_FAILURE(result);
}

TEST(MapDeviceAddressSpace, OverlapTest0)
{
    // map0 -> map1 -> unmap1 -> unmap0
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    nn::svc::Handle handle0;
    nn::svc::Handle handle1;
    size_t align = 0x400000;
    uint64_t deviceAddress = 0;
    nn::svc::MemoryInfo blockInfo;
    size_t outSize;

    result = nn::svc::CreateDeviceAddressSpace(&handle0, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser0(handle0);

    result = nn::svc::CreateDeviceAddressSpace(&handle1, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser1(handle1);

    uintptr_t heapAddr;
    size_t heapSize = align * 2;

    TestHeap heap(heapSize);
    ASSERT_TRUE(heapSize == heap.GetSize());
    heapAddr = (heap.GetAddress() + (align - 1)) & ~(align - 1);

    nn::svc::PhysicalMemoryInfo physInfo = { 0 };
    result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
    ASSERT_RESULT_SUCCESS(result);
    // 4MBアラインされるはず
    uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
    ASSERT_TRUE((physAddr & (align - 1)) == 0);

    heapSize -= heapAddr - heap.GetAddress();

    size_t areaSize = 0x5000;

    for (size_t i = 0; i < areaSize / 0x1000; i++)
    {
        size_t mapOffset0 = 0x1000 * i;
        for (size_t j = 1; j <= (areaSize - mapOffset0) / 0x1000; j++)
        {
            size_t mapSize0 = 0x1000 * j;
            for (size_t k = 0; k < areaSize / 0x1000; k++)
            {
                size_t mapOffset1 = 0x1000 * k;
                for (size_t l = 1; l <= (areaSize - mapOffset1) / 0x1000; l++)
                {
                    size_t mapSize1 = 0x1000 * l;

                    result = nn::svc::MapDeviceAddressSpace( &outSize,
                            handle0, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                            heapAddr + mapOffset0, mapSize0, deviceAddress + mapOffset0,
                            nn::svc::MemoryPermission_ReadWrite);
                    ASSERT_RESULT_SUCCESS(result);

                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                    ASSERT_TRUE(blockInfo.size == mapSize0);
                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                    result = nn::svc::MapDeviceAddressSpace( &outSize,
                            handle1, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                            heapAddr + mapOffset1, mapSize1, deviceAddress + mapOffset1,
                            nn::svc::MemoryPermission_ReadWrite);
                    ASSERT_RESULT_SUCCESS(result);

                    if (mapOffset0 == mapOffset1)
                    {
                        if (mapSize0 == mapSize1)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapSize0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapSize0 < mapSize1)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapSize0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0 + mapSize0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0 + mapSize0);
                            ASSERT_TRUE(blockInfo.size == mapOffset1 + mapSize1 - (mapOffset0 + mapSize0));
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapSize1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1 + mapSize1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1 + mapSize1);
                            ASSERT_TRUE(blockInfo.size == mapOffset0 + mapSize0 - (mapOffset1 + mapSize1));
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                    }
                    else if (mapOffset0 < mapOffset1)
                    {
                        if (mapOffset0 + mapSize0 < mapOffset1)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapSize0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapSize1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapOffset0 + mapSize0 == mapOffset1)
                        {
                            // 連結される
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapSize0 + mapSize1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapOffset0 + mapSize0 < mapOffset1 + mapSize1)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapOffset1 - mapOffset0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapOffset0 + mapSize0 - mapOffset1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0 + mapSize0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0 + mapSize0);
                            ASSERT_TRUE(blockInfo.size == mapOffset1 + mapSize1 - (mapOffset0 + mapSize0));
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapOffset0 + mapSize0 == mapOffset1 + mapSize1)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapOffset1 - mapOffset0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapOffset0 + mapSize0 - mapOffset1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapOffset1 + mapSize1 < mapOffset0 + mapSize0)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapOffset1 - mapOffset0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapSize1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1 + mapSize1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1 + mapSize1);
                            ASSERT_TRUE(blockInfo.size == mapOffset0 + mapSize0 - (mapOffset1 + mapSize1));
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                    }
                    else if (mapOffset1 < mapOffset0)
                    {
                        if (mapOffset1 + mapSize1 < mapOffset0)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapSize1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapSize0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapOffset1 + mapSize1 == mapOffset0)
                        {
                            // 連結される
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapSize1 + mapSize0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapOffset1 + mapSize1 < mapOffset0 + mapSize0)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapOffset0 - mapOffset1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapOffset1 + mapSize1 - mapOffset0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1 + mapSize1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1 + mapSize1);
                            ASSERT_TRUE(blockInfo.size == mapOffset0 + mapSize0 - (mapOffset1 + mapSize1));
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapOffset1 + mapSize1 == mapOffset0 + mapSize0)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapOffset0 - mapOffset1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapOffset1 + mapSize1 - mapOffset0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapOffset0 + mapSize0 < mapOffset1 + mapSize1)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapOffset0 - mapOffset1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapSize0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0 + mapSize0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0 + mapSize0);
                            ASSERT_TRUE(blockInfo.size == mapOffset1 + mapSize1 - (mapOffset0 + mapSize0));
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                    }

                    result = nn::svc::UnmapDeviceAddressSpace(
                            handle1, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                            heapAddr + mapOffset1, mapSize1, deviceAddress + mapOffset1);
                    ASSERT_RESULT_SUCCESS(result);

                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                    ASSERT_TRUE(blockInfo.size == mapSize0);
                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                    result = nn::svc::UnmapDeviceAddressSpace(
                            handle0, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                            heapAddr + mapOffset0, mapSize0, deviceAddress + mapOffset0);
                    ASSERT_RESULT_SUCCESS(result);

                    GetMemoryInfo(&blockInfo, heapAddr);
                    ASSERT_TRUE(blockInfo.baseAddress == heap.GetAddress());
                    ASSERT_TRUE(blockInfo.size == heap.GetSize());
                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                    ASSERT_TRUE(blockInfo.attribute == 0);
                }
            }
        }
    }
} // NOLINT(readability/fn_size)

TEST(MapDeviceAddressSpace, OverlapTest1)
{
    // map0 -> map1 -> unmap0 -> unmap1
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    nn::svc::Handle handle0;
    nn::svc::Handle handle1;
    size_t align = 0x400000;
    uint64_t deviceAddress = 0;
    nn::svc::MemoryInfo blockInfo;

    result = nn::svc::CreateDeviceAddressSpace(&handle0, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser0(handle0);

    result = nn::svc::CreateDeviceAddressSpace(&handle1, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser1(handle1);

    uintptr_t heapAddr;
    size_t heapSize = align * 2;

    TestHeap heap(heapSize);
    ASSERT_TRUE(heapSize == heap.GetSize());
    heapAddr = (heap.GetAddress() + (align - 1)) & ~(align - 1);

    nn::svc::PhysicalMemoryInfo physInfo = { 0 };
    result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
    ASSERT_RESULT_SUCCESS(result);
    // 4MBアラインされるはず
    uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
    ASSERT_TRUE((physAddr & (align - 1)) == 0);

    heapSize -= heapAddr - heap.GetAddress();

    size_t areaSize = 0x5000;
    size_t outSize;

    for (size_t i = 0; i < areaSize / 0x1000; i++)
    {
        size_t mapOffset0 = 0x1000 * i;
        for (size_t j = 1; j <= (areaSize - mapOffset0) / 0x1000; j++)
        {
            size_t mapSize0 = 0x1000 * j;
            for (size_t k = 0; k < areaSize / 0x1000; k++)
            {
                size_t mapOffset1 = 0x1000 * k;
                for (size_t l = 1; l <= (areaSize - mapOffset1) / 0x1000; l++)
                {
                    size_t mapSize1 = 0x1000 * l;

                    result = nn::svc::MapDeviceAddressSpace( &outSize,
                            handle0, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                            heapAddr + mapOffset0, mapSize0, deviceAddress + mapOffset0,
                            nn::svc::MemoryPermission_ReadWrite);
                    ASSERT_RESULT_SUCCESS(result);

                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                    ASSERT_TRUE(blockInfo.size == mapSize0);
                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                    result = nn::svc::MapDeviceAddressSpace( &outSize,
                            handle1, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                            heapAddr + mapOffset1, mapSize1, deviceAddress + mapOffset1,
                            nn::svc::MemoryPermission_ReadWrite);
                    ASSERT_RESULT_SUCCESS(result);

                    if (mapOffset0 == mapOffset1)
                    {
                        if (mapSize0 == mapSize1)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapSize0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapSize0 < mapSize1)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapSize0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0 + mapSize0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0 + mapSize0);
                            ASSERT_TRUE(blockInfo.size == mapOffset1 + mapSize1 - (mapOffset0 + mapSize0));
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapSize1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1 + mapSize1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1 + mapSize1);
                            ASSERT_TRUE(blockInfo.size == mapOffset0 + mapSize0 - (mapOffset1 + mapSize1));
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                    }
                    else if (mapOffset0 < mapOffset1)
                    {
                        if (mapOffset0 + mapSize0 < mapOffset1)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapSize0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapSize1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapOffset0 + mapSize0 == mapOffset1)
                        {
                            // 連結される
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapSize0 + mapSize1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapOffset0 + mapSize0 < mapOffset1 + mapSize1)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapOffset1 - mapOffset0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapOffset0 + mapSize0 - mapOffset1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0 + mapSize0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0 + mapSize0);
                            ASSERT_TRUE(blockInfo.size == mapOffset1 + mapSize1 - (mapOffset0 + mapSize0));
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapOffset0 + mapSize0 == mapOffset1 + mapSize1)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapOffset1 - mapOffset0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapOffset0 + mapSize0 - mapOffset1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapOffset1 + mapSize1 < mapOffset0 + mapSize0)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapOffset1 - mapOffset0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapSize1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1 + mapSize1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1 + mapSize1);
                            ASSERT_TRUE(blockInfo.size == mapOffset0 + mapSize0 - (mapOffset1 + mapSize1));
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                    }
                    else if (mapOffset1 < mapOffset0)
                    {
                        if (mapOffset1 + mapSize1 < mapOffset0)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapSize1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapSize0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapOffset1 + mapSize1 == mapOffset0)
                        {
                            // 連結される
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapSize1 + mapSize0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapOffset1 + mapSize1 < mapOffset0 + mapSize0)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapOffset0 - mapOffset1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapOffset1 + mapSize1 - mapOffset0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1 + mapSize1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1 + mapSize1);
                            ASSERT_TRUE(blockInfo.size == mapOffset0 + mapSize0 - (mapOffset1 + mapSize1));
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapOffset1 + mapSize1 == mapOffset0 + mapSize0)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapOffset0 - mapOffset1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapOffset1 + mapSize1 - mapOffset0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                        else if (mapOffset0 + mapSize0 < mapOffset1 + mapSize1)
                        {
                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                            ASSERT_TRUE(blockInfo.size == mapOffset0 - mapOffset1);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapSize0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0 + mapSize0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0 + mapSize0);
                            ASSERT_TRUE(blockInfo.size == mapOffset1 + mapSize1 - (mapOffset0 + mapSize0));
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                        }
                    }

                    result = nn::svc::UnmapDeviceAddressSpace(
                            handle0, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                            heapAddr + mapOffset0, mapSize0, deviceAddress + mapOffset0);
                    ASSERT_RESULT_SUCCESS(result);

                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                    ASSERT_TRUE(blockInfo.size == mapSize1);
                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                    result = nn::svc::UnmapDeviceAddressSpace(
                            handle1, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                            heapAddr + mapOffset1, mapSize1, deviceAddress + mapOffset1);
                    ASSERT_RESULT_SUCCESS(result);

                    GetMemoryInfo(&blockInfo, heapAddr);
                    ASSERT_TRUE(blockInfo.baseAddress == heap.GetAddress());
                    ASSERT_TRUE(blockInfo.size == heap.GetSize());
                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                    ASSERT_TRUE(blockInfo.attribute == 0);
                }
            }
        }
    }
} // NOLINT(readability/fn_size)

TEST(MapDeviceAddressSpace, OverlapTest2)
{
    // 以下の観点は仕様変更によりテストしない
    // - overlap サポートしているのは RWだけ

    // 代わりに devicePermission が違ってもoverlap が出来ることの確認をする
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t spaceSize = 0x80000000;
    nn::svc::Handle handle0;
    nn::svc::Handle handle1;
    size_t align = 0x400000;
    uint64_t deviceAddress = 0;
    nn::svc::MemoryInfo blockInfo;

    result = nn::svc::CreateDeviceAddressSpace(&handle0, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser0(handle0);

    result = nn::svc::CreateDeviceAddressSpace(&handle1, SpaceAddr, spaceSize);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser1(handle1);

    uintptr_t heapAddr;
    size_t heapSize = align * 2;

    TestHeap heap(heapSize);
    ASSERT_TRUE(heapSize == heap.GetSize());
    heapAddr = (heap.GetAddress() + (align - 1)) & ~(align - 1);

    nn::svc::PhysicalMemoryInfo physInfo = { 0 };
    result = nn::svc::QueryPhysicalAddress(&physInfo, heapAddr);
    ASSERT_RESULT_SUCCESS(result);
    // 4MBアラインされるはず
    uintptr_t physAddr = physInfo.physicalAddress + (heapAddr - physInfo.virtualAddress);
    ASSERT_TRUE((physAddr & (align - 1)) == 0);

    heapSize -= heapAddr - heap.GetAddress();

    size_t areaSize = 0x5000;
    size_t outSize;

    for (size_t i = 0; i < areaSize / 0x1000; i++)
    {
        size_t mapOffset0 = 0x1000 * i;
        for (size_t j = 1; j <= (areaSize - mapOffset0) / 0x1000; j++)
        {
            size_t mapSize0 = 0x1000 * j;
            for (size_t k = 0; k < areaSize / 0x1000; k++)
            {
                size_t mapOffset1 = 0x1000 * k;
                for (size_t l = 1; l <= (areaSize - mapOffset1) / 0x1000; l++)
                {
                    size_t mapSize1 = 0x1000 * l;

                    if (mapOffset0 + mapSize0 <= mapOffset1 || mapOffset1 + mapSize1 <= mapOffset0)
                    {
                        continue;
                    }

                    nn::svc::MemoryPermission permissionArray[] = {
                        nn::svc::MemoryPermission_Write,
                        nn::svc::MemoryPermission_Read,
                        nn::svc::MemoryPermission_ReadWrite,
                    };

                    for (int m = 0; m < 3; m++)
                    {
                        for (int n = 0; n < 3; n++)
                        {
                            nn::svc::MemoryPermission permission0 = permissionArray[m];
                            nn::svc::MemoryPermission permission1 = permissionArray[n];

                            // テスト済み
                            if (permission0 == nn::svc::MemoryPermission_ReadWrite &&
                                    permission1 == nn::svc::MemoryPermission_ReadWrite)
                            {
                                continue;
                            }

                            result = nn::svc::MapDeviceAddressSpace( &outSize,
                                    handle0, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                                    heapAddr + mapOffset0, mapSize0, deviceAddress + mapOffset0,
                                    permission0);
                            ASSERT_RESULT_SUCCESS(result);

                            GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                            ASSERT_TRUE(blockInfo.size == mapSize0);
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                            result = nn::svc::MapDeviceAddressSpace( &outSize,
                                    handle1, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                                    heapAddr + mapOffset1, mapSize1, deviceAddress + mapOffset1,
                                    permission1);
                            ASSERT_RESULT_SUCCESS(result);

                            if (mapOffset0 == mapOffset1)
                            {
                                if (mapSize0 == mapSize1)
                                {
                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.size == mapSize0);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                                }
                                else if (mapSize0 < mapSize1)
                                {
                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.size == mapSize0);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset0 + mapSize0);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0 + mapSize0);
                                    ASSERT_TRUE(blockInfo.size == mapOffset1 + mapSize1 - (mapOffset0 + mapSize0));
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                                }
                                else
                                {
                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.size == mapSize1);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset1 + mapSize1);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1 + mapSize1);
                                    ASSERT_TRUE(blockInfo.size == mapOffset0 + mapSize0 - (mapOffset1 + mapSize1));
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                                }
                            }
                            else if (mapOffset0 < mapOffset1)
                            {
                                if (mapOffset0 + mapSize0 < mapOffset1)
                                {
                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.size == mapSize0);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.size == mapSize1);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                                }
                                else if (mapOffset0 + mapSize0 == mapOffset1)
                                {
                                    // 連結される
                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.size == mapSize0 + mapSize1);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                                }
                                else if (mapOffset0 + mapSize0 < mapOffset1 + mapSize1)
                                {
                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.size == mapOffset1 - mapOffset0);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.size == mapOffset0 + mapSize0 - mapOffset1);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset0 + mapSize0);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0 + mapSize0);
                                    ASSERT_TRUE(blockInfo.size == mapOffset1 + mapSize1 - (mapOffset0 + mapSize0));
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                                }
                                else if (mapOffset0 + mapSize0 == mapOffset1 + mapSize1)
                                {
                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.size == mapOffset1 - mapOffset0);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.size == mapOffset0 + mapSize0 - mapOffset1);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                                }
                                else if (mapOffset1 + mapSize1 < mapOffset0 + mapSize0)
                                {
                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.size == mapOffset1 - mapOffset0);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.size == mapSize1);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset1 + mapSize1);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1 + mapSize1);
                                    ASSERT_TRUE(blockInfo.size == mapOffset0 + mapSize0 - (mapOffset1 + mapSize1));
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                                }
                            }
                            else if (mapOffset1 < mapOffset0)
                            {
                                if (mapOffset1 + mapSize1 < mapOffset0)
                                {
                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.size == mapSize1);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.size == mapSize0);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                                }
                                else if (mapOffset1 + mapSize1 == mapOffset0)
                                {
                                    // 連結される
                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.size == mapSize1 + mapSize0);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                                }
                                else if (mapOffset1 + mapSize1 < mapOffset0 + mapSize0)
                                {
                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.size == mapOffset0 - mapOffset1);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.size == mapOffset1 + mapSize1 - mapOffset0);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset1 + mapSize1);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1 + mapSize1);
                                    ASSERT_TRUE(blockInfo.size == mapOffset0 + mapSize0 - (mapOffset1 + mapSize1));
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                                }
                                else if (mapOffset1 + mapSize1 == mapOffset0 + mapSize0)
                                {
                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.size == mapOffset0 - mapOffset1);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.size == mapOffset1 + mapSize1 - mapOffset0);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                                }
                                else if (mapOffset0 + mapSize0 < mapOffset1 + mapSize1)
                                {
                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset1);
                                    ASSERT_TRUE(blockInfo.size == mapOffset0 - mapOffset1);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0);
                                    ASSERT_TRUE(blockInfo.size == mapSize0);
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);

                                    GetMemoryInfo(&blockInfo, heapAddr + mapOffset0 + mapSize0);
                                    ASSERT_TRUE(blockInfo.baseAddress == heapAddr + mapOffset0 + mapSize0);
                                    ASSERT_TRUE(blockInfo.size == mapOffset1 + mapSize1 - (mapOffset0 + mapSize0));
                                    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                                    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                                    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_DeviceShared);
                                }
                            }

                            result = nn::svc::UnmapDeviceAddressSpace(
                                    handle1, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                                    heapAddr + mapOffset1, mapSize1, deviceAddress + mapOffset1);
                            ASSERT_RESULT_SUCCESS(result);

                            result = nn::svc::UnmapDeviceAddressSpace(
                                    handle0, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                                    heapAddr + mapOffset0, mapSize0, deviceAddress + mapOffset0);
                            ASSERT_RESULT_SUCCESS(result);

                            GetMemoryInfo(&blockInfo, heapAddr);
                            ASSERT_TRUE(blockInfo.baseAddress == heap.GetAddress());
                            ASSERT_TRUE(blockInfo.size == heap.GetSize());
                            ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
                            ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
                            ASSERT_TRUE(blockInfo.attribute == 0);
                        }
                    }
                }
            }
        }
    }
} // NOLINT(readability/fn_size)

