﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/ipc/svc_SessionMessage.h>

extern "C" void nnMain();

namespace {

#ifdef INVALID_POINTER_TEST
const char ConstVar[sizeof(nn::svc::Handle)] = { 0 };
#endif
const size_t StackSize = 0x1000;
const char* NameLengthTestData[] = {
    "1",
    "12",
    "123",
    "1234",
    "12345",
    "123456",
    "1234567",
    "12345678",
    "123456789",
    "1234567890",
    "12345678901",
};

char g_Stack[2][StackSize] __attribute__((aligned(0x1000)));

void NormalClientThread(uintptr_t arg)
{
    AutoThreadExit autoExit;

    nn::Result result;
    char* name = reinterpret_cast<char*>(arg);

    nn::svc::Handle clientSession;
    result = nn::svc::ConnectToNamedPort(&clientSession, name);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sessionCloser(clientSession);

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::SendSyncRequest(clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

void NormalServerThread(uintptr_t arg)
{
    AutoThreadExit autoExit;

    nn::Result result;
    nn::svc::Handle* serverPort = reinterpret_cast<nn::svc::Handle*>(arg);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, serverPort, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, *serverPort);
    AutoHandleClose sessionCloser(serverSession);

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::ReplyAndReceive(
                &index, &serverSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ReplyAndReceive(
                &index, &serverSession, 0, serverSession, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
}

void CheckIpc(nn::svc::Handle serverPort, const char* name)
{
    nn::Result result;

    uintptr_t pc[2];
    uintptr_t sp[2];
    uintptr_t param[2];
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = 0;

    pc[0] = reinterpret_cast<uintptr_t>(NormalServerThread);
    pc[1] = reinterpret_cast<uintptr_t>(NormalClientThread);

    sp[0] = reinterpret_cast<uintptr_t>(g_Stack[0]) + StackSize;
    sp[1] = reinterpret_cast<uintptr_t>(g_Stack[1]) + StackSize;

    param[0] = reinterpret_cast<uintptr_t>(&serverPort);
    param[1] = reinterpret_cast<uintptr_t>(name);

    TestThread serverThread(pc[0], param[0], sp[0], priority, idealCore);
    TestThread clientThread(pc[1], param[1], sp[1], priority, idealCore);

    serverThread.Start();
    clientThread.Start();

    serverThread.Wait();
    clientThread.Wait();
}

} // namespace

#ifdef INVALID_POINTER_TEST
TEST(ManageNamedPort, pOutServerTest)
{
    TestNamedPortLeak leakTest;
    nn::Result result;
    nn::svc::Handle* pNamedPort;

#ifdef INVALID_POINTER_TEST
    // TEST 113-2
    // NULL アドレスを受け付けない
    pNamedPort = NULL;
    result = nn::svc::ManageNamedPort(pNamedPort, "FailPort", 1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 113-3
    // MemoryPermission_None の領域を指定すると失敗する
    pNamedPort = reinterpret_cast<nn::svc::Handle*>(g_FreeAreaBegin);
    result = nn::svc::ManageNamedPort(pNamedPort, "FailPort", 1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 113-4
    // MemoryPermission_Read の領域を指定すると失敗する
    uintptr_t addr = reinterpret_cast<uintptr_t>(ConstVar);
    pNamedPort = reinterpret_cast<nn::svc::Handle*>(addr);
    result = nn::svc::ManageNamedPort(pNamedPort, "FailPort", 1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 113-5
    // MemoryPermission_ReadExecute の領域を指定すると失敗する
    pNamedPort = reinterpret_cast<nn::svc::Handle*>(nnMain);
    result = nn::svc::ManageNamedPort(pNamedPort, "FailPort", 1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

}
#endif // INVALID_POINTER_TEST

TEST(ManageNamedPort, NameTest)
{
    TestNamedPortLeak leakTest;
    nn::Result result;
    nn::svc::Handle namedPort;

    // TEST 113-8
    // NUL 文字終端している文字列を指定することが出来る
    for (uint8_t i = 0; i < sizeof(NameLengthTestData) / sizeof(char*); i++)
    {
        result = nn::svc::ManageNamedPort(&namedPort, NameLengthTestData[i], 1);
        ASSERT_RESULT_SUCCESS(result);

        CheckIpc(namedPort, NameLengthTestData[i]);

        nn::svc::Handle tmpHandle;
        result = nn::svc::ManageNamedPort(&tmpHandle, NameLengthTestData[i], 0);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(tmpHandle == nn::svc::INVALID_HANDLE_VALUE);

        result = nn::svc::CloseHandle(namedPort);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 113-10
    // 12 文字以上は受け付けない
    {
        const char* name = "123456789012";
        result = nn::svc::ManageNamedPort(&namedPort, name, 1);
        ASSERT_RESULT_FAILURE(result);
    }

    char *p;

    // TEST 113-11
    // NULL を受け付けない
    p = NULL;
    result = nn::svc::ManageNamedPort(&namedPort, p, 1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());

    // TEST 113-12
    // MemoryPermission_None の領域を指定すると失敗する
    p = reinterpret_cast<char*>(g_FreeAreaBegin);
    result = nn::svc::ManageNamedPort(&namedPort, p, 1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
}

TEST(ManageNamedPort, MaxSessoinTest)
{
    TestNamedPortLeak leakTest;
    nn::Result result;
    nn::svc::Handle namedPort;
    int32_t successData[] = {
        1,
        nn::svc::ArgumentHandleCountMax, nn::svc::ArgumentHandleCountMax + 1,
        0xFFF, 0x1000, 0x1001,
        0xFFFF, 0x10000, 0x10001,
        INT32_MAX,
    };
    char name[2] = {0};
    NN_STATIC_ASSERT(sizeof(successData) / sizeof(int32_t) < 'z' - 'a');

    // TEST 113-15
    // 1 以上の値が与えられたときに、ポートを作成する
    for (uint8_t i = 0; i < sizeof(successData) / sizeof(int32_t); i++)
    {
        name[0] = 'a' + i;
        result = nn::svc::ManageNamedPort(&namedPort, name, successData[i]);
        ASSERT_RESULT_SUCCESS(result);

        CheckIpc(namedPort, name);

        nn::svc::Handle tmpHandle;
        result = nn::svc::ManageNamedPort(&tmpHandle, name, 0);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(tmpHandle == nn::svc::INVALID_HANDLE_VALUE);

        result = nn::svc::CloseHandle(namedPort);
        ASSERT_RESULT_SUCCESS(result);
    }

    // TEST 113-16
    // 負の値は受け付けない
    {
        const char* testName = "MaxSession";
        result = nn::svc::ManageNamedPort(&namedPort, testName, -1);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());

        result = nn::svc::ManageNamedPort(&namedPort, testName, 0xFFFF0000);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());

        result = nn::svc::ManageNamedPort(&namedPort, testName, 0x80000000);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfRange());
    }
}

