﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Result.h>
#include <cstring>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/util/util_BitPack.h>
#include <nn/nn_BitTypes.h>
#include <nn/os.h>
#include <nn/svc/svc_Tcb.h>

namespace {
char g_Buffer[DefaultStackSize] __attribute__((aligned(0x1000)));
nn::Bit32 g_UserBuffer0[DefaultUserBufferSize / sizeof(nn::Bit32)] __attribute__((aligned(0x1000)));
nn::Bit32 g_UserBuffer1[DefaultUserBufferSize / sizeof(nn::Bit32)] __attribute__((aligned(0x1000)));
char g_RecvBuffer0[13][0x100];
char g_RecvBuffer1[13][0x100];
char g_XferBuffer[0x4000] __attribute__((aligned(0x1000)));
nn::svc::Handle g_Thread0;
nn::svc::Handle g_Thread1;
nn::svc::Handle g_Thread;
nn::Bit64           g_ThreadId;
extern "C" void nnMain();
const int TestNum = 1;
struct TestData
{
    const char* name;
};
const char* TestPortName = "test";
const char* TestMessage[] = {
    "TEST0",
    "xTEST1",
    "xxTEST2",
    "xxxTEST3",
    "xxxxTEST4",
    "xxxxxTEST5",
    "xxxxxxTEST6",
    "xxxxxxxTEST7",
    "xxxxxxxxTEST8",
    "xxxxxxxxxTEST9",
    "xxxxxxxxxxTEST10",
    "xxxxxxxxxxxTEST11",
    "xxxxxxxxxxxxTEST12"
};

// ポインタ転送なし
void NamedPort0(const int x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    if (userBuffer)
    {
        pMsgBuffer = (x == 0? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == 0)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;
    int32_t index;
    if (x == 0)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 0, 0, 0));
        // 転送要求がない要求を送信できる
        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                // TEST 30-42
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-32
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            // TEST 28-23
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        nn::svc::Handle handles[1] = { handle };
        // 転送要求がない要求を受信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-1
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-44
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // TEST 67-1
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-2, 68-45
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-2
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-24, 29-33, 30-43
            // 結果を受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// ポインタ転送は メッセージバッファを使用
void NamedPort1(const int x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    if (userBuffer)
    {
        pMsgBuffer = (x == 0? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == 0)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;
    int32_t index;
    if (x == 0)
    {
        {
            int offset;
            // pointer = 2, 受信指定数 = 1
            // ポインタ転送にメッセージバッファを利用することが出来る
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0003, 0, 2, 0, 0, 0, 0, 1));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[1], ::std::strlen(TestMessage[1]), 1));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));
            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    // TEST 30-44
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    // TEST 29-34
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                // TEST 28-25
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 1));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-46
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-3
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // TEST 67-3
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0003);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 2);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer0(offset + 2, ipcMsg);
            nn::svc::ipc::MessageBuffer::PointerData pointer1(offset, ipcMsg);

            ASSERT_TRUE(pointer0.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer0.GetPointerSize() == ::std::strlen(TestMessage[0]));
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(pMsgBuffer) < pointer0.GetPointerAddress());
            ASSERT_TRUE(pointer0.GetPointerAddress() < reinterpret_cast<uintptr_t>(pMsgBuffer) + msgSize);
            ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer0.GetPointerAddress()), TestMessage[0], pointer0.GetPointerSize()) == 0);

            ASSERT_TRUE(pointer1.GetPointerIndex() == 1);
            ASSERT_TRUE(pointer1.GetPointerSize() == ::std::strlen(TestMessage[1]));
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(pMsgBuffer) < pointer1.GetPointerAddress());
            ASSERT_TRUE(pointer1.GetPointerAddress() < reinterpret_cast<uintptr_t>(pMsgBuffer) + msgSize);
            ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer1.GetPointerAddress()), TestMessage[1], pointer1.GetPointerSize()) == 0);
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0004, 0, 2, 0, 0, 0, 0, 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[1], ::std::strlen(TestMessage[1]), 1));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-4, 68-47
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-4
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-26, 29-35, 30-45
            // 結果を受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0004);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 2);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送でデータを受信できている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer0(offset, ipcMsg);
            nn::svc::ipc::MessageBuffer::PointerData pointer1(offset + 2, ipcMsg);

            ASSERT_TRUE(pointer0.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer0.GetPointerSize() == ::std::strlen(TestMessage[0]));
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(pMsgBuffer) < pointer0.GetPointerAddress());
            ASSERT_TRUE(pointer0.GetPointerAddress() < reinterpret_cast<uintptr_t>(pMsgBuffer) + msgSize);
            ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer0.GetPointerAddress()), TestMessage[0], pointer0.GetPointerSize()) == 0);

            ASSERT_TRUE(pointer1.GetPointerIndex() == 1);
            ASSERT_TRUE(pointer1.GetPointerSize() == ::std::strlen(TestMessage[1]));
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(pMsgBuffer) < pointer1.GetPointerAddress());
            ASSERT_TRUE(pointer1.GetPointerAddress() < reinterpret_cast<uintptr_t>(pMsgBuffer) + msgSize);
            ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer1.GetPointerAddress()), TestMessage[1], pointer1.GetPointerSize()) == 0);
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// ポインタ転送は 一つの受信指定リストを使用
void NamedPort2(const int x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    if (userBuffer)
    {
        pMsgBuffer = (x == 0? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == 0)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;
    int32_t index;
    if (x == 0)
    {
        {
            int offset;
            // pointer = 2, 受信指定数 = 2
            // ポインタ転送に1つの受信指定リストを使用することが出来る
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0005, 0, 2, 0, 0, 0, 0, 2));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[1], ::std::strlen(TestMessage[1]), 1));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[0], sizeof(g_RecvBuffer0[0])));
            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    // TEST 30-46
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    // TEST 29-36
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                // TEST 28-27
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 2));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[0], sizeof(g_RecvBuffer1[0])));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-48
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-5
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // TEST 67-5
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0005);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 2);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 2);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // 受信リストで指定したアドレスにデータが入っている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer0(offset + 2, ipcMsg);
            nn::svc::ipc::MessageBuffer::PointerData pointer1(offset, ipcMsg);

            ASSERT_TRUE(pointer0.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer0.GetPointerSize() == ::std::strlen(TestMessage[0]));
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer1[0]) <= pointer0.GetPointerAddress());
            ASSERT_TRUE(pointer0.GetPointerAddress() < reinterpret_cast<uintptr_t>(g_RecvBuffer1[0]) + sizeof(g_RecvBuffer1[0]));
            ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer0.GetPointerAddress()), TestMessage[0], pointer0.GetPointerSize()) == 0);

            ASSERT_TRUE(pointer1.GetPointerIndex() == 1);
            ASSERT_TRUE(pointer1.GetPointerSize() == ::std::strlen(TestMessage[1]));
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer1[0]) <= pointer1.GetPointerAddress());
            ASSERT_TRUE(pointer1.GetPointerAddress() < reinterpret_cast<uintptr_t>(g_RecvBuffer1[0]) + sizeof(g_RecvBuffer1[0]));
            ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer1.GetPointerAddress()), TestMessage[1], pointer1.GetPointerSize()) == 0);
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0006, 0, 2, 0, 0, 0, 0, 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[1], ::std::strlen(TestMessage[1]), 1));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-6, 68-49
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-6
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-28, 29-37, 30-47
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0006);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 2);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // 受信リストに指定したアドレスに結果が入っている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer0(offset, ipcMsg);
            nn::svc::ipc::MessageBuffer::PointerData pointer1(offset + 2, ipcMsg);

            ASSERT_TRUE(pointer0.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer0.GetPointerSize() == ::std::strlen(TestMessage[0]));
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer0[0]) <= pointer0.GetPointerAddress());
            ASSERT_TRUE(pointer0.GetPointerAddress() < reinterpret_cast<uintptr_t>(g_RecvBuffer0[0]) + sizeof(g_RecvBuffer0[0]));
            ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer0.GetPointerAddress()), TestMessage[0], pointer0.GetPointerSize()) == 0);

            ASSERT_TRUE(pointer1.GetPointerIndex() == 1);
            ASSERT_TRUE(pointer1.GetPointerSize() == ::std::strlen(TestMessage[1]));
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer0[0]) <= pointer1.GetPointerAddress());
            ASSERT_TRUE(pointer1.GetPointerAddress() < reinterpret_cast<uintptr_t>(g_RecvBuffer0[0]) + sizeof(g_RecvBuffer0[0]));
            ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer1.GetPointerAddress()), TestMessage[1], pointer1.GetPointerSize()) == 0);
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// ポインタ転送は 受信指定リストを使用
void NamedPort3(const int x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    if (userBuffer)
    {
        pMsgBuffer = (x == 0? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == 0)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;
    int32_t index;
    if (x == 0)
    {
        {
            int offset;
            // pointer = 13, 受信指定数 = 15
            // ポインタ転送に複数のアドレスを受信指定リストに指定して使用することが出来る
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0007, 0, 13, 0, 0, 0, 0, 15));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[12], ::std::strlen(TestMessage[12]), 12));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[11], ::std::strlen(TestMessage[11]), 11));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[10], ::std::strlen(TestMessage[10]), 10));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[9], ::std::strlen(TestMessage[9]), 9));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[8], ::std::strlen(TestMessage[8]), 8));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[7], ::std::strlen(TestMessage[7]), 7));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[6], ::std::strlen(TestMessage[6]), 6));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[5], ::std::strlen(TestMessage[5]), 5));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[4], ::std::strlen(TestMessage[4]), 4));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[3], ::std::strlen(TestMessage[3]), 3));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[2], ::std::strlen(TestMessage[2]), 2));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[1], ::std::strlen(TestMessage[1]), 1));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[0], sizeof(g_RecvBuffer0[0])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[1], sizeof(g_RecvBuffer0[1])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[2], sizeof(g_RecvBuffer0[2])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[3], sizeof(g_RecvBuffer0[3])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[4], sizeof(g_RecvBuffer0[4])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[5], sizeof(g_RecvBuffer0[5])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[6], sizeof(g_RecvBuffer0[6])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[7], sizeof(g_RecvBuffer0[7])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[8], sizeof(g_RecvBuffer0[8])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[9], sizeof(g_RecvBuffer0[9])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[10], sizeof(g_RecvBuffer0[10])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[11], sizeof(g_RecvBuffer0[11])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[12], sizeof(g_RecvBuffer0[12])));
            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    // TEST 30-48
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    // TEST 29-38
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                // TEST 28-29
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 15));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[0], sizeof(g_RecvBuffer1[0])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[1], sizeof(g_RecvBuffer1[1])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[2], sizeof(g_RecvBuffer1[2])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[3], sizeof(g_RecvBuffer1[3])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[4], sizeof(g_RecvBuffer1[4])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[5], sizeof(g_RecvBuffer1[5])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[6], sizeof(g_RecvBuffer1[6])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[7], sizeof(g_RecvBuffer1[7])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[8], sizeof(g_RecvBuffer1[8])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[9], sizeof(g_RecvBuffer1[9])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[10], sizeof(g_RecvBuffer1[10])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[11], sizeof(g_RecvBuffer1[11])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[12], sizeof(g_RecvBuffer1[12])));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-50
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-7
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // TEST 67-7
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0007);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 13);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 15);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // 受信リストで指定したアドレスにデータが入っている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);

            for (int i = 0; i < 13; i++)
            {
                nn::svc::ipc::MessageBuffer::PointerData pointer(offset + (12 - i) * 2, ipcMsg);

                ASSERT_TRUE(pointer.GetPointerIndex() == i);
                ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[i]));
                ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer1[i]) <= pointer.GetPointerAddress());
                ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(g_RecvBuffer1[i]) + sizeof(g_RecvBuffer1[i]));
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[i], pointer.GetPointerSize()) == 0);
            }
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0007, 0, 13, 0, 0, 0, 0, 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[1], ::std::strlen(TestMessage[1]), 1));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[2], ::std::strlen(TestMessage[2]), 2));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[3], ::std::strlen(TestMessage[3]), 3));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[4], ::std::strlen(TestMessage[4]), 4));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[5], ::std::strlen(TestMessage[5]), 5));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[6], ::std::strlen(TestMessage[6]), 6));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[7], ::std::strlen(TestMessage[7]), 7));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[8], ::std::strlen(TestMessage[8]), 8));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[9], ::std::strlen(TestMessage[9]), 9));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[10], ::std::strlen(TestMessage[10]), 10));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[11], ::std::strlen(TestMessage[11]), 11));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[12], ::std::strlen(TestMessage[12]), 12));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // 68-8, 68-51
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-8
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-30, 29-39, 30-49
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0007);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 13);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // 受信リストに指定したアドレスに結果が入っている
            int offset;
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);

            for (int i = 0; i < 13; i++)
            {
                nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                ASSERT_TRUE(pointer.GetPointerIndex() == i);
                ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[i]));
                ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer0[i]) <= pointer.GetPointerAddress());
                ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(g_RecvBuffer0[i]) + sizeof(g_RecvBuffer0[i]));
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[i], pointer.GetPointerSize()) == 0);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// Send
void NamedPort4(const int x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    if (userBuffer)
    {
        pMsgBuffer = (x == 0? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == 0)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;
    int32_t index;
    // アライン
    if (x == 0)
    {
        ::std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), sizeof(g_XferBuffer));

        // send = 1
        // 送信要求を出すことが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 1, 0, 0, 0, 0));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer, sizeof(g_XferBuffer)));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                // TEST 30-50
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-40
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            // TEST 28-31
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-52
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-9
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // TEST 67-9
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 1);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // マップした領域のデータが読み取れる
            nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
            ASSERT_TRUE(mapData.GetDataSize() == sizeof(g_XferBuffer));
            ASSERT_TRUE(::std::memcmp(g_XferBuffer, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-10, 68-53
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-10
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-32, 29-41, 30-51
            // ヘッダー情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);
        }
    }
    // 非アライン
    if (x == 0)
    {
        ::std::memcpy(g_XferBuffer + 0x100, reinterpret_cast<void*>(nnMain), 0x1100);

        // send = 1
        // 送信要求を出すことが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0003, 0, 0, 1, 0, 0, 0, 0));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x100, 0x1100));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                // TEST 30-52
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-42
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            // TEST 28-33
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-54
            // SendSyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-11
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            // TEST 67-11
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0003);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 1);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // マップした領域のデータが読み取れる
            nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
            ASSERT_TRUE(mapData.GetDataSize() == 0x1100);
            ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x100, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0004, 0, 0, 0, 0, 0, 0, 0));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-10, 68-53
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-10
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-32, 29-41, 30-51
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0004);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);
        }
    }
    // 非アライン(小)
    if (x == 0)
    {
        ::std::memcpy(g_XferBuffer + 0x100, reinterpret_cast<void*>(nnMain), 0x100);

        // send = 1
        // 送信要求を出すことが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0003, 0, 0, 1, 0, 0, 0, 0));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x100, 0x100));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                // TEST 30-53
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-43
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            // TEST 28-34
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // 68-55
            // SendSyncRequestWithUserBuffer から要求を受信できる
            // 68-12
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            // TEST 67-12
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0003);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 1);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // マップした領域のデータが読み取れる
            nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
            ASSERT_TRUE(mapData.GetDataSize() == 0x100);
            ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x100, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0004, 0, 0, 0, 0, 0, 0, 0));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // 相手に reply を返すことが出来る
            // TEST 68-10, 68-53
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-10
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-32, 29-41, 30-51
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0004);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// Recv
void NamedPort5(const int x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    if (userBuffer)
    {
        pMsgBuffer = (x == 0? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == 0)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;
    int32_t index;
    // アライン
    if (x == 0)
    {
        ::std::memset(g_XferBuffer, 0, sizeof(g_XferBuffer));

        // recv = 1
        // 受信要求を出すことが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 1, 0, 0, 0));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer, sizeof(g_XferBuffer)));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                // TEST 30-54
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-44
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            // TEST 28-35
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-56
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-13
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // TEST 67-13
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 1);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // マップ領域にデータを書き込む
            nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
            ASSERT_TRUE(mapData.GetDataSize() == sizeof(g_XferBuffer));
            ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<void*>(nnMain), mapData.GetDataSize());
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-14, 68-57
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-14
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-36, 29-45, 30-55
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // データが受信できている
            ASSERT_TRUE(::std::memcmp(g_XferBuffer, reinterpret_cast<void*>(nnMain), sizeof(g_XferBuffer)) == 0);
        }
    }
    // 非アライン
    if (x == 0)
    {
        ::std::memset(g_XferBuffer, 0, sizeof(g_XferBuffer));

        // recv = 1
        // 受信要求を出すことが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 1, 0, 0, 0));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x100, 0x1100));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                // TEST 30-56
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-46
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            // TEST 28-37
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-56
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-13
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // TEST 67-13
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 1);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // マップ領域にデータを書き込む
            nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
            ASSERT_TRUE(mapData.GetDataSize() == 0x1100);
            ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<void*>(nnMain), mapData.GetDataSize());
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-15, 68-58
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-15
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-38, 29-47, 30-57
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // データが受信できている
            ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x100, reinterpret_cast<void*>(nnMain), 0x1100) == 0);
        }
    }
    // 非アライン(小)
    if (x == 0)
    {
        ::std::memset(g_XferBuffer, 0, sizeof(g_XferBuffer));

        // recv = 1
        // 受信要求を出すことが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 1, 0, 0, 0));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x100, 0x100));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                // TEST 30-58
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-48
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            // TEST 28-39
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-56
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-13
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // TEST 67-13
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 1);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // マップ領域にデータを書き込む
            nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
            ASSERT_TRUE(mapData.GetDataSize() == 0x100);
            ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<void*>(nnMain), mapData.GetDataSize());
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-16, 68-59
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-16
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-40, 29-49, 30-59
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // データが受信できている
            ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x100, reinterpret_cast<void*>(nnMain), 0x100) == 0);
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// Exchange
void NamedPort6(const int x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    if (userBuffer)
    {
        pMsgBuffer = (x == 0? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == 0)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;
    int32_t index;
    // アライン
    if (x == 0)
    {
        ::std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), sizeof(g_XferBuffer));

        // exchange = 1
        // 送受信要求を出すことが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 1, 0, 0));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer, sizeof(g_XferBuffer)));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                // TEST 30-60
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-50
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            // TEST 28-41
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-60
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-17
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // TEST 67-17
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 1);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);


            // マップ領域のデータが受信できている
            nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
            ASSERT_TRUE(mapData.GetDataSize() == sizeof(g_XferBuffer));
            ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(nnMain), reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);

            // マップ領域にデータを書き込む
            ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x200, mapData.GetDataSize());
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-18, 68-61
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-18
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-42, 29-51, 30-61
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // データが受信できている
            ASSERT_TRUE(::std::memcmp(g_XferBuffer, reinterpret_cast<char*>(nnMain) + 0x200, sizeof(g_XferBuffer)) == 0);
        }
    }
    // 非アライン
    if (x == 0)
    {
        ::std::memcpy(g_XferBuffer + 0x100, reinterpret_cast<void*>(nnMain), 0x1100);

        // exchange = 1
        // 送受信要求を出すことが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 1, 0, 0));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x100, 0x1100));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                // TEST 30-52
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-52
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            // TEST 28-43
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-62
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-19
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // TEST 67-19
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 1);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // マップ領域のデータが受信できている
            nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
            ASSERT_TRUE(mapData.GetDataSize() == 0x1100);
            ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x100, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);

            // マップ領域にデータを書き込む
            ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x200, mapData.GetDataSize());
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-20, 68-63
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-20
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-44, 29-53, 30-63
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // データが受信できている
            ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x100, reinterpret_cast<char*>(nnMain) + 0x200, 0x1100) == 0);
        }
    }
    // 非アライン(小)
    if (x == 0)
    {
        ::std::memcpy(g_XferBuffer + 0x100, reinterpret_cast<void*>(nnMain), 0x100);

        // exchange = 1
        // 送受信要求を出すことが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 1, 0, 0));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x100, 0x100));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                // TEST 30-64
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-54
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            // TEST 28-45
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-64
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-21
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // TEST 67-21
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 1);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // マップ領域のデータが受信できている
            nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
            ASSERT_TRUE(mapData.GetDataSize() == 0x100);
            ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x100, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);

            // マップ領域にデータを書き込む
            ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x200, mapData.GetDataSize());
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-22, 68-65
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-22
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-46, 29-55, 30-65
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // データが受信できている
            ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x100, reinterpret_cast<char*>(nnMain) + 0x200, 0x100) == 0);
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// 任意データ
void NamedPort7(const int x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    if (userBuffer)
    {
        pMsgBuffer = (x == 0? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == 0)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;
    int32_t index;
    for (int rawNum = 0; rawNum < 63; rawNum++)
    {
        if (x == 0)
        {
            // 任意データ数 = rawNum
            // 任意データを転送できる
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 0, rawNum, 0));
            offset = ipcMsg.SetRawArray(offset, reinterpret_cast<void*>(nnMain), rawNum * sizeof(nn::Bit32));
            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    // TEST 30-66
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    // TEST 29-56
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                // TEST 28-47
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == 1)
        {
            // 受信バッファの準備
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
            }
            int32_t index;
            nn::svc::Handle handles[1] = { handle };
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                // TEST 68-66
                // SendAsyncRequestWithUserBuffer から要求を受信できる
                // TEST 68-23
                // SendSyncRequestWithUserBuffer から要求を受信できる
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            else
            {
                // TEST 67-23
                // SendSyncRequest から要求を受信できる
                result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);

            {
                // IPC ヘッダが受信できている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == rawNum);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // 任意データが受信できている
                int offset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);
                ASSERT_TRUE(::std::memcmp(pMsgBuffer + offset, reinterpret_cast<void*>(nnMain), rawNum * sizeof(nn::Bit32)) == 0);
            }
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, rawNum, 0));
                offset = ipcMsg.SetRawArray(offset, reinterpret_cast<void*>(nnMain), rawNum * sizeof(nn::Bit32));
            }
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                // TEST 68-24, 68-67
                // 相手に reply を返すことが出来る
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                // TEST 67-24
                // 相手に reply を返すことが出来る
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
        if (x == 0)
        {
            {
                // TEST 28-48, 29-57, 30-67
                // ヘッダ情報が更新されている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == rawNum);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // 任意データが受信できている
                int offset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);
                ASSERT_TRUE(::std::memcmp(pMsgBuffer + offset, reinterpret_cast<void*>(nnMain), rawNum * sizeof(nn::Bit32)) == 0);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// 特殊データ(PID)
void NamedPort8(const int x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    if (userBuffer)
    {
        pMsgBuffer = (x == 0? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == 0)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;
    int32_t index;
    if (x == 0)
    {
        // 特殊データ転送数 = 1
        // Process ID を転送できる
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 1, 0, 0, 0, 0, 0, 0));
        offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(true, 0, 0));
        offset = ipcMsg.SetProcessId(offset, 0);

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                // TEST 30-68
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-58
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            // TEST 28-49
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-68
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-25
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // TEST 67-25
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);

            // TEST 67-26, 68-26, 68-69
            // Process ID が受信できている
            ASSERT_TRUE(ipcSpecial.GetProcessIdFlag());
            {
                nn::Bit64 id;
                result = nn::svc::GetProcessId(&id, static_cast<nn::svc::Handle>(0xFFFF8001));
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(pMsgBuffer[nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial)] == id);
            }
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 1, 0, 0, 0, 0, 0, 0));
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(true, 0, 0));
            offset = ipcMsg.SetProcessId(offset, 0);
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-27, 69-70
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-27
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-50, 29-59, 30-69
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);

            // TEST 28-51, 29-60, 30-70
            // Process ID が受信できている
            ASSERT_TRUE(ipcSpecial.GetProcessIdFlag());
            {
                nn::Bit64 id;
                result = nn::svc::GetProcessId(&id, static_cast<nn::svc::Handle>(0xFFFF8001));
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(pMsgBuffer[nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial)] == id);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)
// 特殊データ(Copy Handle)
void NamedPort9(const int x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    if (userBuffer)
    {
        pMsgBuffer = (x == 0? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == 0)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;
    int32_t index;
    if (x == 0)
    {
        // 特殊データ転送数 = 1
        // Handle をコピーすることが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 1, 0, 0, 0, 0, 0, 0));
        offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 3, 0));
        offset = ipcMsg.SetHandle(offset, g_Thread0);
        offset = ipcMsg.SetHandle(offset, static_cast<nn::svc::Handle>(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD));
        offset = ipcMsg.SetHandle(offset, static_cast<nn::svc::Handle>(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                // TEST 30-71
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-61
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            // TEST 28-52
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-71
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-28
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // TEST 67-28
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);

            // TEST 67-29, 68-29, 68-72
            // ハンドルがコピーされている
            ASSERT_TRUE(!ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 3);
            {
                int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
                nn::Bit64 tid;
                result = nn::svc::GetThreadId(&tid, g_Thread0);
                ASSERT_RESULT_SUCCESS(result);

                nn::Bit64 pid;
                result = nn::svc::GetProcessId(&pid, static_cast<nn::svc::Handle>(0xFFFF8001));
                ASSERT_RESULT_SUCCESS(result);

                nn::Bit64 id;
                result = nn::svc::GetThreadId(&id, ipcMsg.GetHandle(offset));
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(id == tid);
                result = nn::svc::CloseHandle(ipcMsg.GetHandle(offset));
                ASSERT_RESULT_SUCCESS(result);

                result = nn::svc::GetThreadId(&id, ipcMsg.GetHandle(offset + 1));
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(id == tid);
                result = nn::svc::CloseHandle(ipcMsg.GetHandle(offset + 1));
                ASSERT_RESULT_SUCCESS(result);

                result = nn::svc::GetProcessId(&id, ipcMsg.GetHandle(offset + 2));
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(id == pid);
                result = nn::svc::CloseHandle(ipcMsg.GetHandle(offset + 2));
                ASSERT_RESULT_SUCCESS(result);
            }
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 1, 0, 0, 0, 0, 0, 0));
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 3, 0));
            offset = ipcMsg.SetHandle(offset, g_Thread1);
            offset = ipcMsg.SetHandle(offset, static_cast<nn::svc::Handle>(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD));
            offset = ipcMsg.SetHandle(offset, static_cast<nn::svc::Handle>(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-30, 68-73
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-30
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-53, 29-62, 30-72
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
            ASSERT_TRUE(!ipcSpecial.GetProcessIdFlag());

            // TEST 28-54, 29-63, 30-73
            // ハンドルがコピーされている
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 3);
            {
                int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
                nn::Bit64 tid;
                result = nn::svc::GetThreadId(&tid, g_Thread1);
                ASSERT_RESULT_SUCCESS(result);

                nn::Bit64 pid;
                result = nn::svc::GetProcessId(&pid, static_cast<nn::svc::Handle>(0xFFFF8001));
                ASSERT_RESULT_SUCCESS(result);

                nn::Bit64 id;
                result = nn::svc::GetThreadId(&id, ipcMsg.GetHandle(offset));
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(id == tid);
                result = nn::svc::CloseHandle(ipcMsg.GetHandle(offset));
                ASSERT_RESULT_SUCCESS(result);

                result = nn::svc::GetThreadId(&id, ipcMsg.GetHandle(offset + 1));
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(id == tid);
                result = nn::svc::CloseHandle(ipcMsg.GetHandle(offset + 1));
                ASSERT_RESULT_SUCCESS(result);

                result = nn::svc::GetProcessId(&id, ipcMsg.GetHandle(offset + 2));
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(id == pid);
                result = nn::svc::CloseHandle(ipcMsg.GetHandle(offset + 2));
                ASSERT_RESULT_SUCCESS(result);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)
// 特殊データ(Move Handle)
void NamedPort10(const int x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    if (userBuffer)
    {
        pMsgBuffer = (x == 0? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == 0)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;
    int32_t index;
    if (x == 0)
    {
        // 特殊データ転送数 = 1
        // Handle を移動することが出来る
        // 廃止
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 1, 0, 0, 0, 0, 0, 0));
        offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 1, 0));
        offset = ipcMsg.SetHandle(offset, g_Thread);
        nn::svc::Handle tmpHandle = g_Thread;

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                // TEST 30-74
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-64
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            // TEST 28-55
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);

        // For Copy
        result = nn::svc::CloseHandle(tmpHandle);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-74
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-31
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // TEST 67-31
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);

            // TEST 67-32, 68-32, 68-75
            // ハンドルが移動してきた
            ASSERT_TRUE(!ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 1);
            ASSERT_TRUE(ipcSpecial.GetMoveHandleNum() == 0);
            {
                int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
                nn::Bit64 tid;
                result = nn::svc::GetThreadId(&tid, g_Thread);
                ASSERT_RESULT_SUCCESS(result);

                nn::Bit64 id;
                result = nn::svc::GetThreadId(&id, ipcMsg.GetHandle(offset));
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(id == g_ThreadId);
                g_Thread = ipcMsg.GetHandle(offset);
            }
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 1, 0, 0, 0, 0, 0, 0));
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 0, 1));
            offset = ipcMsg.SetHandle(offset, g_Thread);
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-33, 68-76
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-33
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-56, 29-65, 30-75
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);

            // TEST 28-57, 29-66, 30-76
            // ハンドルが移動してきた
            ASSERT_TRUE(!ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 0);
            ASSERT_TRUE(ipcSpecial.GetMoveHandleNum() == 1);
            {
                int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
                nn::Bit64 tid;
                result = nn::svc::GetThreadId(&tid, g_Thread);
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

                nn::Bit64 id;
                result = nn::svc::GetThreadId(&id, ipcMsg.GetHandle(offset));
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(id == g_ThreadId);
                g_Thread = ipcMsg.GetHandle(offset);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// マップ転送の組み合わせ
void NamedPort11(const int x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    if (userBuffer)
    {
        pMsgBuffer = (x == 0? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == 0)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;
    int32_t index;
    // アライン
    if (x == 0)
    {
        ::std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), sizeof(g_XferBuffer));

        // send = 1, recv = 1, exchange = 1
        // 一度に複数の要求を送信することが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 1, 1, 1, 0, 0));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer, 0x1000));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x1000, 0x1000));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x2000, 0x1000));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                // TEST 30-77
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-67
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            // TEST 28-58
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-77
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-34
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // TEST 67-34
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 1);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 1);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 1);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Send
            {
                nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
                ASSERT_TRUE(mapData.GetDataSize() == 0x1000);
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<char*>(nnMain), reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
            }

            // Recv
            {
                nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial) + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32), ipcMsg);
                ASSERT_TRUE(mapData.GetDataSize() == 0x1000);
                ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x100, mapData.GetDataSize());
            }

            // Exchange
            {
                nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial) +  + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32) * 2, ipcMsg);
                ASSERT_TRUE(mapData.GetDataSize() == 0x1000);
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<char*>(nnMain) + 0x2000, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
                ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x200, mapData.GetDataSize());
            }
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-35, 68-78
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-35
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-59, 29-68, 30-78
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // データが受信できている
            ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x1000, reinterpret_cast<char*>(nnMain) + 0x100, 0x1000) == 0);
            ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x2000, reinterpret_cast<char*>(nnMain) + 0x200, 0x1000) == 0);
        }
    }
    // 非アライン
    if (x == 0)
    {
        ::std::memcpy(g_XferBuffer + 0x100, reinterpret_cast<void*>(nnMain), 0x1100);
        ::std::memcpy(g_XferBuffer + 0x2300, reinterpret_cast<void*>(nnMain), 0x1100);

        // send = 1, recv = 1, exchange = 1
        // 一度に複数の要求を送信することが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 1, 1, 1, 0, 0));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x100, 0x1100));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x1200, 0x1100));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x2300, 0x1100));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                // TEST 30-79
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-69
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            // TEST 28-60
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-79
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-36
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // TEST 67-36
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 1);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 1);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 1);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Send
            {
                nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
                ASSERT_TRUE(mapData.GetDataSize() == 0x1100);
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x100, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
            }
            // Recv
            {
                nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial) + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32), ipcMsg);
                ASSERT_TRUE(mapData.GetDataSize() == 0x1100);
                ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x200, mapData.GetDataSize());
            }
            // Exchange
            {
                nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial) + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32) * 2, ipcMsg);
                ASSERT_TRUE(mapData.GetDataSize() == 0x1100);
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x2300, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
                ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x300, mapData.GetDataSize());
            }
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-37, 68-80
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-37
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-61, 29-70, 30-80
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // データが受信できている
            ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x1200, reinterpret_cast<char*>(nnMain) + 0x200, 0x1100) == 0);
            ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x2300, reinterpret_cast<char*>(nnMain) + 0x300, 0x1100) == 0);
        }
    }
    // 非アライン(小)
    if (x == 0)
    {
        ::std::memcpy(g_XferBuffer + 0x100, reinterpret_cast<void*>(nnMain), 0x600);

        // send = 1, recv = 1, exchange = 1
        // 一度に複数の要求を送信することが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 1, 1, 1, 0, 0));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x100, 0x100));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0xf00, 0x200));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x1100, 0x300));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                // TEST 30-81
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-71
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            // TEST 28-62
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == 1)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-81
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-38
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // TEST 67-38
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 1);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 1);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 1);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            {
                nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
                ASSERT_TRUE(mapData.GetDataSize() == 0x100);
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x100, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
            }
            {
                nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial) + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32), ipcMsg);
                ASSERT_TRUE(mapData.GetDataSize() == 0x200);
                ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x200, mapData.GetDataSize());
            }
            {
                nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial) + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32) * 2, ipcMsg);
                ASSERT_TRUE(mapData.GetDataSize() == 0x300);
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x1100, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
                ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x300, mapData.GetDataSize());
            }
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // TEST 68-39, 68-82
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // TEST 67-39
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == 0)
    {
        {
            // TEST 28-63, 29-72, 30-82
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // データが受信できている
            ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0xf00, reinterpret_cast<char*>(nnMain) + 0x200, 0x200) == 0);
            ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x1100, reinterpret_cast<char*>(nnMain) + 0x300, 0x300) == 0);
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void NamedPort(const int x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    NamedPort0(x, handle, userBuffer, async);
    NamedPort1(x, handle, userBuffer, async);
    NamedPort2(x, handle, userBuffer, async);
    NamedPort3(x, handle, userBuffer, async);
    NamedPort4(x, handle, userBuffer, async);
    NamedPort5(x, handle, userBuffer, async);
    NamedPort6(x, handle, userBuffer, async);
    NamedPort7(x, handle, userBuffer, async);
    NamedPort8(x, handle, userBuffer, async);
    NamedPort9(x, handle, userBuffer, async);
    NamedPort10(x, handle, userBuffer, async);
    NamedPort11(x, handle, userBuffer, async);
} // NOLINT (readability/fn_size)

// 受信指定リストがページ境界をまたがるケース
void NamedPortLargeBuffer0(const int x, nn::svc::Handle handle, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    pMsgBuffer = (x == 0? g_UserBuffer0: g_UserBuffer1);
    msgSize = (x == 0)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;
    int32_t index;

    for (int rawNum = 0; rawNum < 1024; rawNum++)
    {
        // 受信リストがページ境界をまたがる要求を送信できる
        if (x == 0)
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 13, 0, 0, 0, rawNum, 15));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[12], ::std::strlen(TestMessage[12]), 12));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[11], ::std::strlen(TestMessage[11]), 11));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[10], ::std::strlen(TestMessage[10]), 10));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[9], ::std::strlen(TestMessage[9]), 9));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[8], ::std::strlen(TestMessage[8]), 8));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[7], ::std::strlen(TestMessage[7]), 7));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[6], ::std::strlen(TestMessage[6]), 6));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[5], ::std::strlen(TestMessage[5]), 5));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[4], ::std::strlen(TestMessage[4]), 4));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[3], ::std::strlen(TestMessage[3]), 3));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[2], ::std::strlen(TestMessage[2]), 2));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[1], ::std::strlen(TestMessage[1]), 1));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));
            offset = ipcMsg.SetRawArray(offset, reinterpret_cast<void*>(nnMain), rawNum * sizeof(nn::Bit32));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[0], sizeof(g_RecvBuffer0[0])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[1], sizeof(g_RecvBuffer0[1])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[2], sizeof(g_RecvBuffer0[2])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[3], sizeof(g_RecvBuffer0[3])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[4], sizeof(g_RecvBuffer0[4])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[5], sizeof(g_RecvBuffer0[5])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[6], sizeof(g_RecvBuffer0[6])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[7], sizeof(g_RecvBuffer0[7])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[8], sizeof(g_RecvBuffer0[8])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[9], sizeof(g_RecvBuffer0[9])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[10], sizeof(g_RecvBuffer0[10])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[11], sizeof(g_RecvBuffer0[11])));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[12], sizeof(g_RecvBuffer0[12])));
            ASSERT_TRUE(offset <= static_cast<int>(msgSize / sizeof(nn::Bit32)));
            BeforeClientIpc(pMsgBuffer, msgSize);
            if (async)
            {
                // TEST 30-83
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-73
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == 1)
        {
            // 受信バッファの準備
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, rawNum, 15));
                offset = ipcMsg.SetRawArray(offset, reinterpret_cast<void*>(nnMain), rawNum * sizeof(nn::Bit32));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[0], sizeof(g_RecvBuffer1[0])));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[1], sizeof(g_RecvBuffer1[1])));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[2], sizeof(g_RecvBuffer1[2])));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[3], sizeof(g_RecvBuffer1[3])));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[4], sizeof(g_RecvBuffer1[4])));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[5], sizeof(g_RecvBuffer1[5])));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[6], sizeof(g_RecvBuffer1[6])));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[7], sizeof(g_RecvBuffer1[7])));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[8], sizeof(g_RecvBuffer1[8])));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[9], sizeof(g_RecvBuffer1[9])));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[10], sizeof(g_RecvBuffer1[10])));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[11], sizeof(g_RecvBuffer1[11])));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[12], sizeof(g_RecvBuffer1[12])));
            }

            // TEST 68-83
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-40
            // SendSyncRequestWithUserBuffer から要求を受信できる
            int32_t index;
            nn::svc::Handle handles[1] = { handle };
            BeforeServerIpc(pMsgBuffer, msgSize);
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);

            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 13);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == rawNum);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 15);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);

                for (int i = 0; i < 13; i++)
                {
                    nn::svc::ipc::MessageBuffer::PointerData pointer(offset + (12 - i) * 2, ipcMsg);

                    ASSERT_TRUE(pointer.GetPointerIndex() == i);
                    ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[i]));
                    ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer1[i]) <= pointer.GetPointerAddress());
                    ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(g_RecvBuffer1[i]) + sizeof(g_RecvBuffer1[i]));
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[i], pointer.GetPointerSize()) == 0);
                }

                offset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);
                ASSERT_TRUE(::std::memcmp(pMsgBuffer + offset, reinterpret_cast<void*>(nnMain), rawNum * sizeof(nn::Bit32)) == 0);
            }
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 13, 0, 0, 0, rawNum, 0));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[1], ::std::strlen(TestMessage[1]), 1));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[2], ::std::strlen(TestMessage[2]), 2));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[3], ::std::strlen(TestMessage[3]), 3));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[4], ::std::strlen(TestMessage[4]), 4));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[5], ::std::strlen(TestMessage[5]), 5));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[6], ::std::strlen(TestMessage[6]), 6));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[7], ::std::strlen(TestMessage[7]), 7));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[8], ::std::strlen(TestMessage[8]), 8));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[9], ::std::strlen(TestMessage[9]), 9));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[10], ::std::strlen(TestMessage[10]), 10));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[11], ::std::strlen(TestMessage[11]), 11));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[12], ::std::strlen(TestMessage[12]), 12));
                offset = ipcMsg.SetRawArray(offset, reinterpret_cast<void*>(nnMain), rawNum * sizeof(nn::Bit32));
            }

            // TEST 68-41, 68-84
            // 相手に reply を返すことが出来る
            BeforeServerIpc(pMsgBuffer, msgSize);
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
        if (x == 0)
        {
            // TEST 29-74, 30-84
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 13);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == rawNum);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // データが受信できている
            int offset;
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);

            for (int i = 0; i < 13; i++)
            {
                nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                ASSERT_TRUE(pointer.GetPointerIndex() == i);
                ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[i]));
                ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer0[i]) <= pointer.GetPointerAddress());
                ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(g_RecvBuffer0[i]) + sizeof(g_RecvBuffer0[i]));
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[i], pointer.GetPointerSize()) == 0);
            }
            offset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);
            ASSERT_TRUE(::std::memcmp(pMsgBuffer + offset, reinterpret_cast<void*>(nnMain), rawNum * sizeof(nn::Bit32)) == 0);
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// メッセージバッファがページ境界をまたがるケース
void NamedPortLargeBuffer1(const int x, nn::svc::Handle handle, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    pMsgBuffer = (x == 0? g_UserBuffer0: g_UserBuffer1);
    msgSize = (x == 0)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;
    int32_t index;

    for (int rawNum = 0; rawNum < 1024; rawNum++)
    {
        if (x == 0)
        {
            // メッセージバッファがページ境界をまたがる要求を送信できる
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 13, 0, 0, 0, rawNum, 1));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[12], ::std::strlen(TestMessage[12]), 12));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[11], ::std::strlen(TestMessage[11]), 11));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[10], ::std::strlen(TestMessage[10]), 10));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[9], ::std::strlen(TestMessage[9]), 9));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[8], ::std::strlen(TestMessage[8]), 8));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[7], ::std::strlen(TestMessage[7]), 7));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[6], ::std::strlen(TestMessage[6]), 6));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[5], ::std::strlen(TestMessage[5]), 5));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[4], ::std::strlen(TestMessage[4]), 4));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[3], ::std::strlen(TestMessage[3]), 3));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[2], ::std::strlen(TestMessage[2]), 2));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[1], ::std::strlen(TestMessage[1]), 1));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));
            offset = ipcMsg.SetRawArray(offset, g_Buffer, rawNum * sizeof(nn::Bit32));
            BeforeClientIpc(pMsgBuffer, msgSize);
            if (async)
            {
                // TEST 30-85
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                // TEST 29-75
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == 1)
        {
            // 受信バッファの準備
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, rawNum, 1));
            }

            // TEST 68-85
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // TEST 68-42
            // SendSyncRequestWithUserBuffer から要求を受信できる
            int32_t index;
            nn::svc::Handle handles[1] = { handle };
            BeforeServerIpc(pMsgBuffer, msgSize);
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);

            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 13);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == rawNum);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 1);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);

                for (int i = 0; i < 13; i++)
                {
                    nn::svc::ipc::MessageBuffer::PointerData pointer(offset + (12 - i) * 2, ipcMsg);

                    ASSERT_TRUE(pointer.GetPointerIndex() == i);
                    ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[i]));
                    ASSERT_TRUE(reinterpret_cast<uintptr_t>(pMsgBuffer) <= pointer.GetPointerAddress());
                    ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(pMsgBuffer) + msgSize);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[i], pointer.GetPointerSize()) == 0);
                }
            }
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 13, 0, 0, 0, 0, 0));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[1], ::std::strlen(TestMessage[1]), 1));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[2], ::std::strlen(TestMessage[2]), 2));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[3], ::std::strlen(TestMessage[3]), 3));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[4], ::std::strlen(TestMessage[4]), 4));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[5], ::std::strlen(TestMessage[5]), 5));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[6], ::std::strlen(TestMessage[6]), 6));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[7], ::std::strlen(TestMessage[7]), 7));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[8], ::std::strlen(TestMessage[8]), 8));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[9], ::std::strlen(TestMessage[9]), 9));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[10], ::std::strlen(TestMessage[10]), 10));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[11], ::std::strlen(TestMessage[11]), 11));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[12], ::std::strlen(TestMessage[12]), 12));
            }

            // TEST 68-43, 68-86
            // 相手に reply を返すことが出来る
            BeforeServerIpc(pMsgBuffer, msgSize);
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
        if (x == 0)
        {
            // TEST 29-76, 30-86
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 13);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // データが受信できている
            int offset;
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);

            for (int i = 0; i < 13; i++)
            {
                nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                ASSERT_TRUE(pointer.GetPointerIndex() == i);
                ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[i]));
                ASSERT_TRUE(reinterpret_cast<uintptr_t>(pMsgBuffer) <= pointer.GetPointerAddress());
                ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(pMsgBuffer) + msgSize);
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[i], pointer.GetPointerSize()) == 0);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void SessionThread(nn::svc::Handle handle)
{
    AutoThreadExit autoExit;
    for (int i = 0; i < TestNum; i++)
    {
        NamedPort(0, handle, false, false);
        NamedPort(0, handle, false, false);
        NamedPort(0, handle, true, true);
        NamedPort(0, handle, true, true);
        NamedPort(0, handle, true, false);
        NamedPort(0, handle, true, false);
    }
    NamedPortLargeBuffer0(0, handle, false);
    NamedPortLargeBuffer0(0, handle, true);
    NamedPortLargeBuffer1(0, handle, false);
    NamedPortLargeBuffer1(0, handle, true);
} // NOLINT (readability/fn_size)


// Handleの取得のためだけのスレッド
void DummyThread()
{
    AutoThreadExit autoExit;
}
}


TEST(NamedPort, Test0)
{
    nn::Result result;
    int32_t index;
    nn::svc::Handle portHandle;
    nn::svc::Handle sessionHandle;
    nn::svc::Handle sessionHandle0;
    nn::svc::Handle sessionHandle1;
    nn::svc::Handle sessionHandle2;

    g_Thread1 = static_cast<nn::svc::Handle>(nn::os::GetCurrentThread()->_handle);

    result = nn::svc::ManageNamedPort(&portHandle, TestPortName, 2);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 26-1
    // 名前付きポートに接続できる
    result = nn::svc::ConnectToNamedPort(&sessionHandle0, TestPortName);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 26-2
    // ManageNamedPort で指定したセッション数以内なら複数個接続できる
    result = nn::svc::ConnectToNamedPort(&sessionHandle1, TestPortName);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 26-3
    // ManageNamedPort で指定したセッション数を超えると接続できない
    result = nn::svc::ConnectToNamedPort(&sessionHandle2, TestPortName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxSessions());

    result = nn::svc::CloseHandle(sessionHandle0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &portHandle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::AcceptSession(&sessionHandle, portHandle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(sessionHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 26-4
    // セッション数が上限値より下回れば、接続が可能になる
    result = nn::svc::ConnectToNamedPort(&sessionHandle2, TestPortName);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &portHandle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::AcceptSession(&sessionHandle, portHandle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(sessionHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &portHandle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::AcceptSession(&sessionHandle, portHandle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(sessionHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(sessionHandle1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(sessionHandle2);
    ASSERT_RESULT_SUCCESS(result);


    {
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
        uintptr_t pc = reinterpret_cast<uintptr_t>(SessionThread);
        nn::svc::Handle serverSessionHandle;
        nn::svc::Handle clientSessionHandle;

        result = nn::svc::CreateThread(&g_Thread, reinterpret_cast<uintptr_t>(DummyThread), 0, sp, 32, 0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::StartThread(g_Thread);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::WaitSynchronization(&index, &g_Thread, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::GetThreadId(&g_ThreadId, g_Thread);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::ConnectToNamedPort(&clientSessionHandle, TestPortName);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 28-1
        // NamedPort のクライアントセッションを通してリクエストが送信できる
        // TEST 28-2
        // リプライが送信されるまで、システムコールが返らない
        // TEST 29-1
        // NamedPort のクライアントセッションを通してリクエストが送信できる
        // TEST 29-2
        // リプライが送信されるまで、システムコールが返らない
        // TEST 30-1
        // NamedPort のクライアントセッションを通してリクエストが送信できる
        // TEST 30-2
        // イベントオブジェクトを通じてリプライの通知を受け取ることが出来る
        result = nn::svc::CreateThread(&g_Thread0, pc, static_cast<nnHandle>(clientSessionHandle).value, sp, 32, 0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::StartThread(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::WaitSynchronization(&index, &portHandle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::AcceptSession(&serverSessionHandle, portHandle);
        ASSERT_RESULT_SUCCESS(result);

        for (int i = 0; i < TestNum; i++)
        {
            NamedPort(1, serverSessionHandle, true, false);
            NamedPort(1, serverSessionHandle, true, true);
            NamedPort(1, serverSessionHandle, false, false);
            NamedPort(1, serverSessionHandle, true, false);
            NamedPort(1, serverSessionHandle, true, true);
            NamedPort(1, serverSessionHandle, false, false);
        }
        NamedPortLargeBuffer0(1, serverSessionHandle, false);
        NamedPortLargeBuffer0(1, serverSessionHandle, true);
        NamedPortLargeBuffer1(1, serverSessionHandle, false);
        NamedPortLargeBuffer1(1, serverSessionHandle, true);
        result = nn::svc::CloseHandle(serverSessionHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &g_Thread0, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(clientSessionHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(g_Thread);
        ASSERT_RESULT_SUCCESS(result);
    }

    result = nn::svc::CloseHandle(portHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ManageNamedPort(&portHandle, TestPortName, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(portHandle == nn::svc::Handle(0));
}

TEST(Port, Test0)
{
    nn::Result result;
    int32_t index;
    nn::svc::Handle clientPortHandle;
    nn::svc::Handle serverPortHandle;
    nn::svc::Handle sessionHandle;
    nn::svc::Handle sessionHandle0;
    nn::svc::Handle sessionHandle1;
    nn::svc::Handle sessionHandle2;

    g_Thread1 = static_cast<nn::svc::Handle>(nn::os::GetCurrentThread()->_handle);
    result = nn::svc::CreatePort(&serverPortHandle, &clientPortHandle, 2, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ConnectToPort(&sessionHandle0, clientPortHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ConnectToPort(&sessionHandle1, clientPortHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ConnectToPort(&sessionHandle2, clientPortHandle);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxSessions());

    result = nn::svc::CloseHandle(sessionHandle0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &serverPortHandle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::AcceptSession(&sessionHandle, serverPortHandle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(sessionHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ConnectToPort(&sessionHandle2, clientPortHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &serverPortHandle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::AcceptSession(&sessionHandle, serverPortHandle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(sessionHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &serverPortHandle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::AcceptSession(&sessionHandle, serverPortHandle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(sessionHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(sessionHandle1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(sessionHandle2);
    ASSERT_RESULT_SUCCESS(result);

    {
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
        uintptr_t pc = reinterpret_cast<uintptr_t>(SessionThread);
        nn::svc::Handle serverSessionHandle;
        nn::svc::Handle clientSessionHandle;

        result = nn::svc::CreateThread(&g_Thread, reinterpret_cast<uintptr_t>(DummyThread), 0, sp, 32, 0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::StartThread(g_Thread);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::WaitSynchronization(&index, &g_Thread, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::GetThreadId(&g_ThreadId, g_Thread);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::ConnectToPort(&clientSessionHandle, clientPortHandle);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 28-3
        // Port のクライアントセッションを通してリクエストが送信できる
        // TEST 28-4
        // リプライが送信されるまで、システムコールが返らない
        // TEST 29-3
        // Port のクライアントセッションを通してリクエストが送信できる
        // TEST 29-4
        // リプライが送信されるまで、システムコールが返らない
        // TEST 30-3
        // Port のクライアントセッションを通してリクエストが送信できる
        // TEST 30-4
        // イベントオブジェクトを通じてリプライの通知を受け取ることが出来る
        result = nn::svc::CreateThread(&g_Thread0, pc, static_cast<nnHandle>(clientSessionHandle).value, sp, 32, 0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::StartThread(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::AcceptSession(&serverSessionHandle, serverPortHandle);
        ASSERT_RESULT_SUCCESS(result);
        for (int i = 0; i < TestNum; i++)
        {
            NamedPort(1, serverSessionHandle, true, false);
            NamedPort(1, serverSessionHandle, true, true);
            NamedPort(1, serverSessionHandle, false, false);
            NamedPort(1, serverSessionHandle, true, false);
            NamedPort(1, serverSessionHandle, true, true);
            NamedPort(1, serverSessionHandle, false, false);
        }
        NamedPortLargeBuffer0(1, serverSessionHandle, false);
        NamedPortLargeBuffer0(1, serverSessionHandle, true);
        NamedPortLargeBuffer1(1, serverSessionHandle, false);
        NamedPortLargeBuffer1(1, serverSessionHandle, true);
        result = nn::svc::CloseHandle(serverSessionHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &g_Thread0, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(clientSessionHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(g_Thread);
        ASSERT_RESULT_SUCCESS(result);
    }

    result = nn::svc::CloseHandle(serverPortHandle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientPortHandle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(CreateSession, Test0)
{
    nn::Result result;
    int32_t index;

    g_Thread1 = static_cast<nn::svc::Handle>(nn::os::GetCurrentThread()->_handle);

    {
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
        uintptr_t pc = reinterpret_cast<uintptr_t>(SessionThread);
        nn::svc::Handle serverSessionHandle;
        nn::svc::Handle clientSessionHandle;

        result = nn::svc::CreateThread(&g_Thread, reinterpret_cast<uintptr_t>(DummyThread), 0, sp, 32, 0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::StartThread(g_Thread);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::WaitSynchronization(&index, &g_Thread, 1, -1);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::GetThreadId(&g_ThreadId, g_Thread);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CreateSession(&serverSessionHandle, &clientSessionHandle, false, 0);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 28-5
        // クライアントセッションを通してリクエストが送信できる
        // TEST 28-6
        // リプライが送信されるまで、システムコールが返らない
        // TEST 29-5
        // クライアントセッションを通してリクエストが送信できる
        // TEST 29-6
        // リプライが送信されるまで、システムコールが返らない
        // TEST 30-5
        // クライアントセッションを通してリクエストが送信できる
        // TEST 30-6
        // イベントオブジェクトを通じてリプライの通知を受け取ることが出来る
        result = nn::svc::CreateThread(&g_Thread0, pc, static_cast<nnHandle>(clientSessionHandle).value, sp, 32, 0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::StartThread(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);
        for (int i = 0; i < TestNum; i++)
        {
            NamedPort(1, serverSessionHandle, true, false);
            NamedPort(1, serverSessionHandle, true, true);
            NamedPort(1, serverSessionHandle, false, false);
            NamedPort(1, serverSessionHandle, true, false);
            NamedPort(1, serverSessionHandle, true, true);
            NamedPort(1, serverSessionHandle, false, false);
        }
        NamedPortLargeBuffer0(1, serverSessionHandle, false);
        NamedPortLargeBuffer0(1, serverSessionHandle, true);
        NamedPortLargeBuffer1(1, serverSessionHandle, false);
        NamedPortLargeBuffer1(1, serverSessionHandle, true);
        result = nn::svc::CloseHandle(serverSessionHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &g_Thread0, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(clientSessionHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(g_Thread);
        ASSERT_RESULT_SUCCESS(result);
    }
}

namespace {
void SessionClientThread1(nn::svc::Handle handle)
{
    AutoThreadExit autoExit;
    nn::Result result;

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::SendSyncRequest(handle);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
}
}

TEST(CreateSession, Test1)
{
    nn::Result result;
    int32_t index;

    g_Thread1 = static_cast<nn::svc::Handle>(nn::os::GetCurrentThread()->_handle);

    for (int i = 0; i < 5000; i++)
    {
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
        uintptr_t pc = reinterpret_cast<uintptr_t>(SessionClientThread1);
        nn::svc::Handle serverSessionHandle;
        nn::svc::Handle clientSessionHandle;

        result = nn::svc::CreateSession(&serverSessionHandle, &clientSessionHandle, false, 0);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CreateThread(&g_Thread0, pc, static_cast<nnHandle>(clientSessionHandle).value, sp, 32, 0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::StartThread(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(serverSessionHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &g_Thread0, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(clientSessionHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);
    }
}

namespace {
void SessionClientThread2(nn::svc::Handle handle)
{
    AutoThreadExit autoExit;
    nn::Result result;

    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::SendSyncRequest(handle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}
}


TEST(CreateSession, Test2)
{
    nn::Result result;
    int32_t index;

    g_Thread1 = static_cast<nn::svc::Handle>(nn::os::GetCurrentThread()->_handle);

    for (int i = 0; i < 5000; i++)
    {
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
        uintptr_t pc = reinterpret_cast<uintptr_t>(SessionClientThread2);
        nn::svc::Handle serverSessionHandle;
        nn::svc::Handle clientSessionHandle;

        result = nn::svc::CreateSession(&serverSessionHandle, &clientSessionHandle, false, 0);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CreateThread(&g_Thread0, pc, static_cast<nnHandle>(clientSessionHandle).value, sp, 32, 0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::StartThread(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);

        {
            nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.SetNull();
            BeforeServerIpc(pMsgBuffer, sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer));
            result = nn::svc::ReplyAndReceive(
                    &index, &serverSessionHandle, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
            AfterServerIpc(pMsgBuffer, sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer));
            ASSERT_RESULT_SUCCESS(result);
        }

        {
            nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.SetNull();
            BeforeServerIpc(pMsgBuffer, sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer));
            result = nn::svc::ReplyAndReceive(
                    &index, &serverSessionHandle, 1, serverSessionHandle, -1);
            AfterServerIpc(pMsgBuffer, sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer));
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
        }

        result = nn::svc::CloseHandle(serverSessionHandle);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::WaitSynchronization(&index, &g_Thread0, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(g_Thread0);
        ASSERT_RESULT_SUCCESS(result);
    }
}


