﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_Thread.h>
#include <nn/svc/svc_Tick.h>
#include <nn/svc/svc_HardwareParamsSelect.h>
#include <nn/svc/svc_MemoryMapSelect.h>

#if defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 ) \
    || defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 ) \
    || defined( NN_BUILD_CONFIG_HARDWARE_NX )
#include "test_TestTmrDevice_Jetson.h"
#endif

#if defined( NN_BUILD_CONFIG_HARDWARE_BDSLIMX6 )

namespace
{
const uint64_t GptBasePhysicalAddress = 0x02098000;

uintptr_t g_GptBase;
#define GPT_CR       (*reinterpret_cast<volatile uint32_t*>(g_GptBase + 0x00))
#define GPT_PR       (*reinterpret_cast<volatile uint32_t*>(g_GptBase + 0x04))
#define GPT_SR       (*reinterpret_cast<volatile uint32_t*>(g_GptBase + 0x08))
#define GPT_IR       (*reinterpret_cast<volatile uint32_t*>(g_GptBase + 0x0c))
#define GPT_OCR1     (*reinterpret_cast<volatile uint32_t*>(g_GptBase + 0x10))
#define GPT_OCR2     (*reinterpret_cast<volatile uint32_t*>(g_GptBase + 0x14))
#define GPT_OCR3     (*reinterpret_cast<volatile uint32_t*>(g_GptBase + 0x18))
#define GPT_ICR1     (*reinterpret_cast<volatile uint32_t*>(g_GptBase + 0x1c))
#define GPT_ICR2     (*reinterpret_cast<volatile uint32_t*>(g_GptBase + 0x20))
#define GPT_CNT      (*reinterpret_cast<volatile uint32_t*>(g_GptBase + 0x24))
const int InterruptNameGpt  = 87;
const int GptPeripheralClock = (132 / 2) * 1000 * 1000;  // 66MHz

void GptStart()
{
    GPT_CR = 0;
    GPT_IR = 0;
    GPT_CR |= 0x1 << 15;

    while (GPT_CR & (0x1 << 15))
    {
        asm volatile ("yield");
    }

    GPT_SR = 0x3f;
    GPT_CR |= 0x1 << 6;
    GPT_CR |= 0x1 << 1;
    GPT_OCR1 = GptPeripheralClock / 10;
    GPT_CR |= 0x1 << 0;
    GPT_IR = 0x01;
}

void GptStop()
{
    GPT_CR = 0;
    GPT_IR = 0;
}

void GptClearInterrupt()
{
    GPT_SR = 0x1;
}
}

TEST(InterruptEvent, Test)
{
    TestInterruptEventLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;

    result = nn::svc::QueryIoMapping(&g_GptBase, GptBasePhysicalAddress, 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 81-9
    // 使用可能な割り込み番号以外の値を指定すると失敗する
    result = nn::svc::CreateInterruptEvent(&handle, static_cast<nn::svc::Interrupt>(0x100), nn::svc::InterruptType_Level);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultNotFound());
}

TEST(InterruptEvent, Manual)
{
    TestInterruptEventLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    int32_t index;

    result = nn::svc::QueryIoMapping(&g_GptBase, GptBasePhysicalAddress, 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 81-1
    // 割り込みイベントを作成することが出来る
    GptStop();
    result = nn::svc::CreateInterruptEvent(&handle, static_cast<nn::svc::Interrupt>(InterruptNameGpt), nn::svc::InterruptType_Level);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    GptStart();

    nn::svc::SleepThread(100 * 1000 * 1000);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);

    GptClearInterrupt();

    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);

    // TEST 16-5
    // 割り込みイベントのシグナル状態を解除できる
    result = nn::svc::ClearEvent(handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    GptStop();
}

TEST(InterruptEvent, Auto)
{
    TestInterruptEventLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    int32_t index;

    result = nn::svc::QueryIoMapping(&g_GptBase, GptBasePhysicalAddress, 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 81-2
    // 割り込みイベントを作成することが出来る
    GptStop();
    result = nn::svc::CreateInterruptEvent(&handle, static_cast<nn::svc::Interrupt>(InterruptNameGpt), nn::svc::InterruptType_Edge);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    GptStart();

    nn::svc::SleepThread(100 * 1000 * 1000);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);

    result = nn::svc::ResetSignal(handle);
    ASSERT_RESULT_SUCCESS(result);

    GptClearInterrupt();

    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // TEST 16-12
    // 割り込みイベントが発生していない場合でも、シグナル状態を解除できる
    result = nn::svc::ClearEvent(handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    GptStop();
}
#endif

#if defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 ) \
    || defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 ) \
    || defined( NN_BUILD_CONFIG_HARDWARE_NX )
TEST(InterruptEvent, Test)
{
    TestInterruptEventLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;

    uintptr_t addr;
    result = nn::svc::QueryIoMapping(&addr, TmrPhysicalBaseAddr, 0x1000);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 81-9
    // 使用可能な割り込み番号以外の値を指定すると失敗する
    result = nn::svc::CreateInterruptEvent(&handle, static_cast<nn::svc::Interrupt>(0x100), nn::svc::InterruptType_Level);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultNotFound());
}

TEST(InterruptEvent, Manual)
{
    TestInterruptEventLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    int32_t index;

    TestTmrDriver tmr;

    // TEST 81-1
    // 割り込みイベントを作成することが出来る
    result = nn::svc::CreateInterruptEvent(&handle, static_cast<nn::svc::Interrupt>(tmr.GetInterruptNumber()), nn::svc::InterruptType_Level);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    tmr.SetTimer(100);
    tmr.StartTimer();

    nn::svc::SleepThread(100 * 1000 * 1000);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);

    tmr.ClearInterrupt();

    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);

    // TEST 16-5
    // 割り込みイベントのシグナル状態を解除できる
    result = nn::svc::ClearEvent(handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(InterruptEvent, Auto)
{
    TestInterruptEventLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    int32_t index;

    TestTmrDriver tmr;

    // TEST 81-2
    // 割り込みイベントを作成することが出来る
    result = nn::svc::CreateInterruptEvent(
            &handle, static_cast<nn::svc::Interrupt>(tmr.GetInterruptNumber()),
            nn::svc::InterruptType_Edge);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    tmr.SetTimer(100);
    tmr.StartTimer();

    nn::svc::SleepThread(100 * 1000 * 1000);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);

    result = nn::svc::ResetSignal(handle);
    ASSERT_RESULT_SUCCESS(result);

    tmr.ClearInterrupt();

    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // TEST 16-12
    // 割り込みイベントが発生していない場合でも、シグナル状態を解除できる
    result = nn::svc::ClearEvent(handle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &handle, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}
#endif

