﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include "util_DebugProcess.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <nn/svc/svc_ServerId.autogen.h>
#include <nn/svc/svc_DdId.autogen.h>
#include <nn/svc/svc_DmntId.autogen.h>
#include <nn/svc/svc_TcbId.autogen.h>
#include <cstring>

extern "C" void nnMain();

namespace {

const int InstructionSize = 4;
const int64_t WaitTime = 100ull * 1000 * 1000 * 1000;

void MovePc(int offset, nn::svc::Handle debugHandle, nn::Bit64 tid)
{
    nn::Result result;
    nn::svc::ThreadContext context;
    nn::Bit32 contextFlag;
    contextFlag = nn::svc::ContextFlag_Control;

    result = nn::svc::GetDebugThreadContext(&context, debugHandle, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    context.pc += offset,

    result = nn::svc::SetDebugThreadContext(debugHandle, tid, context, contextFlag);
    ASSERT_RESULT_SUCCESS(result);
}

} // namespace

TEST(DebugProcess, AccessViolationData32A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum + 1], 0x1);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 2);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000003u, // bne 1f
        0xe3a03000, // mov r3, #0
        0xe5930000, // ldr r0, [r3]
        0xeafffffe, // infinit loop
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc #NN_SVC_ID_EXIT_PROCESS
        0xe3004101, // movw r4, #0x101
        0xe1500004, // cmp r0, r4
        0x1a000001u, // bne 2f
        0xe59f0004, // ldr r0, [pc, #4]
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        0xeafffffe, // b 2f
        nn::svc::ResultNotHandled().InnerValue,
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    nn::svc::Handle debugHandle;
    nn::Bit64 tid;
    InitDebugProcess(&debugHandle, &tid, &process);

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    WaitDebugEvent(&debugInfo, debugHandle);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_AccessViolationData);

    MovePc(InstructionSize * 2, debugHandle, tid);

    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    ExitDebugProcess(debugHandle, 1);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 121-84
    // 作成したハンドルを使用後に閉じることが出来る
    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(DebugProcess, AccessViolationData32T)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum + 1], 0x1);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 2);
    nn::svc::Handle processHandle = process.GetHandle();

    uint16_t invSvc = 0xdf00;
    uint16_t mov = 0x2300; // movs r3, #0
    uint16_t ldr = 0x6818; // ldr r0, [r3, #0]
    uint16_t exitSvc = 0xdf00 | NN_SVC_ID_EXIT_PROCESS;
    uint32_t invalidAccess = mov | ldr << 16;
    uint32_t exitCode = invSvc | exitSvc << 16;
    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000002u, // bne 2f
        0xfaffffff, // blx 1f
        // 1:
        invalidAccess,
        exitCode,
        // 2:
        0xe3004101, // movw r4, #0x101
        0xe1500004, // cmp r0, r4
        0x1a000001u, // bne 3f
        0xe59f0004, // ldr r0, [pc, #4]
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        0xeafffffe, // b 3f
        nn::svc::ResultNotHandled().InnerValue,
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    nn::svc::Handle debugHandle;
    nn::Bit64 tid;
    InitDebugProcess(&debugHandle, &tid, &process);

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    WaitDebugEvent(&debugInfo, debugHandle);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_AccessViolationData);

    MovePc(InstructionSize, debugHandle, tid);

    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    ExitDebugProcess(debugHandle, 1);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
TEST(DebugProcess, AccessViolationData64A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit
        | nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum + 1], 0x1);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 2);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xf100001f, // cmp x0, #0
        0x540000a1, // b.ne 1f
        0xd2800003, // mov x3, #0
        0xf9400060, // ldr x0, [x3]
        0x14000000, // infinit loop
        (0xd4000001 | (NN_SVC_ID_EXIT_PROCESS << 5)),
        // 1:
        0xd2802024, // mov x4, #0x101
        0xeb04001f, // cmp x0, x4
        0x540000c1, // b.ne 2f
        0x58000080, // ldr x0, =result
        (0xd4000001 | (NN_SVC_ID_RETURN_FROM_EXCEPTION << 5)),
        // 2:
        0x14000000, // b 2b
        0x00000000,
        // result:
        nn::svc::ResultNotHandled().InnerValue,
        0x00000000,
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    nn::svc::Handle debugHandle;
    nn::Bit64 tid;
    InitDebugProcess(&debugHandle, &tid, &process);

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    WaitDebugEvent(&debugInfo, debugHandle);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_AccessViolationData);

    MovePc(InstructionSize * 2, debugHandle, tid);

    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    ExitDebugProcess(debugHandle, 1);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}
#endif

TEST(DebugProcess, AccessViolationInstruction32A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum + 1], 0x1);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 2);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000003u, // bne 1f
        0xe3a03000, // mov r3, #0
        0xe1a0f003, // mov pc, r3
        0xeafffffe, // infinit loop
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc #NN_SVC_ID_EXIT_PROCESS
        0xe3004100, // movw r4, #0x100
        0xe1500004, // cmp r0, r4
        0x1a000001u, // bne 2f
        0xe59f0004, // ldr r0, [pc, #4]
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        0xeafffffe, // b 2f
        nn::svc::ResultNotHandled().InnerValue,
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    nn::svc::Handle debugHandle;
    nn::Bit64 tid;
    InitDebugProcess(&debugHandle, &tid, &process);

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    WaitDebugEvent(&debugInfo, debugHandle);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_AccessViolationInstruction);

    // PC は 0 なので、直接番地を入れる
    MovePc(process.GetCodeAddress() + InstructionSize * 5, debugHandle, tid);

    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    ExitDebugProcess(debugHandle, 1);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 121-84
    // 作成したハンドルを使用後に閉じることが出来る
    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(DebugProcess, AccessViolationInstruction32T)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum + 1], 0x1);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 2);
    nn::svc::Handle processHandle = process.GetHandle();

    uint16_t mov = 0x2300; // movs r3, #0
    uint16_t changePc = 0x469f; // movs pc, r3
    uint16_t exitSvc = 0xdf00 | NN_SVC_ID_EXIT_PROCESS;
    uint32_t invalidInstruction = mov | changePc << 16;
    uint32_t exitCode = exitSvc;

    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000002u, // bne 2f
        0xfaffffff, // blx 1f
        // 1:
        invalidInstruction,
        exitCode,
        // 2:
        0xe3004100, // movw r4, #0x100
        0xe1500004, // cmp r0, r4
        0x1a000001u, // bne 3f
        0xe59f0004, // ldr r0, [pc, #4]
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        0xeafffffe, // b 3f
        nn::svc::ResultNotHandled().InnerValue,
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    nn::svc::Handle debugHandle;
    nn::Bit64 tid;
    InitDebugProcess(&debugHandle, &tid, &process);

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    WaitDebugEvent(&debugInfo, debugHandle);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_AccessViolationInstruction);

    // PC は 0 なので、直接番地を入れる
    MovePc(process.GetCodeAddress() + InstructionSize * 4, debugHandle, tid);

    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    ExitDebugProcess(debugHandle, 1);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
TEST(DebugProcess, AccessViolationInstruction64A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit
        | nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum + 1], 0x1);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 2);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xf100001f, // cmp x0, #0
        0x540000a1, // b.ne 1f
        0xd2800003, // mov x3, #0
        0xd61f0060, // br x3
        0x14000000, // infinit loop
        (0xd4000001 | (NN_SVC_ID_EXIT_PROCESS << 5)),
        // 1:
        0xd2802004, // mov x4, #0x100
        0xeb04001f, // cmp x0, x4
        0x540000c1, // b.ne 2f
        0x58000080, // ldr x0, =result
        (0xd4000001 | (NN_SVC_ID_RETURN_FROM_EXCEPTION << 5)),
        // 2:
        0x14000000, // b 2b
        0x00000000,
        // result:
        nn::svc::ResultNotHandled().InnerValue,
        0x00000000,
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    nn::svc::Handle debugHandle;
    nn::Bit64 tid;
    InitDebugProcess(&debugHandle, &tid, &process);

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    WaitDebugEvent(&debugInfo, debugHandle);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_AccessViolationInstruction);

    // PC は 0 なので、直接番地を入れる
    MovePc(process.GetCodeAddress() + InstructionSize * 5, debugHandle, tid);

    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    ExitDebugProcess(debugHandle, 1);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}
#endif

TEST(DebugProcess, UndefinedInstruction32A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum + 1], 0x1);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 2);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000003u, // bne 1f
        0xe7f000f0, // permanentaly undefine instruction
        0xeafffffe, // infinit loop
        0xeafffffe, // infinit loop
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc #NN_SVC_ID_EXIT_PROCESS
        0xe3004104, // movw r4, #0x104
        0xe1500004, // cmp r0, r4
        0x1a000001u, // bne 2f
        0xe59f0004, // ldr r0, [pc, #4]
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        0xeafffffe, // b 2f
        nn::svc::ResultNotHandled().InnerValue,
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    nn::svc::Handle debugHandle;
    nn::Bit64 tid;
    InitDebugProcess(&debugHandle, &tid, &process);

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    WaitDebugEvent(&debugInfo, debugHandle);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_UndefinedInstruction);

    MovePc(InstructionSize * 3, debugHandle, tid);

    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    ExitDebugProcess(debugHandle, 1);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 121-84
    // 作成したハンドルを使用後に閉じることが出来る
    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(DebugProcess, UndefinedInstruction32T)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum + 1], 0x1);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 2);
    nn::svc::Handle processHandle = process.GetHandle();

    uint16_t udf = 0xde00; // movs r3, #0
    uint16_t exitSvc = 0xdf00 | NN_SVC_ID_EXIT_PROCESS;
    uint16_t invSvc = 0xdf00;
    uint32_t undefInstruction = udf | invSvc << 16;
    uint32_t exitCode = invSvc | exitSvc << 16;

    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000002u, // bne 2f
        0xfaffffff, // blx 1f
        // 1:
        undefInstruction,
        exitCode,
        // 2:
        0xe3004104, // movw r4, #0x104
        0xe1500004, // cmp r0, r4
        0x1a000001u, // bne 3f
        0xe59f0004, // ldr r0, [pc, #4]
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        0xeafffffe, // b 3f
        nn::svc::ResultNotHandled().InnerValue,
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    nn::svc::Handle debugHandle;
    nn::Bit64 tid;
    InitDebugProcess(&debugHandle, &tid, &process);

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    WaitDebugEvent(&debugInfo, debugHandle);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_UndefinedInstruction);

    MovePc(InstructionSize + InstructionSize / 2, debugHandle, tid);

    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    ExitDebugProcess(debugHandle, 1);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
TEST(DebugProcess, UndefinedInstruction64A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit
        | nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum + 1], 0x1);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 2);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xf100001f, // cmp x0, #0
        0x540000a1, // b.ne 1f
        0xe7f000f0, // permanentaly undefine instruction
        0x14000000, // infinit loop
        0x14000000, // infinit loop
        (0xd4000001 | (NN_SVC_ID_EXIT_PROCESS << 5)),
        // 1:
        0xd2802084, // mov x4, #0x104
        0xeb04001f, // cmp x0, x4
        0x540000c1, // b.ne 2f
        0x58000080, // ldr x0, =result
        (0xd4000001 | (NN_SVC_ID_RETURN_FROM_EXCEPTION << 5)),
        // 2:
        0x14000000, // b 2b
        0x00000000,
        // result:
        nn::svc::ResultNotHandled().InnerValue,
        0x00000000,
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    nn::svc::Handle debugHandle;
    nn::Bit64 tid;
    InitDebugProcess(&debugHandle, &tid, &process);

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    WaitDebugEvent(&debugInfo, debugHandle);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_UndefinedInstruction);

    MovePc(InstructionSize * 3, debugHandle, tid);

    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    ExitDebugProcess(debugHandle, 1);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}
#endif

TEST(DebugProcess, UndefinedSystemCall32A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum + 1], 0x1);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 2);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000003u, // bne 1f
        0xef000000, // svc #0
        0xeafffffe, // infinit loop
        0xeafffffe, // infinit loop
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc #NN_SVC_ID_EXIT_PROCESS
        0xe3004301, // mov r4, #0x301
        0xe1500004, // cmp r0, r4
        0x1a000001u, // bne 2f
        0xe59f0004, // ldr r0, [pc, #4]
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        0xeafffffe, // b 2f
        nn::svc::ResultNotHandled().InnerValue,
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    nn::svc::Handle debugHandle;
    nn::Bit64 tid;
    InitDebugProcess(&debugHandle, &tid, &process);

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    WaitDebugEvent(&debugInfo, debugHandle);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_UndefinedSystemCall);

    MovePc(InstructionSize * 3, debugHandle, tid);

    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    ExitDebugProcess(debugHandle, 1);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 121-84
    // 作成したハンドルを使用後に閉じることが出来る
    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(DebugProcess, UndefinedSystemCall32T)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum + 1], 0x1);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 2);
    nn::svc::Handle processHandle = process.GetHandle();

    uint16_t exitSvc = 0xdf00 | NN_SVC_ID_EXIT_PROCESS;
    uint16_t invSvc = 0xdf00;
    uint32_t errorCode = invSvc | invSvc << 16;
    uint32_t exitCode = invSvc | exitSvc << 16;

    uint32_t codes[] = {
        0xe3500000, // cmp r0, #0
        0x1a000002u, // bne 2f
        0xfaffffff, // blx 1f
        // 1:
        errorCode,
        exitCode,
        // 2:
        0xe3004301, // mov r4, #0x301
        0xe1500004, // cmp r0, r4
        0x1a000001u, // bne 3f
        0xe59f0004, // ldr r0, [pc, #4]
        (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION
        0xeafffffe, // b 3f
        nn::svc::ResultNotHandled().InnerValue,
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    nn::svc::Handle debugHandle;
    nn::Bit64 tid;
    InitDebugProcess(&debugHandle, &tid, &process);

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    WaitDebugEvent(&debugInfo, debugHandle);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_UndefinedSystemCall);

    MovePc(InstructionSize + InstructionSize / 2, debugHandle, tid);

    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    ExitDebugProcess(debugHandle, 1);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
TEST(DebugProcess, UndefinedSystemCall64A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    TestHeap heap(HeapAlign);
    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());
    nn::Bit32 paramFlag = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit
        | nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum + 1], 0x1);

    TestProcess process(1, paramFlag, flags, DefaultCapabilityFlagNum + 2);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xf100001f, // cmp x0, #0
        0x540000a1, // b.ne 1f
        0xd4000001,
        0x14000000, // infinit loop
        0x14000000, // infinit loop
        (0xd4000001 | (NN_SVC_ID_EXIT_PROCESS << 5)),
        // 1:
        0xd2806024, // mov r4, #0x301
        0xeb04001f, // cmp x0, x4
        0x540000c1, // b.ne 2f
        0x58000080, // ldr x0, =result
        (0xd4000001 | (NN_SVC_ID_RETURN_FROM_EXCEPTION << 5)),
        // 2:
        0x14000000, // b 2b
        0x00000000,
        // result:
        nn::svc::ResultNotHandled().InnerValue,
        0x00000000,
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    nn::svc::Handle debugHandle;
    nn::Bit64 tid;
    InitDebugProcess(&debugHandle, &tid, &process);

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    WaitDebugEvent(&debugInfo, debugHandle);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_UndefinedSystemCall);

    MovePc(InstructionSize * 3, debugHandle, tid);

    result = ContinueDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    ExitDebugProcess(debugHandle, 1);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, WaitTime);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);
}
#endif

