﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Kernel.h>
#include <nn/svc/svc_Thread.h>
#include <nn/svc/svc_Result.h>

namespace {

char g_Buffer[DefaultStackSize] __attribute__((aligned(0x1000)));

void TestThreadFunction0(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
}

} // namespace

extern "C" void nnMain();

/* TEST: 7-10 */
TEST(CreateThreadTest, MaxIdealCore)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    uintptr_t pc;
    uintptr_t sp;
    int32_t idealCore;
    int32_t priority;

    pc = reinterpret_cast<uintptr_t>(TestThreadFunction0);
    sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));

    for (idealCore = NumCore; idealCore >= NumCore && idealCore <= INT32_MAX; idealCore <<= 1)
    {
        for (priority = TestHighestThreadPriority;
                priority <= TestLowestThreadPriority; priority++)
        {
            result = nn::svc::CreateThread(&handle, pc, 0, sp, priority, idealCore);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCoreNumber());
        }
    }
}

#ifdef INVALID_POINTER_TEST
TEST(CreateThreadTest, InvalidPointerTest)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    uintptr_t pc = reinterpret_cast<uintptr_t>(TestThreadFunction0);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
    int32_t idealCore = 0;
    int32_t priority = TestLowestThreadPriority;

#ifdef INVALID_POINTER_TEST
    // TEST 7-13
    // pOut は NULL を受け付けない
    result = nn::svc::CreateThread(NULL, pc, 0, sp, priority, idealCore);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 7-14
    // pOut は MemoryPermission_None を受け付けない
    result = nn::svc::CreateThread(
            reinterpret_cast<nn::svc::Handle*>(g_FreeAreaBegin), pc, 0, sp, priority, idealCore);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 7-15
    uintptr_t heapPtr;
    result = nn::svc::SetHeapSize(&heapPtr, HeapAlign);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SetMemoryPermission(heapPtr, 0x1000, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);

    // pOut は MemoryPermission_Read を受け付けない
    result = nn::svc::CreateThread(
            reinterpret_cast<nn::svc::Handle*>(heapPtr), pc, 0, sp, priority, idealCore);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());

    result = nn::svc::SetMemoryPermission(heapPtr, 0x1000, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::SetHeapSize(&heapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 7-16
    // pOut は MemoryPermission_ReadExecute を受け付けない
    result = nn::svc::CreateThread(
            reinterpret_cast<nn::svc::Handle*>(nnMain), pc, 0, sp, priority, idealCore);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST
}
#endif

#ifdef INVALID_POINTER_TEST
TEST (CreateThreadTest, InvalidFunctionTest)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
    int32_t idealCore = 0;
    int32_t priority = TestLowestThreadPriority;

    // TEST 7-17
    // f に 関数のアドレスが入っていないと実行時にユーザー例外が発生する
    result = nn::svc::CreateThread(&handle, g_FreeAreaBegin, 0, sp, priority, idealCore);
    ASSERT_RESULT_SUCCESS(result);

    // 成功はするが、ユーザー例外が発生する
    result = nn::svc::StartThread(handle);
    ASSERT_RESULT_SUCCESS(result);

    // 念のため終了処理
    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}
#endif

#ifdef INVALID_POINTER_TEST
TEST (CreateThreadTest, InvalidStackPointerTest)
{
    TestThreadLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    uintptr_t pc = reinterpret_cast<uintptr_t>(TestThreadFunction0);
    int32_t idealCore = 0;
    int32_t priority = TestLowestThreadPriority;

    // TEST 7-18
    // stackBottom に有効なメモリ領域が指定されていないとユーザー例外が発生する
    result = nn::svc::CreateThread(&handle, pc, 0, g_FreeAreaBegin, priority, idealCore);
    ASSERT_RESULT_SUCCESS(result);

    // 成功はするが、ユーザー例外が発生する
    result = nn::svc::StartThread(handle);
    ASSERT_RESULT_SUCCESS(result);

    // 念のため終了処理
    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}
#endif

