﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <nn/svc/svc_ServerId.autogen.h>
#include <nn/svc/svc_DdId.autogen.h>
#include <nn/svc/svc_DmntId.autogen.h>
#include <nn/svc/svc_TcbId.autogen.h>
#include <cstring>

extern "C" void nnMain();

namespace {

const int64_t WaitTime = 100ull * 1000 * 1000 * 1000;
const uintptr_t CodeSize = 0x1000;

bool IsEnableSvcId(int32_t id)
{
    if (id < NN_SVC_ID_SET_HEAP_SIZE ||
        id > NN_SVC_ID_GET_INFO)
    {
        return false;
    }
    return true;
}

} // namespace

// TEST 121-109
// No.4 のフラグで指定したシステムコールを呼ぶことが出来る
TEST(CreateProcess, SuccessSyscall32A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    TestHeap heap(HeapAlign);
    uintptr_t CodeAddress;

    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultParam(&param);
#if defined SVC_AGING_TEST
#if defined NN_BUILD_CONFIG_CPU_ARM_V7A
    const int32_t MaxSvcNum = (1 << 24) - 1;
#elif defined NN_BUILD_CONFIG_CPU_ARM_V8A
    const int32_t MaxSvcNum = UINT16_MAX;
#endif // NN_BUILD_CONFIG_CPU_ARM
#else
    const int32_t MaxSvcNum = 8 * 24;
#endif // defined SVC_AGING_TEST

    for (int32_t i = 1; i < MaxSvcNum; i++)
    {
        // これはスレッドのみが死んでしまい、テストが続行できなくなるので、テストしない
        if (i == NN_SVC_ID_EXIT_THREAD)
        {
            continue;
        }
        // プロセスを終了させられないので、これだけはチェックしない
        if (i == NN_SVC_ID_EXIT_PROCESS || i == NN_SVC_ID_RETURN_FROM_EXCEPTION)
        {
            continue;
        }
        if (!IsEnableSvcId(i))
        {
            continue;
        }

        SetDefaultCapability(flags, DefaultCapabilityFlagNum);
        flags[1] = PaddingFlag;
        int32_t flagNum;
        int maxFlagNum = heap.GetSize() / sizeof(nn::Bit32) - DefaultCapabilityFlagNum;
        int32_t syscalls[3] = { NN_SVC_ID_EXIT_PROCESS, NN_SVC_ID_RETURN_FROM_EXCEPTION, i };
        MakeNo4Flag(&flags[DefaultCapabilityFlagNum], &flagNum, maxFlagNum, syscalls, 3);
        if ((param.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
        {
            param.memoryAddress = NN_SVC_ADDR_SMALL_MAP64_BEGIN;
        }
        else
        {
            param.memoryAddress = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
        }
        CodeAddress = param.memoryAddress;

        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum + flagNum);
        if (result <= nn::svc::ResultOutOfRange())
        {
            continue;
        }
        ASSERT_RESULT_SUCCESS(result);

        uint32_t codes[] = {
            0xe3500000, // cmp r0, #0
            0x1a000008u, // bne 1f
            0xe3a01001, // mov r1, #1
            0xe3a02000, // mov r2, #0
            0xe3a03000, // mov r3, #0
            0xe3a04000, // mov r4, #0
            0xe3a05000, // mov r5, #0
            0xe3a06000, // mov r6, #0
            0xe3a07000, // mov r7, #0
            (0xef000000 | i), // svc #i
            (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc #NN_SVC_ID_EXIT_PROCESS

            // 1:
            0xe3004301, // mov r4, #0x301
            0xe1500004, // cmp r0, r4
            0x0a000004u,  // beq 2f
            0xe5915028, // ldr r5, [r1, #40]
            0xe2855004, // add r5, #4
            0xe5815028, // str r5, [r1, #40]
            0xe3a00000, // mov r0, #0
            (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION

            // 2:
            0xeafffffe, // b 2b
        };

        SetExecuteCode(handle, codes, sizeof(codes), CodeAddress, CodeSize);

        result = nn::svc::StartProcess(handle, TestLowestThreadPriority, 0, DefaultStackSize);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::ResetSignal(handle);

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &handle, 1, WaitTime);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
}

// TEST 121-110
// No.4 のフラグに指定したシステムコール以外は呼ぶことが出来ない
TEST(CreateProcess, InvalidSystemCall32A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    TestHeap heap(HeapAlign);
    uintptr_t CodeAddress;

    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultParam(&param);
#if defined SVC_AGING_TEST
#if defined NN_BUILD_CONFIG_CPU_ARM_V7A
    const int32_t MaxSvcNum = (1 << 24) - 1;
#elif defined NN_BUILD_CONFIG_CPU_ARM_V8A
    const int32_t MaxSvcNum = UINT16_MAX;
#endif // NN_BUILD_CONFIG_CPU_ARM
#else
    const int32_t MaxSvcNum = 8 * 24;
#endif // defined SVC_AGING_TEST

    for (int32_t i = 1; i < MaxSvcNum; i++)
    {
        // プロセスを終了させられないので、これだけはチェックしない
        if (i == NN_SVC_ID_EXIT_PROCESS || i == NN_SVC_ID_RETURN_FROM_EXCEPTION)
        {
            continue;
        }
        SetDefaultCapability(flags, DefaultCapabilityFlagNum);
        flags[1] = PaddingFlag;
        int32_t flagNum;
        int maxFlagNum = heap.GetSize() / sizeof(nn::Bit32) - DefaultCapabilityFlagNum;
        int32_t syscalls[2] = { NN_SVC_ID_EXIT_PROCESS, NN_SVC_ID_RETURN_FROM_EXCEPTION };
        MakeNo4Flag(&flags[DefaultCapabilityFlagNum], &flagNum, maxFlagNum, syscalls, 2);
        if ((param.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
        {
            param.memoryAddress = NN_SVC_ADDR_SMALL_MAP64_BEGIN;
        }
        else
        {
            param.memoryAddress = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
        }
        CodeAddress = param.memoryAddress;

        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum + flagNum);
        ASSERT_RESULT_SUCCESS(result);

        uint32_t codes[] = {
            0xe3500000, // cmp r0, #0
            0x1a000009u, // bne 1f
            0xe3a01001, // mov r1, #1
            0xe3a02000, // mov r2, #0
            0xe3a03000, // mov r3, #0
            0xe3a04000, // mov r4, #0
            0xe3a05000, // mov r5, #0
            0xe3a06000, // mov r6, #0
            0xe3a07000, // mov r7, #0
            (0xef000000 | i), // svc #i
            0xeafffffe, // loop
            (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc #NN_SVC_ID_EXIT_PROCESS

            // 1:
            0xe3004301, // mov r4, #0x301
            0xe1500004, // cmp r0, r4
            0x1a000004u, // bne 2f
            0xe5915028, // ldr r5, [r1, #40]
            0xe2855008, // add r5, #8
            0xe5815028, // str r5, [r1, #40]
            0xe3a00000, // mov r0, #0
            (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION

            // 2:
            0xeafffffe, // b 2
        };

        SetExecuteCode(handle, codes, sizeof(codes), CodeAddress, CodeSize);

        result = nn::svc::StartProcess(handle, TestLowestThreadPriority, 0, DefaultStackSize);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::ResetSignal(handle);

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &handle, 1, WaitTime);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
}

// TEST 121-124
// No.4 のフラグで指定したシステムコールを呼ぶことが出来る
TEST(CreateProcess, SuccessSyscall32T)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    TestHeap heap(HeapAlign);
    uintptr_t CodeAddress;

    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultParam(&param);
    const uint8_t MaxSvcNum = UINT8_MAX;

    for (int32_t i = 1; i < MaxSvcNum; i++)
    {
        // これはスレッドのみが死んでしまい、テストが続行できなくなるので、テストしない
        if (i == NN_SVC_ID_EXIT_THREAD)
        {
            continue;
        }
        // プロセスを終了させられないので、これだけはチェックしない
        if (i == NN_SVC_ID_EXIT_PROCESS || i == NN_SVC_ID_RETURN_FROM_EXCEPTION)
        {
            continue;
        }
        if (!IsEnableSvcId(i))
        {
            continue;
        }

        SetDefaultCapability(flags, DefaultCapabilityFlagNum);
        flags[1] = PaddingFlag;
        int32_t flagNum;
        int maxFlagNum = heap.GetSize() / sizeof(nn::Bit32) - DefaultCapabilityFlagNum;
        int32_t syscalls[3] = { NN_SVC_ID_EXIT_PROCESS, NN_SVC_ID_RETURN_FROM_EXCEPTION, i };
        MakeNo4Flag(&flags[DefaultCapabilityFlagNum], &flagNum, maxFlagNum, syscalls, 3);
        if ((param.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
        {
            param.memoryAddress = NN_SVC_ADDR_SMALL_MAP64_BEGIN;
        }
        else
        {
            param.memoryAddress = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
        }
        CodeAddress = param.memoryAddress;

        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum + flagNum);
        ASSERT_RESULT_SUCCESS(result);

        uint8_t svc1 = 0xdf00 | i;
        uint16_t svc2 = 0xdf00 | NN_SVC_ID_EXIT_PROCESS;
        // リトルエンディアンで命令を格納
        uint32_t svcCall1 = svc1 | svc2 << 16;

        uint32_t codes[] = {
            0xe3500000, // cmp r0, #0
            0x1a000008u, // bne 2f
            0xe3a01001, // mov r1, #1
            0xe3a02000, // mov r2, #0
            0xe3a03000, // mov r3, #0
            0xe3a04000, // mov r4, #0
            0xe3a05000, // mov r5, #0
            0xe3a06000, // mov r6, #0
            0xe3a07000, // mov r7, #0
            0xfaffffff, // blx 1f

            // 1:
            svcCall1, // svc #i; svc NN_SVC_ID_EXIT_PROCESS

            // 2:
            0xe3004301, // mov r4, #0x301
            0xe1500004, // cmp r0, r4
            0x0a000004u,  // beq 3f
            0xe5915028, // ldr r5, [r1, #40]
            0xe2855002, // add r5, #2
            0xe5815028, // str r5, [r1, #40]
            0xe3a00000, // mov r0, #0
            (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION

            // 3:
            0,
            //0xeafffffe, // b 3b
        };

        SetExecuteCode(handle, codes, sizeof(codes), CodeAddress, CodeSize);

        result = nn::svc::StartProcess(handle, TestLowestThreadPriority, 0, DefaultStackSize);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::ResetSignal(handle);

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &handle, 1, WaitTime);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
}

// TEST 121-125
// No.4 のフラグで指定したシステムコール以外は呼ぶことが出来ない
TEST(CreateProcess, InvalidSystemCall32T)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    TestHeap heap(HeapAlign);
    uintptr_t CodeAddress;

    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultParam(&param);
    const uint8_t MaxSvcNum = UINT8_MAX;

    for (uint8_t i = 1; i < MaxSvcNum; i++)
    {
        // プロセスを終了させられないので、これだけはチェックしない
        if (i == NN_SVC_ID_EXIT_PROCESS || i == NN_SVC_ID_RETURN_FROM_EXCEPTION)
        {
            continue;
        }
        SetDefaultCapability(flags, DefaultCapabilityFlagNum);
        flags[1] = PaddingFlag;
        int32_t flagNum;
        int maxFlagNum = heap.GetSize() / sizeof(nn::Bit32) - DefaultCapabilityFlagNum;
        int32_t syscalls[2] = { NN_SVC_ID_EXIT_PROCESS, NN_SVC_ID_RETURN_FROM_EXCEPTION };
        MakeNo4Flag(&flags[DefaultCapabilityFlagNum], &flagNum, maxFlagNum, syscalls, 2);
        if ((param.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
        {
            param.memoryAddress = NN_SVC_ADDR_SMALL_MAP64_BEGIN;
        }
        else
        {
            param.memoryAddress = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
        }
        CodeAddress = param.memoryAddress;

        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum + flagNum);
        ASSERT_RESULT_SUCCESS(result);

        uint16_t svc1 = 0xdf00 | i;
        uint16_t svc2 = 0xdf00 | NN_SVC_ID_EXIT_PROCESS;
        uint16_t loop = 0xe7feu;
        // リトルエンディアンで命令を格納
        uint32_t svcCall1 = svc1 | loop << 16;
        uint32_t svcCall2 = svc2 | loop << 16;

        uint32_t codes[] = {
            0xe3500000, // cmp r0, #0
            0x1a000009u, // bne 2f
            0xe3a01001, // mov r1, #1
            0xe3a02000, // mov r2, #0
            0xe3a03000, // mov r3, #0
            0xe3a04000, // mov r4, #0
            0xe3a05000, // mov r5, #0
            0xe3a06000, // mov r6, #0
            0xe3a07000, // mov r7, #0
            0xfaffffff, // blx 1f

            // 1:
            svcCall1, // svc #i; b 1;
            svcCall2, // svc #i; b 1;

            // 2:
            0xe3004301, // mov r4, #0x301
            0xe1500004, // cmp r0, r4
            0x1a000004u, // bne 3f
            0xe5915028, // ldr r5, [r1, #40]
            0xe2855004, // add r5, #4
            0xe5815028, // str r5, [r1, #40]
            0xe3a00000, // mov r0, #0
            (0xef000000 | NN_SVC_ID_RETURN_FROM_EXCEPTION), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION

            // 3:
            0xeafffffe, // b 3f
        };

        SetExecuteCode(handle, codes, sizeof(codes), CodeAddress, CodeSize);

        result = nn::svc::StartProcess(handle, TestLowestThreadPriority, 0, DefaultStackSize);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::ResetSignal(handle);

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &handle, 1, WaitTime);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
}

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
// TEST 121-126
// No.4 のフラグで指定したシステムコールを呼ぶことが出来る
TEST(CreateProcess, SuccessSyscall64A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    TestHeap heap(HeapAlign);
    uintptr_t CodeAddress;

    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultParam(&param);
#if defined SVC_AGING_TEST
    const int32_t MaxSvcNum = UINT16_MAX;
#else
    const int32_t MaxSvcNum = 8 * 24;
#endif // defined SVC_AGING_TEST

    param.flags = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;

    for (int32_t i = 1; i < MaxSvcNum; i++)
    {
        // これはスレッドのみが死んでしまい、テストが続行できなくなるので、テストしない
        if (i == NN_SVC_ID_EXIT_THREAD)
        {
            continue;
        }
        // プロセスを終了させられないので、これだけはチェックしない
        if (i == NN_SVC_ID_EXIT_PROCESS || i == NN_SVC_ID_RETURN_FROM_EXCEPTION)
        {
            continue;
        }
        if (!IsEnableSvcId(i))
        {
            continue;
        }

        SetDefaultCapability(flags, DefaultCapabilityFlagNum);
        MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);
        MakeNo4Flag(&flags[DefaultCapabilityFlagNum], i);
        flags[1] = PaddingFlag;
        int32_t flagNum;
        int maxFlagNum = heap.GetSize() / sizeof(nn::Bit32) - DefaultCapabilityFlagNum;
        int32_t syscalls[3] = { NN_SVC_ID_EXIT_PROCESS, NN_SVC_ID_RETURN_FROM_EXCEPTION, i };
        MakeNo4Flag(&flags[DefaultCapabilityFlagNum], &flagNum, maxFlagNum, syscalls, 3);
        if ((param.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
        {
            param.memoryAddress = NN_SVC_ADDR_SMALL_MAP64_BEGIN;
        }
        else
        {
            param.memoryAddress = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
        }
        CodeAddress = param.memoryAddress;
        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum + flagNum);
        ASSERT_RESULT_SUCCESS(result);

        uint32_t codes[] = {
            0xf100001f, // cmp r0, #0
            0x54000141, // bne 1f
            0xd2800021, // mov r1, #1
            0xd2800002, // mov r2, #0
            0xd2800003, // mov r3, #0
            0xd2800004, // mov r4, #0
            0xd2800005, // mov r5, #0
            0xd2800006, // mov r6, #0
            0xd2800007, // mov r7, #0
            (0xd4000001 | (i << 5)), // svc #i
            (0xd4000001 | (NN_SVC_ID_EXIT_PROCESS << 5)), // svc #NN_SVC_ID_EXIT_PROCESS

            // 1:
            0xd2806024, // mov r4, #0x301
            0xeb04001f, // cmp r0, r4
            0x540000c0u,  // b.eq 2f
            0xf9402c25, // ldr r5, [r1, #88]
            0x910020a5, // add r5, #8
            0xf9002c25, // str r5, [r1, #88]
            0xd2800000, // mov r0, #0
            (0xd4000001 | (NN_SVC_ID_RETURN_FROM_EXCEPTION << 5)), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION

            // 2:
            0x14000000u, // b 2f
        };

        SetExecuteCode(handle, codes, sizeof(codes), CodeAddress, CodeSize);

        result = nn::svc::StartProcess(handle, TestLowestThreadPriority, 0, DefaultStackSize);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::ResetSignal(handle);

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &handle, 1, WaitTime);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
}

// TEST 121-127
// No.4 のフラグに指定したシステムコール以外は呼ぶことが出来ない
TEST(CreateProcess, InvalidSystemCall64A)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    TestHeap heap(HeapAlign);
    uintptr_t CodeAddress;

    nn::Bit32* flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    SetDefaultParam(&param);
#if defined SVC_AGING_TEST
    const int32_t MaxSvcNum = UINT16_MAX;
#else
    const int32_t MaxSvcNum = 8 * 24;
#endif // defined SVC_AGING_TEST

    param.flags = nn::svc::CreateProcessParameterFlag_64Bit
        | nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;

    if ((param.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
    {
        param.memoryAddress = NN_SVC_ADDR_SMALL_MAP64_BEGIN;
    }

    for (uint16_t i = 1; i < MaxSvcNum; i++)
    {
        // プロセスを終了させられないので、これだけはチェックしない
        if (i == NN_SVC_ID_EXIT_PROCESS || i == NN_SVC_ID_RETURN_FROM_EXCEPTION)
        {
            continue;
        }
        SetDefaultCapability(flags, DefaultCapabilityFlagNum);
        MakeNo4Flag(&flags[DefaultCapabilityFlagNum], NN_SVC_ID_RETURN_FROM_EXCEPTION);

        flags[1] = PaddingFlag;
        int32_t flagNum;
        int maxFlagNum = heap.GetSize() / sizeof(nn::Bit32) - DefaultCapabilityFlagNum;
        int32_t syscalls[2] = { NN_SVC_ID_EXIT_PROCESS, NN_SVC_ID_RETURN_FROM_EXCEPTION };
        MakeNo4Flag(&flags[DefaultCapabilityFlagNum], &flagNum, maxFlagNum, syscalls, 2);
        if ((param.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
        {
            param.memoryAddress = NN_SVC_ADDR_SMALL_MAP64_BEGIN;
        }
        else
        {
            param.memoryAddress = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
        }
        CodeAddress = param.memoryAddress;

        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum + flagNum);
        ASSERT_RESULT_SUCCESS(result);

        uint32_t codes[] = {
            0xf100001f, // cmp r0, #0
            0x54000161u, // bne 1f
            0xd2800021, // mov r1, #1
            0xd2800002, // mov r2, #0
            0xd2800003, // mov r3, #0
            0xd2800004, // mov r4, #0
            0xd2800005, // mov r5, #0
            0xd2800006, // mov r6, #0
            0xd2800007, // mov r7, #0
            (0xd4000001 | (i << 5)), // svc #i
            0x1400000au, // b 2f
            (0xd4000001 | (NN_SVC_ID_EXIT_PROCESS << 5)), // svc #NN_SVC_ID_EXIT_PROCESS

            // 1f:
            0xd2806024, // mov r4, #0x301
            0xeb04001f, // cmp r0, r4
            0x540000c1, // bne 2f
            0xf9402c25, // ldr r5, [r1, #40]
            0x910020a5, // add r5, #8
            0xf9002c25, // str r5, [r1, #40]
            0xd2800000, // mov r0, #0
            (0xd4000001 | (NN_SVC_ID_RETURN_FROM_EXCEPTION << 5)), // svc #NN_SVC_ID_RETURN_FROM_EXCEPTION

            // 2f:
            0x14000000u, // b 2f
        };

        SetExecuteCode(handle, codes, sizeof(codes), CodeAddress, CodeSize);

        result = nn::svc::StartProcess(handle, TestLowestThreadPriority, 0, DefaultStackSize);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::ResetSignal(handle);

        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &handle, 1, WaitTime);
        ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);
    }
}
#endif

