﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/ipc/svc_SessionMessage.h>

extern nn::Result ClientLightSession(nn::svc::Handle handle);
extern nn::Result ServerLightSession(nn::svc::Handle handle);

namespace {

const size_t StackSize = 0x1000;
char g_Stack[StackSize] __attribute__((aligned(0x1000)));

void NormalSessionClientThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;

    nn::svc::Handle* clientSession = reinterpret_cast<nn::svc::Handle*>(arg);
    nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    ipcMsg.SetNull();

    result = nn::svc::SendSyncRequest(*clientSession);
    ASSERT_RESULT_SUCCESS(result);

    // LightSessionClientThread に合わせて、こちらもclientSession を閉じる
    result = nn::svc::CloseHandle(*clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

void LightSessionClientThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    nn::svc::Handle* clientSession = reinterpret_cast<nn::svc::Handle*>(arg);

    result = ClientLightSession(*clientSession);
    ASSERT_RESULT_SUCCESS(result);

    // ClientSession が閉じられることが期待されている
    result = nn::svc::CloseHandle(*clientSession);
    ASSERT_RESULT_SUCCESS(result);
}

void CheckIpc(nn::svc::Handle serverPort, nn::svc::Handle clientSession, bool isLight)
{
    nn::Result result;

    uintptr_t pc = (isLight)
        ? reinterpret_cast<uintptr_t>(LightSessionClientThread)
        : reinterpret_cast<uintptr_t>(NormalSessionClientThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + StackSize;
    uintptr_t param = reinterpret_cast<uintptr_t>(&clientSession);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = 0;

    TestThread thread(pc, param, sp, priority, idealCore);
    thread.Start();

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sSessionCloser(serverSession);

    if (isLight)
    {
        result = ServerLightSession(serverSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
    }
    else
    {
        nn::Bit32* pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        ipcMsg.SetNull();

        result = nn::svc::ReplyAndReceive(
                &index, &serverSession, 1, nn::svc::INVALID_HANDLE_VALUE, -1);
        ASSERT_RESULT_SUCCESS(result);

        ipcMsg.SetNull();

        result = nn::svc::ReplyAndReceive(
                &index, &serverSession, 0, serverSession, 0);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }

    thread.Wait();
}

void CloseServerSession(nn::svc::Handle serverPort)
{
    nn::Result result;
    int32_t index;

    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    nn::svc::Handle serverSession;
    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
}

} // namespace

TEST(ConnectToPort, MaxSessionTestForNormalSession)
{
    TestPortLeak leakTest;
    nn::Result result;

    // ポートを作成
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    nn::svc::Handle clientSession;

    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 114-20
    // CreatePort のセッション数以上のセッションを作ることが出来ない
    nn::svc::Handle tmpHandle;
    result = nn::svc::ConnectToPort(&tmpHandle, clientPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxSessions());

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 114-28
    // クライアントセッション を閉じてもサーバーセッションを閉じなければ、セッションが解放されない
    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxSessions());

    CloseServerSession(serverPort);

    // TEST 114-22
    // セッションが解放されたら、再度セッションが作れるようになる
    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    // CheckIpc の中でclientSession が閉じられる
    CheckIpc(serverPort, clientSession, false);
}

TEST(ConnectToPort, MaxSessionTestForLightSession)
{
    TestPortLeak leakTest;
    nn::Result result;

    // ポートを作成
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    nn::svc::Handle clientSession;

    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 114-21
    // CreatePort のセッション数以上のセッションを作ることが出来ない
    nn::svc::Handle tmpHandle;
    result = nn::svc::ConnectToPort(&tmpHandle, clientPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxSessions());

    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 114-29
    // クライアントセッション を閉じてもサーバーセッションを閉じなければ、セッションが解放されない
    result = nn::svc::ConnectToPort(&tmpHandle, clientPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxSessions());

    CloseServerSession(serverPort);

    // TEST 114-23
    // セッションが解放されたら、再度セッションが作れるようになる
    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    // CheckIpc の中でclientSession が閉じられる
    CheckIpc(serverPort, clientSession, true);
}

#ifdef ENABLE_MAX_HANDLE_TEST
TEST(ConnectToPort, MaxHandleTestForNormalSession)
{
    TestPortLeak leakTest;
    nn::Result result;

    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    nn::svc::Handle clientSession;

    {
        ConsumeHandle maxHandle;

        // TEST 114-24
        // ハンドル数が上限に達している状態で呼び出すと失敗する
        result = nn::svc::ConnectToPort(&clientSession, clientPort);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxHandle());
    }

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // TEST 114-26
    // ハンドル数が解放されると、呼び出しが成功する
    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    // CheckIpc の中でclientSession が閉じられる
    CheckIpc(serverPort, clientSession, false);
}

TEST(ConnectToPort, MaxHandleTestForLightSession)
{
    TestPortLeak leakTest;
    nn::Result result;

    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    nn::svc::Handle clientSession;

    {
        ConsumeHandle maxHandle;

        // TEST 114-25
        // ハンドル数が上限に達している状態で呼び出すと失敗する
        nn::svc::Handle clientSession;
        result = nn::svc::ConnectToPort(&clientSession, clientPort);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxHandle());
    }

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // TEST 114-27
    // ハンドルが解放されると、呼び出しが成功する
    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    // CheckIpc の中でclientSession が閉じられる
    CheckIpc(serverPort, clientSession, true);
}
#endif // ENABLE_MAX_HANDLE_TEST

