﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Thread.h>
#include <nn/svc/svc_Server.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_HardwareParamsSelect.h>
#include <nn/svc/svc_InterruptNameSelect.h>
#include <cstring>
#include <nn/nn_SdkAssert.h>

#if defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 ) \
    || defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 ) \
    || defined( NN_BUILD_CONFIG_HARDWARE_NX )
#include "test_TestTmrDevice_Jetson.h"
#endif


namespace {

char g_Buffer[DefaultStackSize] __attribute__((aligned(0x1000)));

void DummyThread(uintptr_t arg)
{
    NN_UNUSED(arg);
    AutoThreadExit autoExit;
}

} // namespace

/* TEST 19-1 */
/* CreateThread で作ったハンドルを閉じることが出来る */
TEST(CloseHandle, ThreadHandleTest)
{
    uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));

    nn::svc::Handle handle;
    nn::Result result;

    result = nn::svc::CreateThread(&handle, pc, 0, sp,
        TestLowestThreadPriority, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-16
    // 一度閉じたハンドルをもう一度閉じることはできない
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(CloseHandle, NamedPortHandleTest)
{
    nn::Result result;
    nn::svc::Handle namedPortHandle;
    nn::svc::Handle sessionHandle;
    nn::svc::Handle serverSession;
    nn::svc::Handle readableEvent;
    const char* PortName = "TestPort";
    size_t bufferSize = 0x1000;

    result = nn::svc::ManageNamedPort(&namedPortHandle, PortName, 1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ConnectToNamedPort(&sessionHandle, PortName);
    ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &namedPortHandle, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, namedPortHandle);
    ASSERT_RESULT_SUCCESS(result);

    nn::Bit32* pMsgBuffer = reinterpret_cast<nn::Bit32*>(g_Buffer);
    SetOnlyIpcTag(pMsgBuffer, DefaultIpcTag_Send);
    result = nn::svc::SendAsyncRequestWithUserBuffer(
            &readableEvent,
            reinterpret_cast<uintptr_t>(g_Buffer),
            bufferSize,
            sessionHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-17
    // サーバー側のセッションを閉じることが出来る
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-16
    // 一度閉じたハンドルをもう一度閉じることはできない
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 19-2
    // リクエストの処理完了の通知を受けるハンドルを閉じることが出来る
    result = nn::svc::CloseHandle(readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-16
    // 一度閉じたハンドルをもう一度閉じることはできない
    result = nn::svc::CloseHandle(readableEvent);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 19-3
    // クライアント側の名前付きポートのハンドルを閉じることが出来る
    result = nn::svc::CloseHandle(sessionHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-16
    // 一度閉じたハンドルをもう一度閉じることはできない
    result = nn::svc::CloseHandle(sessionHandle);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 19-4
    // サーバー側の名前付きポートのハンドルを閉じることが出来る
    result = nn::svc::CloseHandle(namedPortHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-16
    // 一度閉じたハンドルをもう一度閉じることはできない
    result = nn::svc::CloseHandle(namedPortHandle);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::ManageNamedPort(&namedPortHandle, PortName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(CloseHandle, PortHandleTest)
{
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;

    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverSession, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-18
    // サーバー側のセッションを閉じることが出来る
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-16
    // 一度閉じたハンドルをもう一度閉じることはできない
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 19-5
    // クライアント側ポートに対するセッションを閉じることが出来る
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-16
    // 一度閉じたハンドルをもう一度閉じることはできない
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 19-6
    // ポートを閉じることが出来る
    result = nn::svc::CloseHandle(clientPort);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-16
    // 一度閉じたハンドルをもう一度閉じることはできない
    result = nn::svc::CloseHandle(clientPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    result = nn::svc::CloseHandle(serverPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(CloseHandle, EventHandleTest)
{
    nn::Result result;
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;

    result = nn::svc::CreateEvent(&writableEvent, &readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-7
    // イベントのハンドルを閉じることが出来る
    result = nn::svc::CloseHandle(writableEvent);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-16
    // 一度閉じたハンドルをもう一度閉じることはできない
    result = nn::svc::CloseHandle(writableEvent);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    result = nn::svc::CloseHandle(readableEvent);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(CloseHandle, InterruptEventTest)
{
    nn::Result result;
    nn::svc::Handle readableEvent;

    // TODO: 割り込み番号を実装分岐する
#if defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 ) \
    || defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 ) \
    || defined( NN_BUILD_CONFIG_HARDWARE_NX )
    result = nn::svc::CreateInterruptEvent(
            &readableEvent,
            static_cast<nn::svc::Interrupt>(TmrDefaultInterruptNumber),
            nn::svc::InterruptType_Level);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-8
    // 割り込みイベントのハンドルを閉じることが出来る
    result = nn::svc::CloseHandle(readableEvent);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-16
    // 一度閉じたハンドルをもう一度閉じることはできない
    result = nn::svc::CloseHandle(readableEvent);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
#endif //NN_BUILD_CONFIG_HARDWARE_JETSONTK1
}

TEST(CloseHandle, ProcessHandleTest)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    nn::svc::Handle processHandle;
    nn::svc::Handle debugHandle;
    nn::svc::CreateProcessParameter param = {};
    nn::Bit64 processId;

    std::strncpy(param.name, "test", sizeof(param.name));
    param.version        = 0xbabeface;
    param.programId      = 0xdeadbeef0badcafeull;
    param.memoryAddress  = 0x00200000;
    param.memoryNumPages = 0x00000100;
    param.flags          = 0x00000000;

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 0x1);

    result = nn::svc::CreateProcess(&processHandle, param, flags, DefaultCapabilityFlagNum + 1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::GetProcessId(&processId, processHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::DebugActiveProcess(&debugHandle, processId);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-11
    // デバッグのハンドルを閉じることが出来る
    result = nn::svc::CloseHandle(debugHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-16
    // 一度閉じたハンドルをもう一度閉じることはできない
    result = nn::svc::CloseHandle(debugHandle);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 19-9
    // プロセスのハンドルを閉じることが出来る
    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-16
    // 一度閉じたハンドルをもう一度閉じることはできない
    result = nn::svc::CloseHandle(processHandle);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(CloseHandle, SessionHandleTest)
{
    nn::Result result;
    nn::svc::Handle serverSession;
    nn::svc::Handle clientSession;

    result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-10
    // セッションハンドルを閉じることが出来る
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-16
    // 一度閉じたハンドルをもう一度閉じることはできない
    result = nn::svc::CloseHandle(serverSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    result = nn::svc::CloseHandle(clientSession);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(CloseHandle, SharedMemoryHandleTest)
{
    nn::Result result;
    nn::svc::Handle handle;
    size_t memorySize = 0x1000;

    result = nn::svc::CreateSharedMemory(&handle, memorySize,
            nn::svc::MemoryPermission_ReadWrite, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-11
    // 共有メモリのハンドルを閉じることが出来る
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 19-16
    // 一度閉じたハンドルをもう一度閉じることはできない
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

TEST(CloseHandle, InvalidHandleTest)
{
    nn::Result result;

#if defined(NN_SDK_BUILD_RELEASE)
    // TEST 19-13
    result = nn::svc::CloseHandle(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 19-14
    result = nn::svc::CloseHandle(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
#endif // NN_SDK_BUILD_RELEASE

    // TEST 19-15
    result = nn::svc::CloseHandle(nn::svc::INVALID_HANDLE_VALUE);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}

