﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestSdmmc.h"
#include <nn/svc/svc_Dd.h>

namespace {
const uint64_t SpaceAddr = 0x0;

} // namespace

TEST(AttachDeviceAddressSpace, DeviceNameTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t size = 0x80000000;
    nn::svc::Handle handle;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, size);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    // TEST 130-1
    // 既定のデバイス名を指定することが出来る
    for (int32_t i = 0; i < nn::svc::DeviceName_Num; i++)
    {
        nn::svc::DeviceName name = static_cast<nn::svc::DeviceName>(i);
        result = nn::svc::AttachDeviceAddressSpace(name, handle);
        switch(name)
        {
            case nn::svc::DeviceName_Nv:
            case nn::svc::DeviceName_Nv2:
            case nn::svc::DeviceName_Gpu:
#if defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined (NN_BUILD_CONFIG_HARDWARE_NX)
            case nn::svc::DeviceName_Tsec1:
            case nn::svc::DeviceName_Tsecb1:
            case nn::svc::DeviceName_Nvdec1:
#endif // defined (NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined (NN_BUILD_CONFIG_HARDWARE_NX)
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultNotFound());
                continue;
            default:
                break;
        }
        ASSERT_TRUE(result.IsSuccess() || result <= nn::svc::ResultBusy());

        if (result.IsSuccess())
        {
            // TEST 131-1
            // 既定のデバイス名を指定することが出来る
            result = nn::svc::DetachDeviceAddressSpace(name, handle);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

    // TEST 130-2
    // 定義されていないデバイス名を指定することが出来ない
    result = nn::svc::AttachDeviceAddressSpace(
            static_cast<nn::svc::DeviceName>(nn::svc::DeviceName_Num), handle);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultNotFound());
}

#ifdef SUPPORT_SDCARD_TEST
TEST(AttachDeviceAddressSpace, HandleTest)
{
    TestDeviceAddressSpaceLeak leakTest;
    nn::Result result;
    uint64_t size = 0x80000000;
    nn::svc::Handle handle;

    result = nn::svc::CreateDeviceAddressSpace(&handle, SpaceAddr, size);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose addressCloser(handle);

    // TEST 130-3
    // CreateDeviceAddressSpace のハンドルを受け付ける
    result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 131-3
    // CreateDeviceAddressSpace のハンドルを受け付ける
    result = nn::svc::DetachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 130-4
    // INVALID_HANDLE_VALUE　を受け付けない
    result = nn::svc::AttachDeviceAddressSpace(
            DeviceName_SdCard, nn::svc::INVALID_HANDLE_VALUE);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 130-5
    // スレッドの擬似ハンドルを受け付けない
    result = nn::svc::AttachDeviceAddressSpace(
            DeviceName_SdCard, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 130-6
    // プロセスの擬似ハンドルを受け付けない
    result = nn::svc::AttachDeviceAddressSpace(
            DeviceName_SdCard, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 130-7
    // Close したハンドラを受け付けない
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::AttachDeviceAddressSpace(DeviceName_SdCard, handle);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
}
#endif // SUPPORT_SDCARD_TEST

