﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestMemory.h"
#include "util_TestIpc.h"
#include <nn/svc/svc_Result.h>
#include <cstring>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/util/util_BitPack.h>
#include <nn/nn_BitTypes.h>
#include <nn/os.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>

extern "C" bool NotBool(bool var);

#define ENABLE_SHOW_IPC

namespace {
const bool IpcServer = true;
const bool IpcClient = false;
const int64_t SleepTime = 100 * 1000 * 1000;
const int64_t WaitRequestTime = 2 * 1000 * 1000 * 1000ULL;

// 16 byte align されるため、1 つのポインタデータあたり最低でも16 byte 必要になる。
const size_t ThreadLocalPointerNum = 0xA;
const size_t UserBufferSize = 0x200;

const int IpcEndHeaderTag = 0xffff;
const int IpcSuccessHeaderTag = 0x1000;
const int IpcFaildHeaderTag = 0x1001;
const nn::svc::ipc::MessageBuffer::MessageHeader IpcEndHeader(IpcEndHeaderTag, 0, 0, 0, 0, 0, 0, 0);
const nn::svc::ipc::MessageBuffer::MessageHeader IpcSuccessHeader(IpcSuccessHeaderTag, 0, 0, 0, 0, 0, 0, 0);
const nn::svc::ipc::MessageBuffer::MessageHeader IpcFaildHeader(IpcFaildHeaderTag, 0, 0, 0, 0, 0, 0, 0);

char g_TestBuffer[0x1000] __attribute__((aligned(0x1000)));
char g_Buffer[DefaultStackSize] __attribute__((aligned(0x1000)));
char g_DummyBuffer[2][DefaultStackSize] __attribute__((aligned(0x1000)));
nn::Bit32 g_UserBuffer0[DefaultUserBufferSize / sizeof(nn::Bit32)] __attribute__((aligned(0x1000)));
nn::Bit32 g_UserBuffer1[DefaultUserBufferSize / sizeof(nn::Bit32)] __attribute__((aligned(0x1000)));
char g_RecvBuffer0[13][UserBufferSize];
char g_RecvBuffer1[13][UserBufferSize];
char g_XferBuffer[0x4000] __attribute__((aligned(0x1000)));
nn::svc::Handle g_TestHandles[2][0xF * 2];
nn::svc::Handle g_ServerHandle;

extern "C" void nnMain();
const char* TestPortName = "TestIpc";
const char* TestMessage[] = {
    "TEST0",
    "xTEST1",
    "xxTEST2",
    "xxxTEST3",
    "xxxxTEST4",
    "xxxxxTEST5",
    "xxxxxxTEST6",
    "xxxxxxxTEST7",
    "xxxxxxxxTEST8",
    "xxxxxxxxxTEST9",
    "xxxxxxxxxxTEST10",
    "xxxxxxxxxxxTEST11",
    "xxxxxxxxxxxxTEST12",
    "xxxxxxxxxxxxxTEST13",
    "xxxxxxxxxxxxxxTEST14"
};
uintptr_t g_HeapPtr;
const size_t g_HeapSize = (HeapAlign >= 0x10000 * 0x11 * 2)
        ? HeapAlign : ((0x10000 * 0x11 * 2 + HeapAlign - 1) & ~(HeapAlign - 1));
;

enum MapDataType
{
    MapDataType_Send,
    MapDataType_Receive,
    MapDataType_Exchange,
};

// Handleの取得のためだけのスレッド
void DummyThread()
{
    AutoThreadExit autoExit;
}

void GetIpcBuffer(bool x, nn::Bit32** ppMsgBuffer, size_t* pMsgSize, bool userBuffer)
{
    if (userBuffer)
    {
        *ppMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        *pMsgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        *ppMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        *pMsgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }
}

nn::Result SendIpc(
        nn::svc::Handle handle, nn::Bit32* pMsgBuffer, size_t msgSize, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Result tmpResult;

    BeforeClientIpc(pMsgBuffer, msgSize);
    if (userBuffer)
    {
        if (async)
        {
            nn::svc::Handle event;
            result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            if (result.IsFailure())
            {
                return result;
            }
            int32_t index;
            result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
            tmpResult = nn::svc::CloseHandle(event);
            NN_ASSERT_RESULT_SUCCESS(result);
        }
        else
        {
            result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
        }
    }
    else
    {
        result = nn::svc::SendSyncRequest(handle);
    }
    AfterClientIpc(pMsgBuffer, msgSize);
    return result;
}

nn::Result ReceiveIpc(
        nn::svc::Handle* handles, int handleNum,
        nn::Bit32* pMsgBuffer, size_t msgSize, bool userBuffer)
{
    nn::Result result;
    int32_t index;

    BeforeServerIpc(pMsgBuffer, msgSize);
    if (userBuffer)
    {
        // SendAsyncRequestWithUserBuffer から要求を受信できる
        // SendSyncRequestWithUserBuffer から要求を受信できる
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize,
                    handles, handleNum, nn::svc::INVALID_HANDLE_VALUE, -1);
    }
    else
    {
        // SendSyncRequest から要求を受信できる
        result = nn::svc::ReplyAndReceive(
                    &index, handles, handleNum, nn::svc::INVALID_HANDLE_VALUE, -1);
    }
    AfterServerIpc(pMsgBuffer, msgSize);
    return result;
}

nn::Result ReplyIpc(
        nn::svc::Handle handle, nn::Bit32* pMsgBuffer, size_t msgSize, bool userBuffer)
{
    nn::Result result;
    int32_t index;

    BeforeServerIpc(pMsgBuffer, msgSize);
    if (userBuffer)
    {
        result = nn::svc::ReplyAndReceiveWithUserBuffer(
                    &index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize,
                    &handle, 0, handle, 0);
    }
    else
    {
        result = nn::svc::ReplyAndReceive(
                    &index, &handle, 0, handle, 0);
    }
    AfterServerIpc(pMsgBuffer, msgSize);
    return result;
}

void WaitHeapState(uintptr_t heapAddr, size_t size)
{
    nn::svc::MemoryInfo info;
    for(;;)
    {
        GetMemoryInfo(&info, heapAddr);
        if (info.state == nn::svc::MemoryState_Normal &&
                info.permission == nn::svc::MemoryPermission_ReadWrite &&
                info.attribute == 0 &&
                heapAddr + size <= info.baseAddress + info.size)
        {
            break;
        }

        nn::svc::SleepThread(SleepTime);
    }
}

/* General Test */

void SpecialDataMaxNum(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("SpecialDataMaxNum");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
        uintptr_t sp = 0;

        for (int i = 0 ; i < 0xF; i++)
        {
            result = nn::svc::CreateThread(&g_TestHandles[1][i], pc, 0, sp, TestLowestThreadPriority, 0);
            ASSERT_RESULT_SUCCESS(result);
        }

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 1, 0, 0, 0, 0, 0, 0));
        offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(true, 0xF, 0));

        // 特殊データ
        // Process ID
        offset = ipcMsg.SetProcessId(offset, 0);
        // Copy Handle
        for (int i = 0; i < 0xF; i++)
        {
            offset = ipcMsg.SetHandle(offset, g_TestHandles[1][i]);
        }
        // Move Handle
        // Move Handle は Send 出来ないので、送らない

        // TEST 28-64, 29-77, 30-87
        // 最大個数の特殊データを入れた要求を送信出来る
        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);

        // Copy チェック
        {
            for (int i = 0; i < 0xF; i++)
            {
                result = nn::svc::CloseHandle(g_TestHandles[1][i]);
                ASSERT_RESULT_SUCCESS(result);
            }
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        BeforeServerIpc(pMsgBuffer, msgSize);
        // TEST 67-40, 68-87, 68-152
        // 最大個数の特殊データを受信できる
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
            ASSERT_TRUE(ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 0xF);
            ASSERT_TRUE(ipcSpecial.GetMoveHandleNum() == 0x0);

            int dataOffset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
            // Process ID
            {
                nn::Bit64 id;
                result = nn::svc::GetProcessId(&id, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(pMsgBuffer[dataOffset] == id);
                dataOffset += 2;
            }

            // Copy Handle
            {
                for (int i = 0; i < 0xF; i++)
                {
                    result = nn::svc::CloseHandle(ipcMsg.GetHandle(dataOffset++));
                    ASSERT_RESULT_SUCCESS(result);
                }
            }
        }

        // 送信データを作成
        {
            // 全ての要求を送信
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 1, 0, 0, 0, 0, 0, 0));
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(true, 0xF, 0xF));

            // 特殊データ
            uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
            uintptr_t sp = 0;

            for (int i = 0; i < 0xF * 2; i++)
            {
                result = nn::svc::CreateThread(&g_TestHandles[0][i], pc, 0, sp, TestLowestThreadPriority, 0);
                ASSERT_RESULT_SUCCESS(result);
            }

            // Process ID
            offset = ipcMsg.SetProcessId(offset, 0);

            // Copy Handle & MoveHandle
            for (int i = 0; i < 0xF * 2; i++)
            {
                offset = ipcMsg.SetHandle(offset, g_TestHandles[0][i]);
            }
        }

        BeforeServerIpc(pMsgBuffer, msgSize);
        // TEST 67-41, 68-88, 68-153
        // 最大個数の特殊データを入れた結果を送信できる
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        // Copy チェック
        {
            for (int i = 0; i < 0xF; i++)
            {
                result = nn::svc::CloseHandle(g_TestHandles[0][i]);
                ASSERT_RESULT_SUCCESS(result);
            }
        }
        // Move チェック
        {
            for (int i = 0xF; i < 0xF * 2; i++)
            {
                result = nn::svc::CloseHandle(g_TestHandles[0][i]);
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
            }
        }
    }
    if (x == IpcClient)
    {
        // TEST 28-65, 29-78, 30-88
        // 最大個数の特殊データを入れた結果を受信できる
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
            ASSERT_TRUE(ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 0xF);
            ASSERT_TRUE(ipcSpecial.GetMoveHandleNum() == 0xF);

            // Process ID
            int dataOffset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
            {
                nn::Bit64 id;
                result = nn::svc::GetProcessId(&id, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(pMsgBuffer[dataOffset] == id);
                dataOffset += 2;
            }

            // Copy Handle
            {
                for (int i = 0; i < 0xF; i++)
                {
                    result = nn::svc::CloseHandle(ipcMsg.GetHandle(dataOffset++));
                    ASSERT_RESULT_SUCCESS(result);
                }
            }

            // MoveHandle
            {
                for (int i = 0; i < 0xF; i++)
                {
                    result = nn::svc::CloseHandle(ipcMsg.GetHandle(dataOffset++));
                    ASSERT_RESULT_SUCCESS(result);
                }

            }
        }
    }

    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void PointerDataMaxNumWithBuffer(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    int32_t sendNum = userBuffer ? 0xF : ThreadLocalPointerNum;

    if (x == IpcClient)
    {
        SCOPED_TRACE("PointerDataMaxNumWithBuffer");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;
    // TEST 28-66, 29-79, 30-89
    // 最大個数のポインタデータの要求が送信できる
    if (x == IpcClient)
    {
        {
            int offset;
            // ポインタ転送にメッセージバッファを利用することが出来る
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0xFFFF, 0, sendNum, 0, 0, 0, 0, 1));

            // TLS は 0x100 しかサイズがないので、TestMessage[0] と [1] のみを使う
            for (int i = 0; i < sendNum; i++)
            {
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), i));
            }

            ASSERT_TRUE(msgSize > static_cast<size_t>(offset * sizeof(nn::Bit32)));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 1));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // TEST 67-42, 68-89, 68-154
        // 最大個数のポインタデータの要求が受信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0xFFFF);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == sendNum);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            for (int i = 0; i < sendNum; i++)
            {
                nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                ASSERT_TRUE(pointer.GetPointerIndex() == i);
                ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[0]));
                ASSERT_TRUE(reinterpret_cast<uintptr_t>(pMsgBuffer) < pointer.GetPointerAddress());
                ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(pMsgBuffer) + msgSize);
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[0], pointer.GetPointerSize()) == 0);
            }
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0000, 0, sendNum, 0, 0, 0, 0, 0));

            for (int i = 0; i < sendNum; i++)
            {
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[1], ::std::strlen(TestMessage[1]), i));
            }
        }

        // TEST 67-43, 68-90, 68-155
        // 最大個数のポインタデータの結果が送信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // TEST 28-67, 29-80, 30-90
        // 最大個数のポインタデータの結果が受信できる
        {
            // 結果を受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0000);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == sendNum);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送でデータを受信できている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            for (int i = 0; i < sendNum; i++)
            {
                nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                ASSERT_TRUE(pointer.GetPointerIndex() == i);
                ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[1]));
                ASSERT_TRUE(reinterpret_cast<uintptr_t>(pMsgBuffer) < pointer.GetPointerAddress());
                ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(pMsgBuffer) + msgSize);
                ASSERT_TRUE(::std::memcmp(
                    reinterpret_cast<void*>(pointer.GetPointerAddress()),
                    TestMessage[1],
                    pointer.GetPointerSize()) == 0);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void PointerDataMaxNumWithRecieveListBuffer(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    int32_t sendNum = userBuffer ? 0xF : ThreadLocalPointerNum;

    if (x == IpcClient)
    {
        SCOPED_TRACE("PointerDataMaxNumWithRecieveListBuffer");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // TEST 28-68, 29-81, 30-91
    // 最大個数のポインタデータの要求を送信することが出来る
    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, sendNum, 0, 0, 0, 0, 2));

        for (int i = 0; i < sendNum; i++)
        {
            // ポインタ転送
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[i], ::std::strlen(TestMessage[i]), i));
        }
        ASSERT_TRUE(msgSize > static_cast<size_t>(offset * sizeof(nn::Bit32)));


        // 受信指定リスト
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[0], sizeof(g_RecvBuffer0[0])));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 2));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[0], sizeof(g_RecvBuffer1[0])));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // TEST 67-44, 68-91, 68-156
        // 最大個数のポインタデータの要求を受信することが出来る
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == sendNum);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 2);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
                for (int i = 0; i < sendNum; i++)
                {
                    nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                    ASSERT_TRUE(pointer.GetPointerIndex() == i);
                    ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[i]));
                    ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer1[0]) <= pointer.GetPointerAddress());
                    ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(g_RecvBuffer1[0]) + sizeof(g_RecvBuffer1[0]));
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[i], pointer.GetPointerSize()) == 0);
                }
            }
        }

        // 送信データを作成
        {
            // 全ての要求を送信
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, sendNum, 0, 0, 0, 0, 0));

            // ポインタ転送
            for (int i = 0; i < sendNum; i++)
            {
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[sendNum - i - 1], ::std::strlen(TestMessage[sendNum - i - 1]), i));
            }
        }

        // TEST 67-45, 68-92, 68-157
        // 最大個数のポインタデータの結果を送信することが出来る
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // TEST 28-69, 29-82, 30-92
        // 最大個数のポインタデータの結果を受信することが出来る
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == sendNum);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);

                for (int i = 0; i < sendNum; i++)
                {
                    nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                    ASSERT_TRUE(pointer.GetPointerIndex() == i);
                    ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[sendNum - i - 1]));
                    ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer0[0]) <= pointer.GetPointerAddress());
                    ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(g_RecvBuffer0[0]) + sizeof(g_RecvBuffer0[0]));
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[sendNum - i - 1], pointer.GetPointerSize()) == 0);
                }
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void SendDataMaxNum(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("SendDataMaxNum");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // TEST 28-72, 29-85, 30-95
    // 最大個数の Send 要求を送信することが出来る
    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0xF, 0, 0, 0, 0));

        // Send 転送
        uintptr_t addr = g_HeapPtr;
        WaitHeapState(addr, 0x2000 * 5);

        ::std::memcpy(reinterpret_cast<void*>(addr), reinterpret_cast<void*>(nnMain), 0x2000 * 5);
        for (int i = 0; i < 0x5; i++) // 0x5 == 0xF / 0x3
        {
            addr = (addr + 0xFFF) & ~(0xFFF);
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(addr), 0x1000));
            addr += 0x1000;
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(addr), 0x100));
            addr += 0x100;
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(addr), 0xF0));
        }

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // TEST 67-48, 68-95, 68-160
        // 最大個数の Send 要求を受信することが出来る
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0xF);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Send
            {
                int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
                uintptr_t addr = g_HeapPtr;
                for (int i = 0; i < 0x5; i++) // 0x5 == 0xF / 0x3
                {
                    addr = (addr + 0xFFF) & ~(0xFFF);
                    nn::svc::ipc::MessageBuffer::MapData mapData0(offset + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32) * (i * 3 + 0), ipcMsg);
                    ASSERT_TRUE(mapData0.GetDataSize() == 0x1000);
                    ASSERT_TRUE(mapData0.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(addr), reinterpret_cast<void*>(mapData0.GetDataAddress()), mapData0.GetDataSize()) == 0);

                    addr += 0x1000;
                    nn::svc::ipc::MessageBuffer::MapData mapData1(offset + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32) * (i * 3 + 1), ipcMsg);
                    ASSERT_TRUE(mapData1.GetDataSize() == 0x100);
                    ASSERT_TRUE(mapData1.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(addr), reinterpret_cast<void*>(mapData1.GetDataAddress()), mapData1.GetDataSize()) == 0);

                    addr += 0x100;
                    nn::svc::ipc::MessageBuffer::MapData mapData2(offset + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32) * (i * 3 + 2), ipcMsg);
                    ASSERT_TRUE(mapData2.GetDataSize() == 0xF0);
                    ASSERT_TRUE(mapData2.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(addr), reinterpret_cast<void*>(mapData2.GetDataAddress()), mapData2.GetDataSize()) == 0);
                }
            }
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
        }

        // TEST 67-49, 68-96, 68-161
        // 最大個数の Send 結果を送信することが出来る
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // TEST 28-73, 29-86, 30-96
        // 最大個数の Send 結果を受信することが出来る
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Send
            {
                // 特になし
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void RecvDataMaxNum(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    uintptr_t nnMainAddr = reinterpret_cast<uintptr_t>(nnMain);

    if (x == IpcClient)
    {
        SCOPED_TRACE("RecvDataMaxNum");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // TEST 28-74, 29-87, 30-97
    // 最大個数のrecv要求を送信することが出来る
    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0xF, 0, 0, 0));

        // Receive 転送
        uintptr_t addr = g_HeapPtr;
        WaitHeapState(addr, 0x2000 * 5);

        ::std::memcpy(reinterpret_cast<void*>(addr), reinterpret_cast<void*>(nnMain), 0x2000 * 5);
        for (int i = 0; i < 0x5; i++) // 0x5 == 0xF / 0x3
        {
            addr = (addr + 0xFFF) & ~(0xFFF);
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(addr), 0x1000));
            addr += 0x1000;
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(addr), 0x100));
            addr += 0x100;
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(addr), 0xF0));
        }

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // TEST 67-50, 68-97, 68-162
        // 最大個数のrecv要求を受信することが出来る
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        uintptr_t buf[0xF];
        uintptr_t size[0xF];

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0xF);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Recv
            {
                int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
                for (int i = 0; i < 0x5; i++) // 0x5 == 0xF / 0x3
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData0(offset + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32) * (i * 3 + 0), ipcMsg);
                    ASSERT_TRUE(mapData0.GetDataSize() == 0x1000);
                    ASSERT_TRUE(mapData0.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    buf[i * 3 + 0] = mapData0.GetDataAddress();
                    size[i * 3 + 0] = mapData0.GetDataSize();

                    nn::svc::ipc::MessageBuffer::MapData mapData1(offset + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32) * (i * 3 + 1), ipcMsg);
                    ASSERT_TRUE(mapData1.GetDataSize() == 0x100);
                    ASSERT_TRUE(mapData1.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    buf[i * 3 + 1] = mapData1.GetDataAddress();
                    size[i * 3 + 1] = mapData1.GetDataSize();

                    nn::svc::ipc::MessageBuffer::MapData mapData2(offset + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32) * (i * 3 + 2), ipcMsg);
                    ASSERT_TRUE(mapData2.GetDataSize() == 0xF0);
                    ASSERT_TRUE(mapData2.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    buf[i * 3 + 2] = mapData2.GetDataAddress();
                    size[i * 3 + 2] = mapData2.GetDataSize();
                }
            }
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));

            // Receive 転送
            for (int i = 0; i < 0x5; i++) // 0x5 == 0xF / 0x3
            {
                ::std::memcpy(reinterpret_cast<void*>(buf[i * 3 + 0]), reinterpret_cast<char*>(nnMainAddr + 0x100), size[i * 3 + 0]);

                ::std::memcpy(reinterpret_cast<void*>(buf[i * 3 + 1]), reinterpret_cast<char*>(nnMainAddr + 0x200), size[i * 3 + 1]);

                ::std::memcpy(reinterpret_cast<void*>(buf[i * 3 + 2]), reinterpret_cast<char*>(nnMainAddr + 0x300), size[i * 3 + 2]);
            }
        }

        // TEST 67-51, 68-98, 68-163
        // 最大個数のrecv結果を送信することが出来る
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // TEST 28-75, 29-88, 30-98
        // 最大個数のrecv結果を受信することが出来る
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Recv
            {
                uintptr_t addr = g_HeapPtr;
                for (int i = 0; i < 0x5; i++) // 0x5 == 0xF / 0x3
                {
                    addr = (addr + 0xFFF) & ~(0xFFF);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(addr), reinterpret_cast<char*>(nnMainAddr + 0x100), 0x1000) == 0);
                    addr += 0x1000;
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(addr), reinterpret_cast<char*>(nnMainAddr + 0x200), 0x100) == 0);
                    addr += 0x100;
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(addr), reinterpret_cast<char*>(nnMainAddr + 0x300), 0xF0) == 0);
                }
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void ExchangeDataMaxNum(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    uintptr_t nnMainAddr = reinterpret_cast<uintptr_t>(nnMain);

    if (x == IpcClient)
    {
        SCOPED_TRACE("ExchangeDataMaxNum");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // TEST 28-76, 29-89, 30-99
    // 最大個数の exchange 要求を送信することが出来る
    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 0xF, 0, 0));

        // Exchange 転送
        uintptr_t addr = g_HeapPtr;
        WaitHeapState(addr, 0x2000 * 5);

        ::std::memcpy(reinterpret_cast<void*>(g_HeapPtr), reinterpret_cast<void*>(nnMain), 0x2000 * 5);

        for (int i = 0; i < 0x5; i++) // 0x5 == 0xF / 0x3
        {
            addr = (addr + 0xFFF) & ~(0xFFF);
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(addr), 0x1000));
            addr += 0x1000;
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(addr), 0x100));
            addr += 0x100;
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(addr), 0xF0));
        }

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // TEST 67-52, 68-99, 68-164
        // 最大個数の exchange 要求を受信することが出来る
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        uintptr_t buf[0xF];
        uintptr_t size[0xF];

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0xF);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Exchange
            {
                int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
                uintptr_t addr = g_HeapPtr;
                for (int i = 0; i < 0x5; i++) // 0x5 == 0xF / 0x3
                {
                    addr = (addr + 0xFFF) & ~(0xFFF);
                    uintptr_t dataOffset = addr - g_HeapPtr;
                    nn::svc::ipc::MessageBuffer::MapData mapData0(offset + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32) * (i * 3 + 0), ipcMsg);
                    ASSERT_TRUE(mapData0.GetDataSize() == 0x1000);
                    ASSERT_TRUE(mapData0.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<char*>(nnMain) + dataOffset, reinterpret_cast<void*>(mapData0.GetDataAddress()), mapData0.GetDataSize()) == 0);
                    buf[i * 3 + 0] = mapData0.GetDataAddress();
                    size[i * 3 + 0] = mapData0.GetDataSize();

                    addr += 0x1000;
                    dataOffset += 0x1000;
                    nn::svc::ipc::MessageBuffer::MapData mapData1(offset + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32) * (i * 3 + 1), ipcMsg);
                    ASSERT_TRUE(mapData1.GetDataSize() == 0x100);
                    ASSERT_TRUE(mapData1.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<char*>(nnMain) + dataOffset, reinterpret_cast<void*>(mapData1.GetDataAddress()), mapData1.GetDataSize()) == 0);
                    buf[i * 3 + 1] = mapData1.GetDataAddress();
                    size[i * 3 + 1] = mapData1.GetDataSize();

                    addr += 0x100;
                    dataOffset += 0x100;
                    nn::svc::ipc::MessageBuffer::MapData mapData2(offset + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32) * (i * 3 + 2), ipcMsg);
                    ASSERT_TRUE(mapData2.GetDataSize() == 0xF0);
                    ASSERT_TRUE(mapData2.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<char*>(nnMain) + dataOffset, reinterpret_cast<void*>(mapData2.GetDataAddress()), mapData2.GetDataSize()) == 0);
                    buf[i * 3 + 2] = mapData2.GetDataAddress();
                    size[i * 3 + 2] = mapData2.GetDataSize();
                }
            }
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));

            // Exchange 転送
            for (int i = 0; i < 0x5; i++) // 0x5 == 0xF / 0x3
            {
                ::std::memcpy(reinterpret_cast<void*>(buf[i * 3 + 0]), reinterpret_cast<char*>(nnMainAddr + 0x100), size[i * 3 + 0]);

                ::std::memcpy(reinterpret_cast<void*>(buf[i * 3 + 1]), reinterpret_cast<char*>(nnMainAddr + 0x200), size[i * 3 + 1]);

                ::std::memcpy(reinterpret_cast<void*>(buf[i * 3 + 2]), reinterpret_cast<char*>(nnMainAddr + 0x300), size[i * 3 + 2]);
            }
        }

        // TEST 67-53, 68-100, 68-165
        // 最大個数の exchange 結果を送信することが出来る
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // TEST 28-77, 29-90, 30-100
        // 最大個数の exchange 結果を受信することが出来る
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Exchange
            {
                uintptr_t addr = g_HeapPtr;
                for (int i = 0; i < 0x5; i++) // 0x5 == 0xF / 0x3
                {
                    addr = (addr + 0xFFF) & ~(0xFFF);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(addr), reinterpret_cast<char*>(nnMainAddr + 0x100), 0x1000) == 0);
                    addr += 0x1000;
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(addr), reinterpret_cast<char*>(nnMainAddr + 0x200), 0x100) == 0);
                    addr += 0x100;
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(addr), reinterpret_cast<char*>(nnMainAddr + 0x300), 0xF0) == 0);
                }
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void RawDataMaxNum(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    uintptr_t nnMainAddr = reinterpret_cast<uintptr_t>(nnMain);
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("RawDataMaxNum");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    int32_t maxNum = ((msgSize / sizeof(nn::Bit32) - 2) >= 1023)
        ? 1023 : (msgSize / sizeof(nn::Bit32) - 2);

    // TEST 28-78, 29-91, 30-101
    // 最大個数の任意データ要求を送信することが出来る
    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 0, maxNum, 0));

        // 任意データ
        offset = ipcMsg.SetRawArray(offset, reinterpret_cast<void*>(nnMain), maxNum * sizeof(nn::Bit32));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // TEST 67-54, 68-101, 68-166
        // 最大個数の任意データ要求を受信することが出来る
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == maxNum);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // 任意データ
            {
                int offset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);
                ASSERT_TRUE(::std::memcmp(pMsgBuffer + offset, reinterpret_cast<void*>(nnMain), sizeof(nn::Bit32) * maxNum) == 0);
            }
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, maxNum, 0));

            // 任意データ
            offset = ipcMsg.SetRawArray(offset, reinterpret_cast<void*>(nnMainAddr + 0x100), sizeof(nn::Bit32) * maxNum);
        }

        // TEST 67-55, 68-102, 68-167
        // 最大個数の任意データ結果を送信することが出来る
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // TEST 28-79, 29-92, 30-102
        // 最大個数の任意データ結果を受信することが出来る
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == maxNum);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);

            // 任意データ
            {
                int offset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);
                ASSERT_TRUE(::std::memcmp(pMsgBuffer + offset, reinterpret_cast<void*>(nnMainAddr + 0x100), sizeof(nn::Bit32) * maxNum) == 0);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void SendAndReceiveIpc(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    nn::Result result;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("SendAndReceiveIpc");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // アライン
    {
        // TEST 28-80, 29-93, 30-103
        // Send, Receive の要求を送信
        if (x == IpcClient)
        {
            ::std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), sizeof(g_XferBuffer));

            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 1, 1, 0, 0, 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer, 0x1000));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x1000, 0x1000));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == IpcServer)
        {
            // 受信バッファの準備
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
            }
            int32_t index;
            nn::svc::Handle handles[1] = { handle };

            // TEST 67-56, 68-103, 68-168
            // Send, Receive の要求を受信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);

            {
                // IPC ヘッダが受信できている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 1);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 1);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // Send
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x1000);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(g_XferBuffer, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
                }
                // Receive
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial) + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x1000);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x100, mapData.GetDataSize());
                }
            }
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
            }

            // TEST 67-57, 68-104, 68-169
            // Send, Receive の結果を送信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
        if (x == IpcClient)
        {
            // TEST 28-81, 29-94, 30-104
            // Send, Receive の結果を受信できる
            {
                // ヘッダ情報が更新されている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // データが受信できている
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x1000, reinterpret_cast<char*>(nnMain) + 0x100, 0x1000) == 0);
            }
        }
    }

    // 非アライン
    {
        // TEST 28-82, 29-95, 30-105
        // Send, Receive の要求を送信できる
        if (x == IpcClient)
        {
            ::std::memcpy(g_XferBuffer + 0x100, reinterpret_cast<void*>(nnMain), 0x1100);

            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 1, 1, 0, 0, 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x100, 0x1100));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x1200, 0x1100));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == IpcServer)
        {
            // 受信バッファの準備
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
            }
            int32_t index;
            nn::svc::Handle handles[1] = { handle };

            // TEST 67-58, 68-105, 68-170
            // Send, Receive の要求を受信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);

            {
                // IPC ヘッダが受信できている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 1);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 1);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // Send
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x1100);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x100, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
                }
                // Receive
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial) + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x1100);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x200, mapData.GetDataSize());
                }
            }
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
            }

            // TEST 67-59, 68-106, 68-171
            // Send, Receive の結果を送信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
        if (x == IpcClient)
        {
            {
                // TEST 28-83, 29-96, 30-106
                // Send, Receive の結果を受信できる
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // データが受信できている
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x1200, reinterpret_cast<char*>(nnMain) + 0x200, 0x1100) == 0);
            }
        }
    }

    // 非アライン(小)
    {
        // TEST 28-84, 29-97, 30-107
        // Send, Receive の要求を送信できる
        if (x == IpcClient)
        {
            ::std::memcpy(g_XferBuffer + 0x100, reinterpret_cast<void*>(nnMain), 0x600);

            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 1, 1, 0, 0, 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x100, 0x100));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0xf00, 0x200));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == IpcServer)
        {
            // 受信バッファの準備
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
            }
            int32_t index;
            nn::svc::Handle handles[1] = { handle };

            // TEST 67-60, 68-107, 68-172
            // Send, Receive の要求を受信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);

            {
                // IPC ヘッダが受信できている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 1);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 1);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // Send
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x100);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x100, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
                }
                // Receive
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial) + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x200);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x200, mapData.GetDataSize());
                }
            }
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
            }

            // TEST 67-61, 68-108, 68-173
            // Send, Receive の結果を送信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
        if (x == IpcClient)
        {
            // TEST 28-85, 29-98, 30-108
            // Send, Receive の結果を受信できる
            {
                // ヘッダ情報が更新されている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // データが受信できている
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0xf00, reinterpret_cast<char*>(nnMain) + 0x200, 0x200) == 0);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void ReceiveAndExchangeIpc(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("ReceiveAndExchangeIpc");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // アライン
    {
        // TEST 28-86, 29-99, 30-109
        // Receive, Exchange の要求を送信できる
        if (x == IpcClient)
        {
            ::std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), sizeof(g_XferBuffer));

            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 1, 1, 0, 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x1000, 0x1000));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x2000, 0x1000));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == IpcServer)
        {
            // 受信バッファの準備
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
            }
            int32_t index;
            nn::svc::Handle handles[1] = { handle };

            // TEST 67-62, 68-109, 68-174
            // Receive, Exchange の要求を受信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);

            {
                // IPC ヘッダが受信できている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 1);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 1);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // Recv
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x1000);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x100, mapData.GetDataSize());
                }
                // Exchange
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial) + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x1000);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<char*>(nnMain) + 0x2000, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
                    ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x200, mapData.GetDataSize());
                }
            }
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
            }

            // TEST 67-63, 68-110, 68-175
            // Receive, Exchange の結果を送信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
        if (x == IpcClient)
        {
            // TEST 28-87, 29-100, 30-110
            // Receive, Exchange の結果を受信できる
            {
                // ヘッダ情報が更新されている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // データが受信できている
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x1000, reinterpret_cast<char*>(nnMain) + 0x100, 0x1000) == 0);
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x2000, reinterpret_cast<char*>(nnMain) + 0x200, 0x1000) == 0);
            }
        }
    }

    // 非アライン
    {
        // TEST 28-88, 29-101, 30-111
        // Receive, Exchange の要求を送信できる
        if (x == IpcClient)
        {
            ::std::memcpy(g_XferBuffer + 0x100, reinterpret_cast<void*>(nnMain), 0x1100);
            ::std::memcpy(g_XferBuffer + 0x2300, reinterpret_cast<void*>(nnMain), 0x1100);

            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 1, 1, 0, 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x1200, 0x1100));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x2300, 0x1100));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == IpcServer)
        {
            // 受信バッファの準備
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
            }
            int32_t index;
            nn::svc::Handle handles[1] = { handle };

            // TEST 67-64, 68-111, 68-176
            // Receive, Exchange の要求を受信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);

            {
                // IPC ヘッダが受信できている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 1);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 1);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // Recv
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x1100);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x200, mapData.GetDataSize());
                }
                // Exchange
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial) + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x1100);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<char*>(nnMain), reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
                    ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x300, mapData.GetDataSize());
                }
            }
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
            }

            // TEST 67-65, 68-112, 68-177
            // Receive, Exchange の結果を送信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
        if (x == IpcClient)
        {
            // TEST 28-89, 29-102, 30-112
            // Receive, Exchange の結果を受信できる
            {
                // ヘッダ情報が更新されている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // データが受信できている
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x1200, reinterpret_cast<char*>(nnMain) + 0x200, 0x1100) == 0);
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x2300, reinterpret_cast<char*>(nnMain) + 0x300, 0x1100) == 0);
            }
        }
    }

    // 非アライン(小)
    {
        // TEST 28-90, 29-103, 30-113
        // Receive, Exchange の要求を送信できる
        if (x == IpcClient)
        {
            ::std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), sizeof(g_XferBuffer));

            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 1, 1, 0, 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0xf00, 0x200));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x1100, 0x300));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == IpcServer)
        {
            // 受信バッファの準備
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
            }
            int32_t index;
            nn::svc::Handle handles[1] = { handle };

            // TEST 67-66, 68-113, 68-178
            // Receive, Exchange の要求を受信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);

            {
                // IPC ヘッダが受信できている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 1);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 1);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // Receive
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x200);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x200, mapData.GetDataSize());
                }
                // Exchange
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial) + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x300);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<char*>(nnMain) + 0x1100, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
                    ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x300, mapData.GetDataSize());
                }
            }
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
            }

            // TEST 67-67, 68-114, 68-179
            // Receive, Exchange の結果を送信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
        if (x == IpcClient)
        {
            // TEST 28-91, 29-104, 30-114
            // Receive, Exchange の結果を受信できる
            {
                // ヘッダ情報が更新されている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // データが受信できている
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0xf00, reinterpret_cast<char*>(nnMain) + 0x200, 0x200) == 0);
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x1100, reinterpret_cast<char*>(nnMain) + 0x300, 0x300) == 0);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void SendAndExchange(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("SendAndExchange");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // アライン
    {
        // TEST 28-92, 29-105, 30-115
        // Send, Exchange の要求を送信できる
        if (x == IpcClient)
        {
            ::std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), sizeof(g_XferBuffer));

            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 1, 0, 1, 0, 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer, 0x1000));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x2000, 0x1000));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == IpcServer)
        {
            // 受信バッファの準備
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
            }
            int32_t index;
            nn::svc::Handle handles[1] = { handle };

            // TEST 67-68, 68-115, 68-180
            // Send, Exchange の要求を受信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);

            {
                // IPC ヘッダが受信できている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 1);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 1);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // Send
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x1000);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<char*>(nnMain), reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
                }
                // Exchange
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial) +  + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x1000);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<char*>(nnMain) + 0x2000, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
                    ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x200, mapData.GetDataSize());
                }
            }
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
            }

            // TEST 67-69, 68-116, 68-181
            // Send, Exchange の結果を送信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
        if (x == IpcClient)
        {
            // TEST 28-93, 29-106, 30-116
            // Send, Exchange の結果を受信できる
            {
                // ヘッダ情報が更新されている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // データが受信できている
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x2000, reinterpret_cast<char*>(nnMain) + 0x200, 0x1000) == 0);
            }
        }
    }

    // 非アライン
    {
        // TEST 28-94, 29-107, 30-117
        // Send, Exchange の要求を送信できる
        if (x == IpcClient)
        {
            ::std::memcpy(g_XferBuffer + 0x100, reinterpret_cast<void*>(nnMain), 0x1100);
            ::std::memcpy(g_XferBuffer + 0x2300, reinterpret_cast<void*>(nnMain), 0x1100);

            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 1, 0, 1, 0, 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x100, 0x1100));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x2300, 0x1100));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == IpcServer)
        {
            // 受信バッファの準備
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
            }
            int32_t index;
            nn::svc::Handle handles[1] = { handle };

            // TEST 67-70, 68-117, 68-182
            // Send, Exchange の要求を受信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);

            {
                // IPC ヘッダが受信できている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 1);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 1);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // Send
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x1100);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<char*>(nnMain), reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
                }
                // Exchange
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial) + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x1100);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<char*>(nnMain), reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
                    ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x300, mapData.GetDataSize());
                }
            }
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
            }

            // TEST 67-71, 68-118, 68-183
            // Send, Exchange の結果を送信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
        if (x == IpcClient)
        {
            // TEST 28-95, 29-108, 30-118
            // Send, Exchange の結果を受信できる
            {
                // ヘッダ情報が更新されている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // データが受信できている
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x2300, reinterpret_cast<char*>(nnMain) + 0x300, 0x1100) == 0);
            }
        }
    }

    // 非アライン(小)
    {
        // TEST 28-96, 29-109, 30-119
        // Send, Exchange の要求を送信できる
        if (x == IpcClient)
        {
            ::std::memcpy(g_XferBuffer + 0x100, reinterpret_cast<void*>(nnMain), 0x600);

            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 1, 0, 1, 0, 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x100, 0x100));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x1100, 0x300));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == IpcServer)
        {
            // 受信バッファの準備
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
            }
            int32_t index;
            nn::svc::Handle handles[1] = { handle };

            // TEST 67-72, 68-119, 68-184
            // Send, Exchange の要求を受信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);

            {
                // IPC ヘッダが受信できている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 1);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 1);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // Send
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x100);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<char*>(g_XferBuffer) + 0x100, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
                }
                // Exchange
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial) + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x300);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    ASSERT_TRUE(::std::memcmp(reinterpret_cast<char*>(g_XferBuffer) + 0x1100, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
                    ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMain) + 0x300, mapData.GetDataSize());
                }
            }
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
            }

            // TEST 67-73, 68-120, 68-185
            // Send, Exchange の結果を送信できる
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
        if (x == IpcClient)
        {
            // TEST 28-97, 29-110, 30-120
            // Send, Exchange の結果を受信できる
            {
                // ヘッダ情報が更新されている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // データが受信できている
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x1100, reinterpret_cast<char*>(nnMain) + 0x300, 0x300) == 0);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void SendAll(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    uintptr_t nnMainAddr = reinterpret_cast<uintptr_t>(nnMain);
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("SendAll");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // TEST 28-98, 29-111, 30-121
    // 全ての要求を一度に送信できる
    if (x == IpcClient)
    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
        uintptr_t sp[2];
        nn::svc::Handle threads[2];
        sp[0] = reinterpret_cast<uintptr_t>(&g_DummyBuffer[0]) + sizeof(g_DummyBuffer[0]);

        result = nn::svc::CreateThread(&threads[0], pc, 0, sp[0], TestLowestThreadPriority, 0);
        ASSERT_RESULT_SUCCESS(result);

        ::std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), sizeof(g_XferBuffer));

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 1, 1, 1, 1, 1, 1, 3));
        offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(true, 1, 0));

        // 特殊データ
        // Process ID
        offset = ipcMsg.SetProcessId(offset, 0);
        // Copy Handle
        offset = ipcMsg.SetHandle(offset, threads[0]);
        // Move Handle
        // Send は Move Handle 出来ないので送らない

        // ポインタ転送
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));

        // Send 転送
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer, 0x1000));

        // Receive 転送
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x1000, 0x1000));

        // Exchange 転送
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x2000, 0x1000));

        // 任意データ
        offset = ipcMsg.SetRawArray(offset, reinterpret_cast<void*>(nnMain), sizeof(nn::Bit32));

        // 受信指定リスト
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[0], sizeof(g_RecvBuffer0[0])));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);

        // Copy チェック
        {
            result = nn::svc::CloseHandle(threads[0]);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 3));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[0], sizeof(g_RecvBuffer1[0])));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // TEST 67-74, 68-121, 68-186
        // 全ての要求を一度に受信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        nn::svc::Handle threads[2];

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 1);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 1);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 1);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 1);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 3);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
            ASSERT_TRUE(ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 1);
            ASSERT_TRUE(ipcSpecial.GetMoveHandleNum() == 0);

            // Process ID
            {
                nn::Bit64 id;
                result = nn::svc::GetProcessId(&id, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(pMsgBuffer[nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial)] == id);
            }

            // Copy Handle
            {
                int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
                threads[0] = ipcMsg.GetHandle(offset + 2);

                result = nn::svc::StartThread(threads[0]);
                ASSERT_RESULT_SUCCESS(result);

                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &threads[0], 1, -1);
                ASSERT_RESULT_SUCCESS(result);

                result = nn::svc::CloseHandle(threads[0]);
                ASSERT_RESULT_SUCCESS(result);
            }

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
                nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);

                ASSERT_TRUE(pointer.GetPointerIndex() == 0);
                ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[0]));
                ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer1[0]) <= pointer.GetPointerAddress());
                ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(g_RecvBuffer1[0]) + sizeof(g_RecvBuffer1[0]));
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[0], pointer.GetPointerSize()) == 0);
            }

            // Send
            {
                nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
                ASSERT_TRUE(mapData.GetDataSize() == 0x1000);
                ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<char*>(nnMain), reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
            }

            // Recv
            {
                // 送信データ作成
                nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial) + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32), ipcMsg);
                ASSERT_TRUE(mapData.GetDataSize() == 0x1000);
                ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMainAddr + 0x100), mapData.GetDataSize());
            }

            // Exchange
            {
                nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial) + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32) * 2, ipcMsg);
                ASSERT_TRUE(mapData.GetDataSize() == 0x1000);
                ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<char*>(nnMain) + 0x2000, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);

                // 送信データ作成
                ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMainAddr + 0x200), mapData.GetDataSize());
            }

            // 任意データ
            {
                int offset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);
                ASSERT_TRUE(::std::memcmp(pMsgBuffer + offset, reinterpret_cast<void*>(nnMain), sizeof(nn::Bit32)) == 0);
            }
        }

        // 送信データを作成
        {
            // 全ての要求を送信
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 1, 1, 0, 0, 0, 1, 0));
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(true, 1, 1));

            // 特殊データ
            uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
            uintptr_t sp[2];
            sp[0] = reinterpret_cast<uintptr_t>(&g_DummyBuffer[0]) + sizeof(g_DummyBuffer[0]);
            sp[1] = reinterpret_cast<uintptr_t>(&g_DummyBuffer[1]) + sizeof(g_DummyBuffer[1]);

            result = nn::svc::CreateThread(&threads[0], pc, 0, sp[0], TestLowestThreadPriority, 0);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::CreateThread(&threads[1], pc, 0, sp[1], TestLowestThreadPriority, 0);
            ASSERT_RESULT_SUCCESS(result);


            // Process ID
            offset = ipcMsg.SetProcessId(offset, 0);
            // Copy Handle
            offset = ipcMsg.SetHandle(offset, threads[0]);
            // Move Handle
            offset = ipcMsg.SetHandle(offset, threads[1]);

            // ポインタ転送（受信指定リスト）
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[12], ::std::strlen(TestMessage[12]), 0));

            // Send 転送

            // Receive 転送

            // Exchange 転送

            // 任意データ
            offset = ipcMsg.SetRawArray(offset, reinterpret_cast<void*>(nnMainAddr + 0x100), sizeof(nn::Bit32));
        }

        // TEST 67-75, 68-122, 68-187
        // 全ての結果を一度に送信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        // Copy チェック
        {
            result = nn::svc::CloseHandle(threads[0]);
            ASSERT_RESULT_SUCCESS(result);
        }
        // Move チェック
        {
            result = nn::svc::CloseHandle(threads[1]);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
        }
    }
    if (x == IpcClient)
    {
        // TEST 28-99, 29-112, 30-122
        // 全ての結果を一度に受信できる
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 1);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
            ASSERT_TRUE(ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 1);
            ASSERT_TRUE(ipcSpecial.GetMoveHandleNum() == 1);

            // Process ID
            {
                nn::Bit64 id;
                result = nn::svc::GetProcessId(&id, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(pMsgBuffer[nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial)] == id);
            }

            // Copy Handle
            {
                int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);

                // Process ID は 2個分のデータを使う
                result = nn::svc::CloseHandle(ipcMsg.GetHandle(offset + 2));
                ASSERT_RESULT_SUCCESS(result);
            }

            // MoveHandle
            {
                int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);

                result = nn::svc::CloseHandle(ipcMsg.GetHandle(offset + 3));
                ASSERT_RESULT_SUCCESS(result);
            }

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
                nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);

                ASSERT_TRUE(pointer.GetPointerIndex() == 0);
                ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[12]));
                ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer0[0]) <= pointer.GetPointerAddress());
                ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(g_RecvBuffer0[0]) + sizeof(g_RecvBuffer0[0]));
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[12], pointer.GetPointerSize()) == 0);
            }

            // Send
            {
                // 特になし
            }

            // Recv
            {
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x1000, reinterpret_cast<char*>(nnMain) + 0x100, 0x1000) == 0);
            }

            // Exchange
            {
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x2000, reinterpret_cast<char*>(nnMain) + 0x200, 0x1000) == 0);
            }

            // 任意データ
            {
                int offset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);
                ASSERT_TRUE(::std::memcmp(pMsgBuffer + offset, reinterpret_cast<void*>(nnMainAddr + 0x100), sizeof(nn::Bit32)) == 0);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

/* TEST 67-76, 68-123, 68-188 */
/*
   アライメントされていないアドレスを用いて recv 転送したときに、
   クライアントがマップ領域に書き込んだデータが書き戻されない
 */
void UnAlignedMapIsCopyWithRecvRequest(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    nn::Result result;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("UnAlignedMapIsCopyWithRecvRequest");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // 非アライン
    {
        if (x == IpcClient)
        {
            // マップ元にデータを書き込んでおく
            ::std::memcpy(g_XferBuffer + 0x100, reinterpret_cast<void*>(nnMain), 0x1100);

            // Recv の要求を送信
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 1, 0, 0, 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x100, 0x1100));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == IpcServer)
        {
            // 受信バッファの準備
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
            }
            int32_t index;
            nn::svc::Handle handles[1] = { handle };
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);

            uintptr_t buf;
            uintptr_t size;
            {
                // IPC ヘッダが受信できている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 1);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // Recv
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x1100);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    // マップされている領域にデータが書き戻されていない
                    ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x100, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) != 0);

                    // マップされている領域にデータを書き込む
                    buf = mapData.GetDataAddress();
                    size = mapData.GetDataSize();
                    ::std::memcpy(
                        reinterpret_cast<void*>(buf),
                        reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(nnMain) + 0x100),
                        size);
                }
            }
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
            }
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                // 相手に reply を返すことが出来る
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                // 相手に reply を返すことが出来る
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
        if (x == IpcClient)
        {
            {
                // ヘッダ情報が更新されている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // データが受信されているか確認
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x100, reinterpret_cast<char*>(nnMain) + 0x100, 0x1100) == 0);
            }
        }
    }

    // 非アライン(小)
    {
        if (x == IpcClient)
        {
            ::std::memcpy(g_XferBuffer + 0x100, reinterpret_cast<void*>(nnMain), 0x100);

            // Recv の要求を送信
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 1, 0, 0, 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x100, 0x100));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == IpcServer)
        {
            // 受信バッファの準備
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
            }
            int32_t index;
            nn::svc::Handle handles[1] = { handle };
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                // SendAsyncRequestWithUserBuffer から要求を受信できる
                // SendSyncRequestWithUserBuffer から要求を受信できる
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            else
            {
                // SendSyncRequest から要求を受信できる
                result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);

            uintptr_t buf;
            uintptr_t size;
            {
                // IPC ヘッダが受信できている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 1);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // Recv
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0x100);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                    // マップされている領域にデータが書き戻されていない
                    ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x100, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) != 0);

                    // マップされている領域にデータを書き込む
                    buf = mapData.GetDataAddress();
                    size = mapData.GetDataSize();
                    ::std::memcpy(
                        reinterpret_cast<void*>(buf),
                        reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(nnMain) + 0x100),
                        size);
                }
            }
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
            }
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                // 相手に reply を返すことが出来る
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                // 相手に reply を返すことが出来る
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
        if (x == IpcClient)
        {
            {
                // ヘッダ情報が更新されている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x100, reinterpret_cast<char*>(nnMain) + 0x100, 0x100) == 0);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void UseCopiedHandle(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("UseCopiedHandle");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(&g_DummyBuffer[0]) + sizeof(g_DummyBuffer[0]);
        nn::svc::Handle thread;

        result = nn::svc::CreateThread(&thread, pc, 0, sp, TestLowestThreadPriority, 0);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 1, 0, 0, 0, 0, 0, 0));
        offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 1, 0));

        // 特殊データ
        // Copy Handle
        offset = ipcMsg.SetHandle(offset, thread);

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);

        // TEST 28-101, 29-114, 30-124
        // Copy したハンドルが使える
        {
            result = nn::svc::CloseHandle(thread);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
            ASSERT_TRUE(!ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 1);
            ASSERT_TRUE(ipcSpecial.GetMoveHandleNum() == 0);

            // Copy Handle
            {
                int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
                nn::svc::Handle thread;
                thread = ipcMsg.GetHandle(offset);

                result = nn::svc::StartThread(thread);
                ASSERT_RESULT_SUCCESS(result);

                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &thread, 1, -1);
                ASSERT_RESULT_SUCCESS(result);

                result = nn::svc::CloseHandle(thread);
                ASSERT_RESULT_SUCCESS(result);
            }
        }

        nn::svc::Handle newThread;

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
            uintptr_t sp = reinterpret_cast<uintptr_t>(&g_DummyBuffer[1]) + sizeof(g_DummyBuffer[1]);

            result = nn::svc::CreateThread(&newThread, pc, 0, sp, TestLowestThreadPriority, 0);
            ASSERT_RESULT_SUCCESS(result);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 1, 0, 0, 0, 0, 0, 0));
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 1, 0));

            // Copy Handle
            offset = ipcMsg.SetHandle(offset, newThread);
        }

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        // TEST 67-77, 68-124, 68-189
        // Copy したハンドルが使える
        {
            result = nn::svc::CloseHandle(newThread);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
    if (x == IpcClient)
    {
        // 受信データの確認
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
            ASSERT_TRUE(!ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 1);
            ASSERT_TRUE(ipcSpecial.GetMoveHandleNum() == 0);

            // Copy Handle
            {
                int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
                nn::svc::Handle thread = ipcMsg.GetHandle(offset);

                result = nn::svc::StartThread(thread);
                ASSERT_RESULT_SUCCESS(result);

                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &thread, 1, -1);
                ASSERT_RESULT_SUCCESS(result);

                result = nn::svc::CloseHandle(thread);
                ASSERT_RESULT_SUCCESS(result);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void UseMovedHandle(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("UseMovedHandle");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
        uintptr_t sp = reinterpret_cast<uintptr_t>(&g_DummyBuffer[0]) + sizeof(g_DummyBuffer[0]);
        nn::svc::Handle thread;

        result = nn::svc::CreateThread(&thread, pc, 0, sp, TestLowestThreadPriority, 0);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 1, 0, 0, 0, 0, 0, 0));
        offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 0, 1));

        // 特殊データ
        // Move Handle
        offset = ipcMsg.SetHandle(offset, thread);

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
                ASSERT_RESULT_SUCCESS(result);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);

        if (async)
        {
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
        }

        result = nn::svc::CloseHandle(thread);
        ASSERT_RESULT_SUCCESS(result);

        // 何もデータを付けずに送信
        ipcMsg.SetNull();
        offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 0, 0, 0));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);
        }

        nn::svc::Handle newThread;

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
            uintptr_t sp = reinterpret_cast<uintptr_t>(&g_DummyBuffer[1]) + sizeof(g_DummyBuffer[1]);

            result = nn::svc::CreateThread(&newThread, pc, 0, sp, TestLowestThreadPriority, 0);
            ASSERT_RESULT_SUCCESS(result);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 1, 0, 0, 0, 0, 0, 0));
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 0, 1));

            // Move Handle
            offset = ipcMsg.SetHandle(offset, newThread);
        }

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            // 相手に reply を返すことが出来る
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        // TEST 67-78, 68-125, 68-190
        // Move したハンドルは使えない
        {
            result = nn::svc::CloseHandle(newThread);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
        }
    }
    if (x == IpcClient)
    {
        // 受信データの確認
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
            ASSERT_TRUE(!ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 0);
            ASSERT_TRUE(ipcSpecial.GetMoveHandleNum() == 1);

            // Move Handle
            {
                int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
                nn::svc::Handle thread = ipcMsg.GetHandle(offset);

                result = nn::svc::StartThread(thread);
                ASSERT_RESULT_SUCCESS(result);

                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &thread, 1, -1);
                ASSERT_RESULT_SUCCESS(result);

                result = nn::svc::CloseHandle(thread);
                ASSERT_RESULT_SUCCESS(result);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void CopyInvalidHandle(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("CopyInvalidHandle");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // TEST 28-103, 29-116, 30-126
    // 無効なハンドルコピー要求を送信できる
    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 1, 0, 0, 0, 0, 0, 0));
        offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 1, 0));

        // 特殊データ
        // Copy Handle
        offset = ipcMsg.SetHandle(offset, nn::svc::INVALID_HANDLE_VALUE);

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // TEST 67-79, 68-126, 68-191
        // 無効なハンドルコピー要求を受信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
            ASSERT_TRUE(!ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 1);
            ASSERT_TRUE(ipcSpecial.GetMoveHandleNum() == 0);

            // Copy Handle
            {
                int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
                result = nn::svc::CloseHandle(ipcMsg.GetHandle(offset));
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
            }
        }

        nn::svc::Handle newThread;

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 1, 0, 0, 0, 0, 0, 0));
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 1, 0));

            // Copy Handle
            offset = ipcMsg.SetHandle(offset, nn::svc::INVALID_HANDLE_VALUE);
        }

        // TEST 67-80, 68-127, 68-192
        // 無効なハンドルコピー結果を送信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // TEST 28-104, 29-117, 30-127
        // 無効なハンドルコピー結果を受信できる
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
            ASSERT_TRUE(!ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 1);
            ASSERT_TRUE(ipcSpecial.GetMoveHandleNum() == 0);

            // Copy Handle
            {
                int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
                result = nn::svc::CloseHandle(ipcMsg.GetHandle(offset));
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void MoveInvalidHandle(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("MoveInvalidHandle");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // TEST 28-105, 29-118, 30-128
    // 無効なハンドルに対してムーブ要求を送信することができる
    // ムーブ要求は出せないので、廃止
    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 0, 0, 0));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // TEST 67-81, 68-128, 68-193
        // 無効なハンドルに対してムーブ要求を受信することができる
        // 廃止
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 1, 0, 0, 0, 0, 0, 0));
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 0, 1));

            // Move Handle
            offset = ipcMsg.SetHandle(offset, nn::svc::INVALID_HANDLE_VALUE);
        }

        // TEST 67-82, 68-129, 68-194
        // 無効なハンドルに対してムーブ結果を送信することができる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // TEST 28-106, 29-119, 30-129
        // 無効なハンドルに対してムーブ結果を受信することができる
        // 廃止
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
            ASSERT_TRUE(!ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 0);
            ASSERT_TRUE(ipcSpecial.GetMoveHandleNum() == 1);

            int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
            nn::svc::Handle invalidHandle = ipcMsg.GetHandle(offset);
            ASSERT_TRUE(!invalidHandle.IsValid());
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// TEST 901-40
// 擬似スレッドのハンドルをコピー出来る
void CopyPseudoThreadHanldle(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("CopyPseudoThreadHanldle");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // 擬似スレッドハンドルのコピー要求を送信できる
    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 1, 0, 0, 0, 0, 0, 0));
        offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 1, 0));

        // 特殊データ
        // Copy Handle
        offset = ipcMsg.SetHandle(offset, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // 擬似スレッドハンドルのコピー要求を受信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
            ASSERT_TRUE(!ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 1);
            ASSERT_TRUE(ipcSpecial.GetMoveHandleNum() == 0);

            // Copy Handle
            {
                int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
                result = nn::svc::CloseHandle(ipcMsg.GetHandle(offset));
                ASSERT_RESULT_SUCCESS(result);
            }
        }

        nn::svc::Handle newThread;

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 1, 0, 0, 0, 0, 0, 0));
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 1, 0));

            // Copy Handle
            offset = ipcMsg.SetHandle(offset, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
        }

        // 擬似スレッドハンドルのコピー結果を送信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // 擬似ハンドルコピー結果を受信できる
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
            ASSERT_TRUE(!ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 1);
            ASSERT_TRUE(ipcSpecial.GetMoveHandleNum() == 0);

            // Copy Handle
            {
                int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
                result = nn::svc::CloseHandle(ipcMsg.GetHandle(offset));
                ASSERT_RESULT_SUCCESS(result);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// TEST 901-41
// 擬似プロセスのハンドルをコピー出来る
void CopyPseudoProcessHanldle(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("CopyPseudoProcessHanldle");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // 擬似プロセスハンドルのコピー要求を送信できる
    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 1, 0, 0, 0, 0, 0, 0));
        offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 1, 0));

        // 特殊データ
        // Copy Handle
        offset = ipcMsg.SetHandle(offset, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // 擬似スレッドハンドルのコピー要求を受信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
            ASSERT_TRUE(!ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 1);
            ASSERT_TRUE(ipcSpecial.GetMoveHandleNum() == 0);

            // Copy Handle
            {
                int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
                result = nn::svc::CloseHandle(ipcMsg.GetHandle(offset));
                ASSERT_RESULT_SUCCESS(result);
            }
        }

        nn::svc::Handle newThread;

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 1, 0, 0, 0, 0, 0, 0));
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 1, 0));

            // Copy Handle
            offset = ipcMsg.SetHandle(offset, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
        }

        // 擬似プロセスハンドルのコピー結果を送信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // 擬似ハンドルコピー結果を受信できる
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
            ASSERT_TRUE(!ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 1);
            ASSERT_TRUE(ipcSpecial.GetMoveHandleNum() == 0);

            // Copy Handle
            {
                int offset = ipcMsg.GetSpecialDataOffset(ipcHeader, ipcSpecial);
                result = nn::svc::CloseHandle(ipcMsg.GetHandle(offset));
                ASSERT_RESULT_SUCCESS(result);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// TEST 901-42
// 擬似スレッドのハンドルをムーブ出来ない
void MovePseudoThreadHanldle(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("MovePseudoThreadHanldle");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // 擬似スレッドハンドルのムーブ要求を送信できる
    // ムーブ要求を送信できないので、廃止
    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 0, 0, 0));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
                ASSERT_RESULT_SUCCESS(result);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        if (async)
        {
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // 擬似スレッドハンドルのムーブ要求を受信できる
        // 廃止
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 1, 0, 0, 0, 0, 0, 0));
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 0, 1));

            // Move Handle
            offset = ipcMsg.SetHandle(offset, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
        }

        // 擬似スレッドハンドルのムーブ結果を送信できない
        // クライアント側がエラーになる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // 擬似ハンドルのムーブ結果を受信できる
        // 廃止
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// TEST 901-43
// 擬似プロセスのハンドルをムーブ出来ない
void MovePseudoProcessHanldle(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("MovePseudoProcessHanldle");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // 擬似プロセスハンドルのムーブ要求を送信できる
    // 廃止
    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 0, 0, 0));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
                ASSERT_RESULT_SUCCESS(result);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);

        if (async)
        {
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // 擬似プロセスハンドルのムーブ要求を受信できる
        // 廃止
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 1, 0, 0, 0, 0, 0, 0));
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 0, 1));

            // Move Handle
            offset = ipcMsg.SetHandle(offset, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
        }

        // 擬似プロセスハンドルのムーブ結果を送信できない
        // クライアント側がエラーになる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // 擬似プロセスハンドルムーブ結果を受信できる
        // 廃止
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

/* TEST 67-83, 68-130, 68-195 */
/*
   ポインタ転送 (バッファ利用) の要求を受けて、任意データを結果として返すことができる
 */
void ReplyDifferentInfoFromPointerToRaw(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("ReplyDifferentInfoFromPointerToRaw");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;
    if (x == IpcClient)
    {
        {
            int offset;
            // ポインタ転送にメッセージバッファを利用することが出来る
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0xFFFF, 0, 1, 0, 0, 0, 0, 1));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 1));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0xFFFF);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);
            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[0]));
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(pMsgBuffer) < pointer.GetPointerAddress());
            ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(pMsgBuffer) + msgSize);
            ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[0], pointer.GetPointerSize()) == 0);
        }
        {
            // 要求とは違った情報を送る
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0000, 0, 0, 0, 0, 0, 1, 0));

            // 任意データ
            offset = ipcMsg.SetRawArray(offset, reinterpret_cast<void*>(nnMain), sizeof(nn::Bit32));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        {
            // 結果を受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0000);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 1);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // 任意データ
            {
                int offset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);
                ASSERT_TRUE(::std::memcmp(pMsgBuffer + offset, reinterpret_cast<void*>(nnMain), sizeof(nn::Bit32)) == 0);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

/* TEST 67-84, 68-131, 68-196 */
/*
   reply時のマップ転送の情報は無視される
 */
// 仕様変更により削除

/* TEST 67-85, 68-132, 68-197 */
/*
   任意データの要求に対して、任意データと特殊データを返すことが出来る
 */
void ReplyDifferentInfoFromRawToSpecial(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    uintptr_t nnMainAddr = reinterpret_cast<uintptr_t>(nnMain);
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("ReplyDifferentInfoFromRawToSpecial");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 0, 1, 0));

        // 任意データ
        offset = ipcMsg.SetRawArray(offset, reinterpret_cast<void*>(nnMain), sizeof(nn::Bit32));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 1);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // 任意データ
            {
                int offset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);
                ASSERT_TRUE(::std::memcmp(pMsgBuffer + offset, reinterpret_cast<void*>(nnMain), sizeof(nn::Bit32)) == 0);
            }
        }

        // 送信データを作成
        {
            // 全ての要求を送信
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 1, 0, 0, 0, 0, 1, 0));
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(true, 0, 0));

            // 特殊データ
            // Process ID
            offset = ipcMsg.SetProcessId(offset, 0);

            // 任意データ
            offset = ipcMsg.SetRawArray(offset, reinterpret_cast<void*>(nnMainAddr + 0x100), sizeof(nn::Bit32));
        }

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // 受信データの確認
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 1);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 1);
            ASSERT_TRUE(ipcSpecial.GetProcessIdFlag());
            ASSERT_TRUE(ipcSpecial.GetCopyHandleNum() == 0);
            ASSERT_TRUE(ipcSpecial.GetMoveHandleNum() == 0);

            // Process ID
            {
                nn::Bit64 id;
                result = nn::svc::GetProcessId(&id, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
                ASSERT_RESULT_SUCCESS(result);
                ASSERT_TRUE(pMsgBuffer[nn::svc::ipc::MessageBuffer::GetSpecialDataOffset(ipcHeader, ipcSpecial)] == id);
            }

            // 任意データ
            {
                int offset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);
                ASSERT_TRUE(::std::memcmp(pMsgBuffer + offset, reinterpret_cast<void*>(nnMainAddr + 0x100), sizeof(nn::Bit32)) == 0);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

/* TEST 67-86, 68-133, 68-198 */
/*
   マップ転送のexchange 要求を受けて、exchange データに加え、
   ポインタ転送(受信指定リスト)を返すことが出来る
 */
void ReplyDifferentInfoFromExchangeToPointer(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    uintptr_t nnMainAddr = reinterpret_cast<uintptr_t>(nnMain);
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("ReplyDifferentInfoFromExchangeToPointer");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        ::std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), sizeof(g_XferBuffer));

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 1, 0, 3));

        // Exchange 転送
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x2000, 0x1000));

        // 受信指定リスト
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[0], sizeof(g_RecvBuffer0[0])));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 1);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 3);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Exchange
            {
                nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
                ASSERT_TRUE(mapData.GetDataSize() == 0x1000);
                ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<char*>(nnMain) + 0x2000, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);

                // 送信データ作成
                ::std::memcpy(reinterpret_cast<void*>(mapData.GetDataAddress()), reinterpret_cast<char*>(nnMainAddr + 0x200), mapData.GetDataSize());
            }
        }

        // 送信データを作成
        {
            // 全ての要求を送信
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 1, 0, 0, 0, 0, 0));

            // ポインタ転送（受信指定リスト）
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[12], ::std::strlen(TestMessage[12]), 0));

            // Exchange 転送
        }

        // 相手に reply を返すことが出来る
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // 受信データの確認
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
                nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);

                ASSERT_TRUE(pointer.GetPointerIndex() == 0);
                ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[12]));
                ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer0[0]) <= pointer.GetPointerAddress());
                ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(g_RecvBuffer0[0]) + sizeof(g_RecvBuffer0[0]));
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[12], pointer.GetPointerSize()) == 0);
            }

            // Exchange
            {
                ASSERT_TRUE(::std::memcmp(g_XferBuffer + 0x2000, reinterpret_cast<char*>(nnMain) + 0x200, 0x1000) == 0);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

/* TEST 67-87, 68-134, 68-199 */
/*
   send 要求に対して、マップ先のアドレスを変えて、結果を送信することはできない
 */
// 仕様変更により廃止

/* TEST 67-88, 68-135, 68-200 */
/*
   recv 要求に対して、マップ先のアドレスを変えて、結果を送信しても無視される
 */
// 仕様変更により廃止

/* TEST 67-89, 68-136, 68-201 */
/*
   exchange 要求に対して、マップ先のアドレスを変えて、結果を送信しても無視される
 */
// 仕様変更により廃止

void ClientReceiveListWithInvalidMememoryPermmision(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("ClientReceiveListWithInvalidMememoryPermmision");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // TEST 28-107, 29-120, 30-130
    // MemoryPermission_None を受信指定リストに指定した要求送信は失敗する
    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0xFFFF, 0, 1, 0, 0, 0, 0, 3));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));

            // 受信指定リスト
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(reinterpret_cast<char*>(g_FreeAreaBegin), ::std::strlen(TestMessage[1])));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {

                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {

                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {

                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                // MemoryPermission_None に結果を書き込めないので、失敗する
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
            }
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 1));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {

            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0xFFFF);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 3);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);
            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[0]));
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(pMsgBuffer) < pointer.GetPointerAddress());
            ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(pMsgBuffer) + msgSize);
            ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[0], pointer.GetPointerSize()) == 0);
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0000, 0, 1, 0, 0, 0, 0, 0));

            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[1], ::std::strlen(TestMessage[1]), 0));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        // MemoryPermission_None に対して、書き込むことが出来ない
        ASSERT_TRUE(result.IsSuccess() ||
                result <= nn::svc::ResultTimeout() ||
                result <= nn::svc::ResultReceptionSpecificationListBroken());
    }
    if (x == IpcClient)
    {
        {
            // 受信に失敗しているので、受信確認はない
        }
    }

    // TEST 28-108, 29-121, 30-131
    // MemoryPermission_Read  を受信指定リストに指定した要求送信は失敗する
    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0xFFFF, 0, 1, 0, 0, 0, 0, 3));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));

            // 受信指定リスト
            // MemoryPermission_Read
            WaitHeapState(g_HeapPtr, 0x1000);
            result = nn::svc::SetMemoryPermission(g_HeapPtr, 0x1000, nn::svc::MemoryPermission_Read);
            ASSERT_RESULT_SUCCESS(result);
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(reinterpret_cast<char*>(g_HeapPtr), 0x1000));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {

                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {

                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {

                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);

            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                // MemoryPermission_Read に結果を書き込めないので、失敗する
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
            }

            result = nn::svc::SetMemoryPermission(g_HeapPtr, 0x1000, nn::svc::MemoryPermission_ReadWrite);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 1));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0xFFFF);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 3);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);
            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[0]));
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(pMsgBuffer) < pointer.GetPointerAddress());
            ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(pMsgBuffer) + msgSize);
            ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[0], pointer.GetPointerSize()) == 0);
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0000, 0, 1, 0, 0, 0, 0, 0));

            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[1], ::std::strlen(TestMessage[1]), 0));
        }
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        // MemoryPermission_Read に対して、書き込むことが出来ない
        ASSERT_TRUE(result.IsSuccess() ||
                result <= nn::svc::ResultTimeout() ||
                result <= nn::svc::ResultReceptionSpecificationListBroken());
    }
    if (x == IpcClient)
    {
        {
            // 受信に失敗しているので、受信確認はない
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void ServerReceiveListWithInvalidMememoryPermmision(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("ServerReceiveListWithInvalidMememoryPermmision");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;

    // TEST 67-90, 68-137, 68-202
    // MemoryPermission_None を受信指定リストに指定して要求の受信は出来ない
    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0xFFFF, 0, 1, 0, 0, 0, 0, 3));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));

            // 受信指定リスト
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[0], sizeof(g_RecvBuffer0[0])));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);

            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                // MemoryPermission_None に結果を書き込めないので、失敗する
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
            }
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 3));
            // MemoryPermission_None を受信指定リストに指定
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(reinterpret_cast<char*>(g_FreeAreaBegin), ::std::strlen(TestMessage[0])));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        // 受信指定リストがNone なので失敗する
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultReceptionSpecificationListBroken());
    }

    // TEST 67-91, 68-138, 68-203
    // MemoryPermission_Read を受信指定リストに指定して要求の受信は出来ない
    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0xFFFF, 0, 1, 0, 0, 0, 0, 3));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));

            // 受信指定リスト
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[0], sizeof(g_RecvBuffer0[0])));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);

            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                // MemoryPermission_Read に結果を書き込めないので、失敗する
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
            }
        }
    }

    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 3));
            // MemoryPermission_Read を受信指定リストに指定
            WaitHeapState(g_HeapPtr, 0x1000);
            result = nn::svc::SetMemoryPermission(g_HeapPtr, 0x1000, nn::svc::MemoryPermission_Read);
            ASSERT_RESULT_SUCCESS(result);
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(reinterpret_cast<char*>(g_HeapPtr), 0x1000));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {

            // SendAsyncRequestWithUserBuffer から要求を受信できる

            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {

            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        // 受信指定リストがRead なので失敗する
        ASSERT_TRUE(result.IsSuccess() ||
                result <= nn::svc::ResultTimeout() ||
                result <= nn::svc::ResultReceptionSpecificationListBroken());

        result = nn::svc::SetMemoryPermission(g_HeapPtr, 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void ClientPointerWithInvalidMememoryPermmision(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE(__func__);
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // MemoryPermission_None をポインタ転送に指定した要求送信は失敗する
    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0xFFFF, 0, 1, 0, 0, 0, 0, 0));
            std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), 0x1000);

            TestMemoryPermission testPerm(reinterpret_cast<uintptr_t>(g_XferBuffer), 0x1000, nn::svc::MemoryPermission_None);

            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(g_XferBuffer, 0x1000, 0));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {

                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {

                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {

                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            // MemoryPermission_None に結果を書き込めないので、失敗する
            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                ASSERT_TRUE(
                        result <= nn::svc::ResultOutOfResource() ||
                        result <= nn::svc::ResultInvalidCurrentMemory());
            }

        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 1));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_TRUE(result.IsSuccess() ||
                result <= nn::svc::ResultTimeout() ||
                result <= nn::svc::ResultReceptionSpecificationListBroken());

        if (result.IsSuccess())
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.SetNull();

            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);

            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
    }
    if (x == IpcClient)
    {
        {
            // 受信に失敗しているので、受信確認はない
        }
    }

    // MemoryPermission_Read  をポインタ転送に指定した要求送信は成功する
    // TestMessage を使った転送を他で行っているので省略

    // MemoryPermission_ReadWrite をポインタ転送に指定した要求送信は成功する
    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0xFFFF, 0, 1, 0, 0, 0, 0, 0));

            CheckMemory(reinterpret_cast<uintptr_t>(g_XferBuffer),
                    nn::svc::MemoryState_CodeData,
                    nn::svc::MemoryPermission_ReadWrite,
                    0);
            std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), 0x10);

            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(g_XferBuffer, 0x10, 0));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {

                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {

                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 1));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0xFFFF);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);
            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == 0x10);
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(pMsgBuffer) < pointer.GetPointerAddress());
            ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(pMsgBuffer) + msgSize);
            ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), reinterpret_cast<void*>(nnMain), pointer.GetPointerSize()) == 0);
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 0, 0, 0));

            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);

            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
    }
    if (x == IpcClient)
    {
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void ServerPointerWithInvalidMememoryPermmision(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("ServerReceiveListWithInvalidMememoryPermmision");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;

    // MemoryPermission_None をポインタ転送の結果の送信に利用できない
    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0xFFFF, 0, 1, 0, 0, 0, 0, 3));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));

            // 受信指定リスト
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[0], sizeof(g_RecvBuffer0[0])));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);

            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                // MemoryPermission_None に結果を書き込めないので、失敗する
                ASSERT_TRUE(
                        result <= nn::svc::ResultOutOfResource() ||
                        result <= nn::svc::ResultInvalidCurrentMemory());
            }
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 3));
            // MemoryPermission_None を受信指定リストに指定
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(reinterpret_cast<char*>(&g_RecvBuffer1[0]), sizeof(g_RecvBuffer1[0])));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, WaitRequestTime);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, WaitRequestTime);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0xFFFF);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 3);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);
            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[0]));
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer1[0]) <= pointer.GetPointerAddress());
            ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(g_RecvBuffer1[0]) + msgSize);
            ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[0], pointer.GetPointerSize()) == 0);


            // 送信データ
            ::std::memcpy(g_XferBuffer, TestMessage[1], ::std::strlen(TestMessage[1]));
            TestMemoryPermission testPerm(reinterpret_cast<uintptr_t>(g_XferBuffer), 0x1000, nn::svc::MemoryPermission_None);
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 1, 0, 0, 0, 0, 0));


                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(g_XferBuffer, ::std::strlen(TestMessage[1]), 0));
            }
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
        }

        // MemoryPermission_None に対して、書き込むことが出来ない
        ASSERT_TRUE(
            result.IsSuccess() ||
            result <= nn::svc::ResultTimeout() ||
            result <= nn::svc::ResultReceptionSpecificationListBroken());
    }

    // MemoryPermission_Read をポインタ転送の結果の送信に利用できる
    // TestMessage を使った転送を他で行っているので省略

    // MemoryPermission_ReadWrite をポインタ転送の結果の送信に利用できる
    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0xFFFF, 0, 1, 0, 0, 0, 0, 3));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0));

            // 受信指定リスト
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[0], sizeof(g_RecvBuffer0[0])));

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                if (async)
                {
                    nn::svc::Handle event;
                    result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                    ASSERT_RESULT_SUCCESS(result);
                    int32_t index;
                    result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                    ASSERT_RESULT_SUCCESS(result);
                    result = nn::svc::CloseHandle(event);
                }
                else
                {
                    result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                }
            }
            else
            {
                result = nn::svc::SendSyncRequest(handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);

            ASSERT_RESULT_SUCCESS(result);
            /*
            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                // MemoryPermission_Read に結果を書き込めないので、失敗する
                ASSERT_TRUE(
                        result <= nn::svc::ResultOutOfResource() ||
                        result <= nn::svc::ResultInvalidCurrentMemory());
            }
            */
        }
    }

    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 3));
            // MemoryPermission_Read を受信指定リストに指定
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[0], ::std::strlen(TestMessage[0])));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0xFFFF);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 3);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);
            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[0]));
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer1[0]) <= pointer.GetPointerAddress());
            ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(g_RecvBuffer1[0]) + msgSize);
            ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[0], pointer.GetPointerSize()) == 0);


            // 送信データ
            ::std::memcpy(g_XferBuffer, TestMessage[1], ::std::strlen(TestMessage[1]));
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 1, 0, 0, 0, 0, 0));
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(g_XferBuffer, ::std::strlen(TestMessage[1]), 0));
            }
            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);

            // MemoryPermission_ReadWrite に対して、書き込むことが出来る
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
    }
    if (x == IpcClient)
    {
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);
            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[1]));
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer0[0]) <= pointer.GetPointerAddress());
            ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(g_RecvBuffer0[0]) + msgSize);
            ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), TestMessage[1], pointer.GetPointerSize()) == 0);

        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

/* TEST 28-109, 29-122, 30-132 */
/*
   MemoryPermission_None 領域を指定したsend 要求の送信は失敗する
 */
void CannotUseMemoryPermissionNoneForSend(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("CannotUseMemoryPermissionNoneForSend");
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        ::std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), sizeof(g_XferBuffer));

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 1, 0, 0, 0, 0));

        // MemoryPermission_None は使えない
        // Send 転送
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(g_FreeAreaBegin), 0x1000));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {

                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);

        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, WaitRequestTime);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, WaitRequestTime);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_TRUE(result.IsSuccess() ||
                result <= nn::svc::ResultTimeout() ||
                result <= nn::svc::ResultReceptionSpecificationListBroken());
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

/* TEST 67-92, 68-139, 68-204 */
/*
   MemoryPermission_Read 領域を指定したsend 要求に対して、結果を送信できる
 */
void CanUseMemoryPermissionReadForSend(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("CanUseMemoryPermissionReadForSend");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        ::std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), sizeof(g_XferBuffer));

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 1, 0, 0, 0, 0));

        // MemoryPermission_Read が使える
        // Send 転送
        result = nn::svc::SetMemoryPermission(reinterpret_cast<uintptr_t>(g_XferBuffer), 0x1000, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer, 0x1000));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {

                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 1);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Send
            {
                nn::svc::ipc::MessageBuffer::MapData mapData(nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial), ipcMsg);
                ASSERT_TRUE(mapData.GetDataSize() == 0x1000);
                ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                ASSERT_TRUE(::std::memcmp(g_XferBuffer, reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
            }
        }

        // 送信データを作成
        {
            // 全ての要求を送信
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
        }

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // 受信データの確認
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Send
            {
                // 特になし
            }
        }

        result = nn::svc::SetMemoryPermission(reinterpret_cast<uintptr_t>(g_XferBuffer), 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

/* TEST 28-110, 29-123, 30-133 */
/*
   MemoryPermission_None 領域を指定した recv 要求の送信は失敗する
 */
void CannotUseMemoryPermissionNoneForRecv(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("CannotUseMemoryPermissionNoneForRecv");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 1, 0, 0, 0));

        // Receive 転送
        // MemoryPermission_None は使えない
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(g_FreeAreaBegin), 0x1000));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {

                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {

            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);

        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, WaitRequestTime);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, WaitRequestTime);
        }
        AfterServerIpc(pMsgBuffer, msgSize);

        // 読み込めないので、失敗する
        ASSERT_TRUE(result.IsSuccess() ||
                result <= nn::svc::ResultTimeout() ||
                result <= nn::svc::ResultReceptionSpecificationListBroken());
    }
    if (x == IpcClient)
    {
        // 受信データの確認はしない
    }
    assertObj.Cancel();
}

/* TEST 67-93, 68-140, 68-205 */
/*
   MemoryPermission_Read 領域を指定した recv 結果の送信は失敗する
 */
void CannotUseMemoryPermissionReadForRecv(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("CannotUseMemoryPermissionReadForRecv");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 1, 0, 0, 0));

        result = nn::svc::SetMemoryPermission(reinterpret_cast<uintptr_t>(g_XferBuffer), 0x1000, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);

        // Receive 転送
        // MemoryPermission_Read は使えない
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(g_XferBuffer), 0x1000));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);

        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, WaitRequestTime);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, WaitRequestTime);
        }
        AfterServerIpc(pMsgBuffer, msgSize);

        ASSERT_TRUE(result.IsSuccess() ||
                result <= nn::svc::ResultTimeout() ||
                result <= nn::svc::ResultReceptionSpecificationListBroken());
    }
    if (x == IpcClient)
    {
        // 受信データの確認はしない

        result = nn::svc::SetMemoryPermission(reinterpret_cast<uintptr_t>(g_XferBuffer), 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

/* TEST 28-111, 29-124, 30-134 */
/*
   MemoryPermission_None 領域を指定した exchange 要求の送信は失敗する
 */
void CannotUseMemoryPermissionNoneForExchange(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("CannotUseMemoryPermissionNoneForExchange");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 1, 0, 0));

        // Exchange 転送
        // MemoryPermission_None を指定すると失敗する
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(g_FreeAreaBegin), 0x1000));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);

        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, WaitRequestTime);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, WaitRequestTime);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_TRUE(result.IsSuccess() ||
                result <= nn::svc::ResultTimeout() ||
                result <= nn::svc::ResultReceptionSpecificationListBroken());
    }
    if (x == IpcClient)
    {
        // 受信データの確認はしない
    }

    assertObj.Cancel();
} // NOLINT (readability/fn_size)

/* TEST 67-94, 68-141, 68-206 */
/*
   MemoryPermission_Read 領域を指定した exchange 結果の送信は失敗する
 */
void CannotUseMemoryPermissionReadForExchange(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("CannotUseMemoryPermissionReadForExchange");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        ::std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), sizeof(g_XferBuffer));

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 1, 0, 0));

        result = nn::svc::SetMemoryPermission(reinterpret_cast<uintptr_t>(g_XferBuffer) + 0x2000, 0x1000, nn::svc::MemoryPermission_Read);
        ASSERT_RESULT_SUCCESS(result);

        // Exchange 転送
        // MemoryPermission_Read を与えると失敗する
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + 0x2000, 0x1000));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);

        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, WaitRequestTime);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, WaitRequestTime);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_TRUE(result.IsSuccess() ||
                result <= nn::svc::ResultTimeout() ||
                result <= nn::svc::ResultReceptionSpecificationListBroken());
    }
    if (x == IpcClient)
    {
        // 受信データの確認はなし

        result = nn::svc::SetMemoryPermission(reinterpret_cast<uintptr_t>(g_XferBuffer) + 0x2000, 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// ポインタ転送でデータサイズ以上のバッファがないとき通信に失敗する
void SendOverSizePointerDataWithIpcBuffer(
        const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(x, &pMsgBuffer, &msgSize, userBuffer);

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;

    if (x == IpcClient)
    {
        SCOPED_TRACE("SendOverSizePointerDataWithIpcBuffer");
        NN_LOG("[%s]\n", __func__);
    }

    // TEST 28-112, 29-125, 30-135
    // バッファの大きさ以上の要求は送信できない
    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 1, 0, 0, 0, 0, 1);
            offset = ipcMsg.Set(ipcHeader);

            // バッファのサイズを超えるデータを入れる
            // ヘッダのサイズ分を考慮に入れると必ずオーバーする
            nn::svc::ipc::MessageBuffer::PointerData
                pointerData(reinterpret_cast<char*>(nnMain), msgSize - sizeof(nn::Bit32) * 4 + 1, 0);
            offset = ipcMsg.Set(offset, pointerData);

            result = SendIpc(handle, pMsgBuffer, msgSize, userBuffer, async);
            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                // 失敗しているので、ヘッダが 0 クリアされている
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            }
        }

    }

    // TEST 67-95, 68-142, 68-207
    // バッファの大きさ以上の要求を受信できない
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 1);
            ipcMsg.Set(ipcHeader);
        }
        result = ReceiveIpc(&handle, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_TRUE(result.IsSuccess() ||
                result <= nn::svc::ResultTimeout() ||
                result <= nn::svc::ResultReceptionSpecificationListBroken());
    }

    // TEST 28-113, 29-126, 30-136
    // バッファの大きさ以上の結果の受信はできない
    if (x == IpcClient)
    {
        int offset;
        // ポインタ転送にメッセージバッファを利用することが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 1, 0, 0, 0, 0, 1);
        offset = ipcMsg.Set(ipcHeader);

        nn::svc::ipc::MessageBuffer::PointerData
            pointerData(reinterpret_cast<char*>(nnMain), msgSize - sizeof(nn::Bit32) * 4, 0);
        offset = ipcMsg.Set(offset, pointerData);

        result = SendIpc(handle, pMsgBuffer, msgSize, userBuffer, async);
        // 結果送信時にバッファのサイズ以上のデータを送ろうとして失敗してくるはず
        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
        }
    }

    // TEST 67-96, 68-143, 68-208
    // バッファの大きさ以上の結果の送信はできない
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 1);
            ipcMsg.Set(ipcHeader);
        }
        result = ReceiveIpc(&handle, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_SUCCESS(result);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0x1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);

            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == msgSize - sizeof(nn::Bit32) * 4);
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(pMsgBuffer) < pointer.GetPointerAddress());
            ASSERT_TRUE(
                    pointer.GetPointerAddress()
                    < reinterpret_cast<uintptr_t>(pMsgBuffer) + msgSize);
            ASSERT_TRUE(
                    ::std::memcmp(
                        reinterpret_cast<void*>(pointer.GetPointerAddress()),
                        reinterpret_cast<void*>(nnMain),
                        pointer.GetPointerSize())
                    == 0);
        }
        {
            // バッファのサイズを超えるデータを入れる
            // ヘッダのサイズ分を考慮に入れると必ずオーバーする
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0000, 0, 0x1, 0, 0, 0, 0, 0);
            int offset = ipcMsg.Set(ipcHeader);

            nn::svc::ipc::MessageBuffer::PointerData
                pointerData(reinterpret_cast<char*>(nnMain), msgSize - sizeof(nn::Bit32) * 4 + 1, 0);
            offset = ipcMsg.Set(offset, pointerData);
        }
        result = ReplyIpc(handle, pMsgBuffer, msgSize, userBuffer);
        // サイズがオーバーしたためエラー
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// ポインタ転送で受信指定リストに指定した１つのバッファにすべてのデータを入れるときに、
// バッファのサイズが足りないと通信に失敗する
void SendOverSizePointerDataWithOneBuffer(
        const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(x, &pMsgBuffer, &msgSize, userBuffer);

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;

    if (x == IpcClient)
    {
        SCOPED_TRACE("SendOverSizePointerDataWithOneBuffer");
        NN_LOG("[%s]\n", __func__);
    }

    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 1, 0, 0, 0, 0, 2);
            offset = ipcMsg.Set(ipcHeader);

            // バッファのサイズを超えるデータを入れる
            nn::svc::ipc::MessageBuffer::PointerData
                pointerData(reinterpret_cast<char*>(nnMain), UserBufferSize + 1, 0);
            offset = ipcMsg.Set(offset, pointerData);
            nn::svc::ipc::MessageBuffer::ReceiveListEntry
                recvEntry(g_RecvBuffer0[0], UserBufferSize);
            offset = ipcMsg.Set(offset, recvEntry);

            result = SendIpc(handle, pMsgBuffer, msgSize, userBuffer, async);
            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            }
        }
    }

    // TEST 67-95, 68-142, 68-207
    // バッファの大きさ以上の要求を受信できない
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 2);
            int offset = ipcMsg.Set(ipcHeader);
            nn::svc::ipc::MessageBuffer::ReceiveListEntry
                recvEntry(g_RecvBuffer1[0], UserBufferSize);
            ipcMsg.Set(offset, recvEntry);
        }
        result = ReceiveIpc(&handle, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_TRUE(result.IsSuccess() ||
                result <= nn::svc::ResultTimeout() ||
                result <= nn::svc::ResultReceptionSpecificationListBroken());
    }

    // TEST 28-113, 29-126, 30-136
    // バッファの大きさ以上の結果の受信はできない
    if (x == IpcClient)
    {
        int offset;
        // ポインタ転送にメッセージバッファを利用することが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 1, 0, 0, 0, 0, 2);
        nn::svc::ipc::MessageBuffer::PointerData
            pointerData(reinterpret_cast<char*>(nnMain), UserBufferSize, 0);
        offset = ipcMsg.Set(ipcHeader);
        offset = ipcMsg.Set(offset, pointerData);
        nn::svc::ipc::MessageBuffer::ReceiveListEntry
            recvEntry(g_RecvBuffer0[0], UserBufferSize);
        offset = ipcMsg.Set(offset, recvEntry);

        result = SendIpc(handle, pMsgBuffer, msgSize, userBuffer, async);
        // 結果送信時にバッファのサイズ以上のデータを送ろうとして失敗してくるはず
        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
        }
    }

    // TEST 67-96, 68-143, 68-208
    // バッファの大きさ以上の結果の送信はできない
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 2);
            int offset = ipcMsg.Set(ipcHeader);
            nn::svc::ipc::MessageBuffer::ReceiveListEntry
                recvEntry(g_RecvBuffer1[0], UserBufferSize);
            ipcMsg.Set(offset, recvEntry);
        }
        result = ReceiveIpc(&handle, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_SUCCESS(result);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0x1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 2);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);

            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == UserBufferSize);
            ASSERT_TRUE(
                    reinterpret_cast<uintptr_t>(g_RecvBuffer1[0]) <= pointer.GetPointerAddress());
            ASSERT_TRUE(
                    pointer.GetPointerAddress()
                    < reinterpret_cast<uintptr_t>(g_RecvBuffer1[0]) + UserBufferSize);
            ASSERT_TRUE(
                    ::std::memcmp(
                        reinterpret_cast<void*>(pointer.GetPointerAddress()),
                        reinterpret_cast<void*>(nnMain),
                        pointer.GetPointerSize())
                    == 0);
        }
        {
            // バッファのサイズを超えるデータを入れる
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0000, 0, 0x1, 0, 0, 0, 0, 0);
            int offset = ipcMsg.Set(ipcHeader);

            nn::svc::ipc::MessageBuffer::PointerData
                pointerData(reinterpret_cast<char*>(nnMain), UserBufferSize + 1, 0);
            offset = ipcMsg.Set(offset, pointerData);
        }
        result = ReplyIpc(handle, pMsgBuffer, msgSize, userBuffer);
        // サイズがオーバーしたためエラー
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// ポインタ転送で受信指定リストに指定した複数のバッファの内、
// バッファのサイズが足りないものがあると通信に失敗する
void SendOverSizePointerDataWithMultiBuffer(
        const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(x, &pMsgBuffer, &msgSize, userBuffer);

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;

    if (x == IpcClient)
    {
        SCOPED_TRACE("SendOverSizePointerDataWithMultiBuffer");
        NN_LOG("[%s]\n", __func__);
    }

    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 2, 0, 0, 0, 0, 4);
            offset = ipcMsg.Set(ipcHeader);

            // バッファのサイズを超えるデータを入れる
            nn::svc::ipc::MessageBuffer::PointerData
                inSize(reinterpret_cast<char*>(nnMain), UserBufferSize, 0);
            nn::svc::ipc::MessageBuffer::PointerData
                outSize(reinterpret_cast<char*>(nnMain), UserBufferSize + 1, 1);
            offset = ipcMsg.Set(offset, inSize);
            offset = ipcMsg.Set(offset, outSize);
            for (int i = 0; i < 2; i++)
            {
                nn::svc::ipc::MessageBuffer::ReceiveListEntry
                    recvEntry(g_RecvBuffer0[i], UserBufferSize);
                offset = ipcMsg.Set(offset, recvEntry);
            }

            result = SendIpc(handle, pMsgBuffer, msgSize, userBuffer, async);
            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            }
        }
    }

    // バッファの大きさ以上の要求を受信できない
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 4);
            int offset = ipcMsg.Set(ipcHeader);
            for (int i = 0; i < 2; i++)
            {
                nn::svc::ipc::MessageBuffer::ReceiveListEntry
                    recvEntry(g_RecvBuffer1[i], UserBufferSize);
                offset = ipcMsg.Set(offset, recvEntry);
            }
        }
        result = ReceiveIpc(&handle, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_TRUE(result.IsSuccess() ||
                result <= nn::svc::ResultTimeout() ||
                result <= nn::svc::ResultReceptionSpecificationListBroken());
    }

    // バッファの大きさ以上の結果の受信はできない
    if (x == IpcClient)
    {
        int offset;
        // ポインタ転送にメッセージバッファを利用することが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 2, 0, 0, 0, 0, 4);
        offset = ipcMsg.Set(ipcHeader);

        for (int i = 0; i < 2; i++)
        {
            nn::svc::ipc::MessageBuffer::PointerData
                pointerData(reinterpret_cast<char*>(nnMain), UserBufferSize, i);
            offset = ipcMsg.Set(offset, pointerData);
        }
        for (int i = 0; i < 2; i++)
        {
            nn::svc::ipc::MessageBuffer::ReceiveListEntry
                recvEntry(g_RecvBuffer0[i], UserBufferSize);
            offset = ipcMsg.Set(offset, recvEntry);
        }

        result = SendIpc(handle, pMsgBuffer, msgSize, userBuffer, async);

        // 結果送信時にバッファのサイズ以上のデータを送ろうとして失敗してくるはず
        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
        }
    }

    // バッファの大きさ以上の結果の送信はできない
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 4);
            int offset = ipcMsg.Set(ipcHeader);
            for (int i = 0; i < 2; i++)
            {
                nn::svc::ipc::MessageBuffer::ReceiveListEntry
                    recvEntry(g_RecvBuffer1[i], UserBufferSize);
                offset = ipcMsg.Set(offset, recvEntry);
            }
        }
        result = ReceiveIpc(&handle, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_SUCCESS(result);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0x2);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 4);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            for (int i = 0; i < 2; i++)
            {
                nn::svc::ipc::MessageBuffer::PointerData pointer(offset + 2 * i, ipcMsg);

                ASSERT_TRUE(pointer.GetPointerIndex() == i);
                ASSERT_TRUE(pointer.GetPointerSize() == UserBufferSize);
                ASSERT_TRUE(
                        reinterpret_cast<uintptr_t>(g_RecvBuffer1[i]) <= pointer.GetPointerAddress());
                ASSERT_TRUE(
                        pointer.GetPointerAddress()
                        < reinterpret_cast<uintptr_t>(g_RecvBuffer1[i]) + UserBufferSize);
                ASSERT_TRUE(
                        ::std::memcmp(
                            reinterpret_cast<void*>(pointer.GetPointerAddress()),
                            reinterpret_cast<void*>(nnMain),
                            pointer.GetPointerSize())
                        == 0);
            }
        }
        {
            // バッファのサイズを超えるデータを入れる
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0000, 0, 0x2, 0, 0, 0, 0, 0);
            int offset = ipcMsg.Set(ipcHeader);

            nn::svc::ipc::MessageBuffer::PointerData
                inSize(reinterpret_cast<char*>(nnMain), UserBufferSize, 0);
            nn::svc::ipc::MessageBuffer::PointerData
                outSize(reinterpret_cast<char*>(nnMain), UserBufferSize + 1, 1);
            offset = ipcMsg.Set(offset, inSize);
            offset = ipcMsg.Set(offset, outSize);
        }
        result = ReplyIpc(handle, pMsgBuffer, msgSize, userBuffer);
        // サイズがオーバーしたためエラー
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// ポインタ転送を送信するときに、受信先で受信指定リストを指定していないと失敗する
void SendOverPointerNumZero(
        const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(x, &pMsgBuffer, &msgSize, userBuffer);

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;

    if (x == IpcClient)
    {
        SCOPED_TRACE("SendOverPointerNumZero");
        NN_LOG("[%s]\n", __func__);
    }

    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 1, 0, 0, 0, 0, 0);
            offset = ipcMsg.Set(ipcHeader);

            // 受信側のPointer転送の個数を超えて送信する
            nn::svc::ipc::MessageBuffer::PointerData
                pointerData(reinterpret_cast<char*>(nnMain), UserBufferSize, 0);
            offset = ipcMsg.Set(offset, pointerData);

            result = SendIpc(handle, pMsgBuffer, msgSize, userBuffer, async);
            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            }
        }
    }

    // 指定した受信指定リスト以上のポインタ転送を受信できない
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 0);
            ipcMsg.Set(ipcHeader);
        }
        result = ReceiveIpc(&handle, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_TRUE(result.IsSuccess() ||
                result <= nn::svc::ResultTimeout() ||
                result <= nn::svc::ResultReceptionSpecificationListBroken());
    }

    // 指定した受信指定リスト以上のポインタ転送を受信できない
    if (x == IpcClient)
    {
        int offset;
        // ポインタ転送にメッセージバッファを利用することが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 1, 0, 0, 0, 0, 0);
        offset = ipcMsg.Set(ipcHeader);

        nn::svc::ipc::MessageBuffer::PointerData
            pointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0);
        offset = ipcMsg.Set(offset, pointerData);

        result = SendIpc(handle, pMsgBuffer, msgSize, userBuffer, async);

        // 結果送信時に受信指定リストの数とポインタ転送の数が合わなくて失敗するはず
        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
        }
    }

    // 受信側の受信指定リストで指定したポインタ転送の数を超えて送信できない
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 1);

            ipcMsg.Set(ipcHeader);
        }

        result = ReceiveIpc(&handle, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_SUCCESS(result);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0x1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);

            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[0]));
            ASSERT_TRUE(
                    reinterpret_cast<uintptr_t>(pMsgBuffer) <= pointer.GetPointerAddress());
            ASSERT_TRUE(
                    pointer.GetPointerAddress()
                    < reinterpret_cast<uintptr_t>(pMsgBuffer) + msgSize);
            ASSERT_TRUE(
                    ::std::memcmp(
                        reinterpret_cast<void*>(pointer.GetPointerAddress()),
                        TestMessage[0],
                        pointer.GetPointerSize())
                        == 0);
        }
        {
            // ポインタ転送
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0000, 0, 0x1, 0, 0, 0, 0, 0);
            int offset = ipcMsg.Set(ipcHeader);

            nn::svc::ipc::MessageBuffer::PointerData
                pointerData(TestMessage[1], ::std::strlen(TestMessage[1]), 0);
            offset = ipcMsg.Set(offset, pointerData);
        }
        result = ReplyIpc(handle, pMsgBuffer, msgSize, userBuffer);
        // サイズがオーバーしたためエラー
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// ポインタ転送の個数が、受信指定リストで指定した個数よりも多いとき、失敗する
void SendOverPointerNum(
        const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(x, &pMsgBuffer, &msgSize, userBuffer);

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;

    if (x == IpcClient)
    {
        SCOPED_TRACE("SendOverPointerNum");
        NN_LOG("[%s]\n", __func__);
    }

    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 2, 0, 0, 0, 0, 0);
            offset = ipcMsg.Set(ipcHeader);

            // 受信側のPointer転送の個数を超えて送信する
            for (int i = 0; i < 2; i++)
            {
                nn::svc::ipc::MessageBuffer::PointerData
                    pointerData(TestMessage[i], ::std::strlen(TestMessage[i]), i);
                offset = ipcMsg.Set(offset, pointerData);
            }

            result = SendIpc(handle, pMsgBuffer, msgSize, userBuffer, async);
            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            }
        }
    }

    // 指定した受信指定リスト以上のポインタ転送を受信できない
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 3);
            int offset = ipcMsg.Set(ipcHeader);

            nn::svc::ipc::MessageBuffer::ReceiveListEntry
                recvEntry(g_RecvBuffer1[0], UserBufferSize);
            offset = ipcMsg.Set(offset, recvEntry);
        }
        result = ReceiveIpc(&handle, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_TRUE(result.IsSuccess() ||
                result <= nn::svc::ResultTimeout() ||
                result <= nn::svc::ResultReceptionSpecificationListBroken());
    }

    // 指定した受信指定リスト以上のポインタ転送を受信できない
    if (x == IpcClient)
    {
        int offset;
        // ポインタ転送にメッセージバッファを利用することが出来る
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 1, 0, 0, 0, 0, 3);
        offset = ipcMsg.Set(ipcHeader);

        nn::svc::ipc::MessageBuffer::PointerData
            pointerData(TestMessage[0], ::std::strlen(TestMessage[0]), 0);
        offset = ipcMsg.Set(offset, pointerData);

        nn::svc::ipc::MessageBuffer::ReceiveListEntry
            recvEntry(g_RecvBuffer0[0], UserBufferSize);
        offset = ipcMsg.Set(offset, recvEntry);

        result = SendIpc(handle, pMsgBuffer, msgSize, userBuffer, async);

        // 結果送信時に受信指定リストの数とポインタ転送の数が合わなくて失敗するはず
        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
        }
    }

    // 受信側の受信指定リストで指定したポインタ転送の数を超えて送信できない
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 3);
            int offset = ipcMsg.Set(ipcHeader);

            nn::svc::ipc::MessageBuffer::ReceiveListEntry
                recvEntry(g_RecvBuffer1[0], UserBufferSize);
            offset = ipcMsg.Set(offset, recvEntry);
        }

        result = ReceiveIpc(&handle, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_SUCCESS(result);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0x1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 3);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);

            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == ::std::strlen(TestMessage[0]));
            ASSERT_TRUE(
                    reinterpret_cast<uintptr_t>(g_RecvBuffer1[0]) <= pointer.GetPointerAddress());
            ASSERT_TRUE(
                    pointer.GetPointerAddress()
                    < reinterpret_cast<uintptr_t>(g_RecvBuffer1[0]) + msgSize);
            ASSERT_TRUE(
                    ::std::memcmp(
                        reinterpret_cast<void*>(pointer.GetPointerAddress()),
                        TestMessage[0],
                        pointer.GetPointerSize())
                        == 0);
        }

        {
            // ポインタ転送
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0000, 0, 0x2, 0, 0, 0, 0, 0);
            int offset = ipcMsg.Set(ipcHeader);

            for (int i = 0; i < 2; i++)
            {
                nn::svc::ipc::MessageBuffer::PointerData
                    pointerData(TestMessage[i], ::std::strlen(TestMessage[i]), i);
                offset = ipcMsg.Set(offset, pointerData);
            }
        }
        result = ReplyIpc(handle, pMsgBuffer, msgSize, userBuffer);
        // サイズがオーバーしたためエラー
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// 任意データのサイズがバッファのサイズを超えると失敗する
void SendOverSizeRawData(
        const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    GetIpcBuffer(x, &pMsgBuffer, &msgSize, userBuffer);
    int rawNum = userBuffer ? 0x3ff : msgSize / sizeof(nn::Bit32);

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;

    if (x == IpcClient)
    {
        SCOPED_TRACE("SendOverSizeRawData");
        NN_LOG("[%s]\n", __func__);
    }

    if (x == IpcClient)
    {
        // 送信側のバッファサイズ以上の任意データを送れない
        if (userBuffer)
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 0, 0, 0, 0, rawNum, 0);
            offset = ipcMsg.Set(ipcHeader);
            ASSERT_TRUE(msgSize > 0x1000);

            // バッファサイズ以上の任意データを送信すると失敗する
            offset = ipcMsg.SetRawArray(
                        offset, reinterpret_cast<void*>(nnMain), rawNum * sizeof(nn::Bit32));
            result = SendIpc(handle, pMsgBuffer, 0x1000, userBuffer, async);
            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
            }
        }

        // 受信側のバッファサイズ以上の任意データを送れない
        {
            // 大きなバッファで送る必要があるため、ユーザーバッファを使う
            pMsgBuffer = reinterpret_cast<nn::Bit32*>(g_UserBuffer0);
            msgSize = sizeof(g_UserBuffer0);
            ASSERT_TRUE(msgSize > rawNum * sizeof(nn::Bit32));
            TestIpcAssertObject assertObj2(handle, pMsgBuffer, msgSize);

            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0002, 0, 0, 0, 0, 0, rawNum, 0);
            offset = ipcMsg.Set(ipcHeader);

            offset = ipcMsg.SetRawArray(
                        offset, reinterpret_cast<void*>(nnMain), rawNum * sizeof(nn::Bit32));

            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);

            result = SendIpc(handle, pMsgBuffer, msgSize, true, false);
#ifdef SUPPORT_RESULT_TOO_LARGE_MESSAGE
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTooLargeMessage());
#else
            ASSERT_RESULT_FAILURE(result);
#endif
            assertObj2.Cancel();
        }

        GetIpcBuffer(x, &pMsgBuffer, &msgSize, userBuffer);
    }

    // バッファサイズ以上の任意データを結果として受け取れない
    if (x == IpcClient)
    {
        int offset;
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0003, 0, 0, 0, 0, 0, 1, 0);
        offset = ipcMsg.Set(ipcHeader);

        offset = ipcMsg.SetRawArray(
                offset, reinterpret_cast<void*>(nnMain), sizeof(nn::Bit32));
        size_t limitSize = (msgSize > 0x1000) ? 0x1000 : msgSize;

        // 多くとも 0x1000 を最大とするバッファを用いる
        result = SendIpc(handle, pMsgBuffer, limitSize, userBuffer, async);

        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
#ifdef SUPPORT_RESULT_TOO_LARGE_MESSAGE
            if (userBuffer)
            {
                ASSERT_TRUE(result <= nn::svc::ResultInvalidCombination());
            }
            else
            {
                ASSERT_TRUE(result <= nn::svc::ResultTooLargeMessage());
            }
#else
            ASSERT_RESULT_FAILURE(result);
#endif
        }

        GetIpcBuffer(x, &pMsgBuffer, &msgSize, userBuffer);


        if (userBuffer)
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0004, 0, 0, 0, 0, 0, 1, 0);
            offset = ipcMsg.Set(ipcHeader);

            offset = ipcMsg.SetRawArray(
                    offset, reinterpret_cast<void*>(nnMain), sizeof(nn::Bit32));
            size_t limitSize = (msgSize > 0x1000) ? 0x1000 : msgSize;

            // 多くとも 0x1000 を最大とするバッファを用いる
            result = SendIpc(handle, pMsgBuffer, limitSize, userBuffer, async);

            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
#ifdef SUPPORT_RESULT_TOO_LARGE_MESSAGE
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTooLargeMessage());
#else
                ASSERT_RESULT_FAILURE(result);
#endif
            }


        }
    }

    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 0);
            ipcMsg.Set(ipcHeader);
        }
        size_t limitSize = (msgSize > 0x1000) ? 0x1000 : msgSize;

        // ユーザーバッファのサイズを 0x1000 にして、大きすぎるメッセージは受け取れないことを確認する
        result = ReceiveIpc(&handle, 1, pMsgBuffer, limitSize, userBuffer);
        ASSERT_RESULT_SUCCESS(result);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0003);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 1);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // 任意データが受け取れている
            {
                int offset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);
                ASSERT_TRUE(::std::memcmp(pMsgBuffer + offset, reinterpret_cast<void*>(nnMain), sizeof(nn::Bit32)) == 0);
            }
        }

        // 送信に利用するバッファ以上の任意データを送れない
        if (userBuffer)
        {
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0000, 0, 0, 0, 0, 0, rawNum, 0);
                int offset = ipcMsg.Set(ipcHeader);

                offset = ipcMsg.SetRawArray(
                        offset, reinterpret_cast<void*>(nnMain), sizeof(nn::Bit32) * rawNum);
            }
            size_t limitSize = (msgSize > 0x1000) ? 0x1000 : msgSize;

            result = ReplyIpc(handle, pMsgBuffer, limitSize, userBuffer);
            // サイズがオーバーしたためエラー
            ASSERT_TRUE(result.IsSuccess() ||
                    result <= nn::svc::ResultTimeout() ||
                    result <= nn::svc::ResultReceptionSpecificationListBroken());
            GetIpcBuffer(x, &pMsgBuffer, &msgSize, userBuffer);

            // 受信バッファの準備
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 0);
                ipcMsg.Set(ipcHeader);
            }

            result = ReceiveIpc(&handle, 1, pMsgBuffer, limitSize, userBuffer);
            ASSERT_RESULT_SUCCESS(result);

            {
                // IPC ヘッダが受信できている
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
                nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
                ASSERT_TRUE(ipcHeader.GetTag() == 0x0004);
                ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
                ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
                ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
                ASSERT_TRUE(ipcHeader.GetRawNum() == 1);
                ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
                ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

                // 任意データが受け取れている
                {
                    int offset = nn::svc::ipc::MessageBuffer::GetRawDataOffset(ipcHeader, ipcSpecial);
                    ASSERT_TRUE(::std::memcmp(pMsgBuffer + offset, reinterpret_cast<void*>(nnMain), sizeof(nn::Bit32)) == 0);
                }
            }
        }

        // 受信側のバッファ以上の任意データを送れない
        {
            // ユーザーバッファを利用する

            pMsgBuffer = reinterpret_cast<nn::Bit32*>(g_UserBuffer1);
            msgSize = sizeof(g_UserBuffer1);
            ASSERT_TRUE(msgSize > 0x1000);
            TestIpcAssertObject assertObj2(handle, pMsgBuffer, msgSize);
            {
                nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
                nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 0, 0, 0, 0, rawNum, 0);
                int offset = ipcMsg.Set(ipcHeader);

                offset = ipcMsg.SetRawArray(
                        offset, reinterpret_cast<void*>(nnMain), sizeof(nn::Bit32) * rawNum);
            }


            result = ReplyIpc(handle, pMsgBuffer, msgSize, true);
            // サイズがオーバーしたためエラー
            ASSERT_TRUE(result.IsSuccess() ||
                    result <= nn::svc::ResultTimeout() ||
                    result <= nn::svc::ResultReceptionSpecificationListBroken());
            assertObj2.Cancel();
            GetIpcBuffer(x, &pMsgBuffer, &msgSize, userBuffer);
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void PointerZeroDataWithBuffer(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("PointerZeroDataWithBuffer");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // TEST 28-119, 29-133, 30-144
    // ポインタ転送でバッファを用いるとき、サイズが 0 のデータを持つ要求を送信できる
    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0xF, 0, 0, 0, 0, 1));

        for (int i = 0; i < 0xF; i++)
        {
            // ポインタ転送
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[i], 0, i));
        }

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 1));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        // TEST 67-103, 68-253, 68-269
        // ポインタ転送でバッファを用いるとき、サイズが 0 のデータを持つ要求を受信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0xF);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
                for (int i = 0; i < 0xF; i++)
                {
                    nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                    ASSERT_TRUE(pointer.GetPointerIndex() == i);
                    ASSERT_TRUE(pointer.GetPointerSize() == 0);
                    ASSERT_TRUE(pointer.GetPointerAddress() == 0);
                }
            }
        }

        // 送信データを作成
        {
            // 全ての要求を送信
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0xF, 0, 0, 0, 0, 0));

            // ポインタ転送
            for (int i = 0; i < 0xF; i++)
            {
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0xF - i - 1], 0, i));
            }
        }

        // TEST 67-104, 68-254, 68-270
        // ポインタ転送でバッファを用いるとき、サイズが 0 のデータを持つ結果を送信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // TEST 28-120, 29-134, 30-145
        // ポインタ転送でバッファを用いるとき、サイズが 0 のデータを持つ結果を受信できる
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0xF);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);

                for (int i = 0; i < 0xF; i++)
                {
                    nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                    ASSERT_TRUE(pointer.GetPointerIndex() == i);
                    ASSERT_TRUE(pointer.GetPointerSize() == 0);
                    ASSERT_TRUE(pointer.GetPointerAddress() == 0);
                }
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void PointerZeroDataWithReceiveListBuffer(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("PointerZeroDataWithReceiveListBuffer");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // TEST 28-121, 29-135, 30-146
    // ポインタ転送が受信指定リストの1つのバッファを利用する時、サイズが 0 のデータを持つ要求を送信できる
    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0xF, 0, 0, 0, 0, 2));

        for (int i = 0; i < 0xF; i++)
        {
            // ポインタ転送
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[i], 0, i));
        }


        // 受信指定リスト
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[0], sizeof(g_RecvBuffer0[0])));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 2));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[0], sizeof(g_RecvBuffer1[0])));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        // TEST 67-105, 68-255, 68-271
        // ポインタ転送が受信指定リストの1つのバッファを利用する時、サイズが 0 のデータを持つ要求を受信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0xF);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 2);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
                for (int i = 0; i < 0xF; i++)
                {
                    nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                    ASSERT_TRUE(pointer.GetPointerIndex() == i);
                    ASSERT_TRUE(pointer.GetPointerSize() == 0);
                    ASSERT_TRUE(pointer.GetPointerAddress() == 0);
                }
            }
        }

        // 送信データを作成
        {
            // 全ての要求を送信
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0xF, 0, 0, 0, 0, 0));

            // ポインタ転送
            for (int i = 0; i < 0xF; i++)
            {
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0xF - i - 1], 0, i));
            }
        }

        // TEST 67-106, 68-256, 68-272
        // ポインタ転送が受信指定リストの1つのバッファを利用する時、サイズが 0 のデータを持つ結果を送信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // TEST 28-122, 29-136, 30-147
        // ポインタ転送が受信指定リストの1つのバッファを利用する時、サイズが 0 のデータを持つ結果を受信できる
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0xF);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);

                for (int i = 0; i < 0xF; i++)
                {
                    nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                    ASSERT_TRUE(pointer.GetPointerIndex() == i);
                    ASSERT_TRUE(pointer.GetPointerSize() == 0);
                    ASSERT_TRUE(pointer.GetPointerAddress() == 0);
                }
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void PointerZeroDataWithReceiveList(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("PointerZeroDataWithReceiveList");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // TEST 28-123, 29-137, 30-148
    // ポインタ転送が受信指定リストを利用する時、サイズが 0 のデータを持つ要求を送信できる
    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 13, 0, 0, 0, 0, 15));

        for (int i = 0; i < 13; i++)
        {
            // ポインタ転送
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[i], 0, i));
        }


        // 受信指定リスト
        for (int i = 0; i < 13; i++)
        {
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[i], sizeof(g_RecvBuffer0[i])));
        }

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 15));
            for (int i = 0; i < 13; i++)
            {
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[i], sizeof(g_RecvBuffer1[i])));
            }
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        // TEST 67-107, 68-257, 68-273
        // ポインタ転送が受信指定リストを利用する時、サイズが 0 のデータを持つ要求を受信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 13);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 15);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
                for (int i = 0; i < 13; i++)
                {
                    nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                    ASSERT_TRUE(pointer.GetPointerIndex() == i);
                    ASSERT_TRUE(pointer.GetPointerSize() == 0);
                    ASSERT_TRUE(pointer.GetPointerAddress() == 0);
                }
            }
        }

        // 送信データを作成
        {
            // 全ての要求を送信
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 13, 0, 0, 0, 0, 0));

            // ポインタ転送
            for (int i = 0; i < 13; i++)
            {
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0xF - i - 1], 0, i));
            }
        }

        // TEST 67-108, 68-258, 68-274
        // ポインタ転送が受信指定リストを利用する時、サイズが 0 のデータを持つ結果を送信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // TEST 28-124, 29-138, 30-149
        // ポインタ転送が受信指定リストを利用する時、サイズが 0 のデータを持つ結果を受信できる
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 13);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);

                for (int i = 0; i < 13; i++)
                {
                    nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                    ASSERT_TRUE(pointer.GetPointerIndex() == i);
                    ASSERT_TRUE(pointer.GetPointerSize() == 0);
                    ASSERT_TRUE(pointer.GetPointerAddress() == 0);
                }
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void PointeWithZeroSizeReceiveListBuffer(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("PointeWithZeroSizeReceiveListBuffer");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // TEST 28-125, 29-139, 30-150
    // ポインタ転送で受信指定リストに指定した1つのバッファのサイズが 0 のデータを持つ要求を送信できる
    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0xF, 0, 0, 0, 0, 2));

        for (int i = 0; i < 0xF; i++)
        {
            // ポインタ転送
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[i], 0, i));
        }


        // 受信指定リスト
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[0], 0));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 2));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[0], 0));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        // TEST 67-109, 68-259, 68-275
        // ポインタ転送で受信指定リストに指定した1つのバッファのサイズが 0 のデータを持つ要求を受信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0xF);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 2);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
                for (int i = 0; i < 0xF; i++)
                {
                    nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                    ASSERT_TRUE(pointer.GetPointerIndex() == i);
                    ASSERT_TRUE(pointer.GetPointerSize() == 0);
                    ASSERT_TRUE(pointer.GetPointerAddress() == 0);
                }
            }
        }

        // 送信データを作成
        {
            // 全ての要求を送信
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0xF, 0, 0, 0, 0, 0));

            // ポインタ転送
            for (int i = 0; i < 0xF; i++)
            {
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0xF - i - 1], 0, i));
            }
        }

        // TEST 67-110, 68-260, 68-276
        // ポインタ転送で受信指定リストに指定した1つのバッファのサイズが 0 のデータを持つ結果を送信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // TEST 28-126, 29-140, 30-151
        // ポインタ転送で受信指定リストに指定した1つのバッファのサイズが 0 のデータを持つ結果を受信できる
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0xF);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);

                for (int i = 0; i < 0xF; i++)
                {
                    nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                    ASSERT_TRUE(pointer.GetPointerIndex() == i);
                    ASSERT_TRUE(pointer.GetPointerSize() == 0);
                    ASSERT_TRUE(pointer.GetPointerAddress() == 0);
                }
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void PointerWithZeroSizeReceiveList(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("PointerWithZeroSizeReceiveList");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // TEST 28-127, 29-141, 30-152
    // ポインタ転送で受信指定リストのバッファのサイズが 0 のデータを持つ要求を送信できる
    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 13, 0, 0, 0, 0, 15));

        for (int i = 0; i < 13; i++)
        {
            // ポインタ転送
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[i], 0, i));
        }


        // 受信指定リスト
        for (int i = 0; i < 13; i++)
        {
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer0[i], 0));
        }

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 15));
            for (int i = 0; i < 13; i++)
            {
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(g_RecvBuffer1[i], 0));
            }
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        // TEST 67-111, 68-261, 68-277
        // ポインタ転送で受信指定リストのバッファのサイズが 0 のデータを持つ要求を受信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 13);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 15);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
                for (int i = 0; i < 13; i++)
                {
                    nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                    ASSERT_TRUE(pointer.GetPointerIndex() == i);
                    ASSERT_TRUE(pointer.GetPointerSize() == 0);
                    ASSERT_TRUE(pointer.GetPointerAddress() == 0);
                }
            }
        }

        // 送信データを作成
        {
            // 全ての要求を送信
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 13, 0, 0, 0, 0, 0));

            // ポインタ転送
            for (int i = 0; i < 13; i++)
            {
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(TestMessage[0xF - i - 1], 0, i));
            }
        }

        // TEST 67-112, 68-262, 68-278
        // ポインタ転送で受信指定リストのバッファのサイズが 0 のデータを持つ結果を送信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // TEST 28-128, 29-142, 30-153
        // ポインタ転送で受信指定リストのバッファのサイズが 0 のデータを持つ結果を受信できる
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 13);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);

                for (int i = 0; i < 13; i++)
                {
                    nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                    ASSERT_TRUE(pointer.GetPointerIndex() == i);
                    ASSERT_TRUE(pointer.GetPointerSize() == 0);
                    ASSERT_TRUE(pointer.GetPointerAddress() == 0);
                }
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void SendZeroData(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("SendZeroData");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // TEST 28-129, 29-143, 30-154
    // send 転送で、マップサイズを 0 にして要求を送信できる
    if (x == IpcClient)
    {
        ::std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), sizeof(g_XferBuffer));

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0xF, 0, 0, 0, 0));

        // Send 転送
        for (int i = 0; i < 0xF; i++)
        {
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + i, 0));
        }

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        // TEST 67-113, 68-263, 68-279
        // send 転送で、マップサイズを 0 にして要求を受信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);


        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0xF);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Send
            {
                int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
                for (int i = 0; i < 0xF; i++)
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(offset + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32) * i, ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0);
                    ASSERT_TRUE(mapData.GetDataAddress() == 0);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                }
            }
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));

            // Send 転送
        }

        // TEST 67-114, 68-264, 68-280
        // send 転送で、マップサイズを 0 にして結果を送信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // TEST 28-130, 29-144, 30-155
        // send 転送で、マップサイズを 0 にして結果を受信できる
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Send
            {
                // 特になし
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void RecvZeroData(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("RecvZeroData");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // TEST 28-131, 29-145, 30-156
    // recv 転送で、マップサイズを 0 にして要求を送信できる
    if (x == IpcClient)
    {
        ::std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), sizeof(g_XferBuffer));

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0xF, 0, 0, 0));

        // Recv 転送
        for (int i = 0; i < 0xF; i++)
        {
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + i, 0));
        }

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        // TEST 67-115, 68-255, 68-281
        // recv 転送で、マップサイズを 0 にして要求を受信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0xF);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Recv
            {
                int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
                for (int i = 0; i < 0xF; i++)
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(offset + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32) * i, ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0);
                    ASSERT_TRUE(mapData.GetDataAddress() == 0);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                }
            }
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));

            // Recv 転送
        }

        // TEST 67-116, 68-256, 68-282
        // Recv 転送で、マップサイズを 0 にして結果を送信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // TEST 28-132, 29-146, 30-157
        // Recv 転送で、マップサイズを 0 にして結果を受信できる
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Recv
            {
                // データサイズが 0 なので、特になし
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void ExchangeZeroData(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("ExchangeZeroData");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    // TEST 28-133, 29-147, 30-158
    // Exchange 転送で、マップサイズを 0 にして要求を送信できる
    if (x == IpcClient)
    {
        ::std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), sizeof(g_XferBuffer));

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 0xF, 0, 0));

        // Exchange 転送
        for (int i = 0; i < 0xF; i++)
        {
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(g_XferBuffer + i, 0));
        }

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };
        // TEST 67-117, 68-257, 68-283
        // Exchange 転送で、マップサイズを 0 にして要求を受信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            // SendAsyncRequestWithUserBuffer から要求を受信できる
            // SendSyncRequestWithUserBuffer から要求を受信できる
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            // SendSyncRequest から要求を受信できる
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0xF);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Exchange
            {
                int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
                for (int i = 0; i < 0xF; i++)
                {
                    nn::svc::ipc::MessageBuffer::MapData mapData(offset + nn::svc::ipc::MessageBuffer::MapData::GetSize() / sizeof(nn::Bit32) * i, ipcMsg);
                    ASSERT_TRUE(mapData.GetDataSize() == 0);
                    ASSERT_TRUE(mapData.GetDataAddress() == 0);
                    ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                }
            }
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));

            // Exchange 転送
        }

        // TEST 67-118, 68-258, 68-284
        // Exchange 転送で、マップサイズを 0 にして結果を送信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // TEST 28-134, 29-148, 30-159
        // Recv 転送で、マップサイズを 0 にして結果を受信できる
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Exchange
            {
                // データサイズが 0 なので、特になし
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void MapDiscontiguousArea(uintptr_t heapAddr, size_t heapSize, uintptr_t freeAddr, size_t freeSize, int blockNum)
{
    size_t blockSize = 0x1000;
    ASSERT_TRUE(heapSize >= (2 * blockNum - 1) * blockSize);
    ASSERT_TRUE(freeSize >= blockNum * blockSize);
    nn::svc::PhysicalMemoryInfo info;

    for (int i = 0; i < blockNum; i++)
    {
        uintptr_t toAddr = freeAddr + i * blockSize;
        uintptr_t fromAddr = heapAddr + 2 * i * blockSize;

        nn::svc::MemoryInfo blockInfo;
        GetMemoryInfo(&blockInfo, toAddr);
        NN_ASSERT(blockInfo.state == nn::svc::MemoryState_Free);
        NN_ASSERT(blockInfo.permission == nn::svc::MemoryPermission_None);
        NN_ASSERT(blockInfo.attribute == 0);
        NN_ASSERT(blockInfo.baseAddress + blockInfo.size >= toAddr + blockSize);

        GetMemoryInfo(&blockInfo, fromAddr);
        NN_ASSERT(blockInfo.state == nn::svc::MemoryState_Normal);
        NN_ASSERT(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        NN_ASSERT(blockInfo.attribute == 0);
        NN_ASSERT(blockInfo.baseAddress + blockInfo.size >= fromAddr + blockSize);
        nn::Result result = nn::svc::MapMemory(toAddr, fromAddr, blockSize);
        ASSERT_RESULT_SUCCESS(result);

        if (i > 0)
        {
            result = nn::svc::QueryPhysicalAddress(&info, toAddr - blockSize);
            ASSERT_RESULT_SUCCESS(result);
            uint64_t beforeAddr = info.physicalAddress;

            result = nn::svc::QueryPhysicalAddress(&info, toAddr);
            ASSERT_RESULT_SUCCESS(result);
            uint64_t afterAddr = info.physicalAddress;

            ASSERT_TRUE(beforeAddr != afterAddr);
        }
    }
}

void UnmapDiscontiguousArea(uintptr_t heapAddr, size_t heapSize, uintptr_t freeAddr, size_t freeSize, int blockNum)
{
    size_t blockSize = 0x1000;
    ASSERT_TRUE(heapSize >= (2 * blockNum - 1) * blockSize);
    ASSERT_TRUE(freeSize >= blockNum * blockSize);

    for (int i = 0; i < blockNum; i++)
    {
        uintptr_t toAddr = freeAddr + i * blockSize;
        uintptr_t fromAddr = heapAddr + 2 * i * blockSize;
        nn::Result result = nn::svc::UnmapMemory(toAddr, fromAddr, blockSize);
        ASSERT_RESULT_SUCCESS(result);
    }
}

// TEST 901-19
// ポインタ転送で物理不連続な領域を指定することが出来る
void DiscontinuousPointerBuffer(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    size_t blockSize = 0x1000;
    int blockNum = 4;

    if (x == IpcClient)
    {
        SCOPED_TRACE("DiscontinuousPointerBuffer");
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        WaitHeapState(g_HeapPtr, (2 * blockNum - 1) * blockSize);
        MapDiscontiguousArea(
                g_HeapPtr, (2 * blockNum - 1) * blockSize, g_FreeAreaBegin, blockNum * blockSize, blockNum);

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 1, 0, 0, 0, 0, 3));

        // ポインタ転送
        uintptr_t pointerAddr = g_FreeAreaBegin;
        char* buffer = reinterpret_cast<char*>(pointerAddr);
        std::memcpy(buffer, reinterpret_cast<char*>(nnMain), blockSize * 2);
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(buffer, blockSize * 2, 0));

        // 受信指定リスト
        uintptr_t recvListAddr = g_FreeAreaBegin + blockSize * 2;
        char* recvListBuffer = reinterpret_cast<char*>(recvListAddr);
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(recvListBuffer, blockSize * 2));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        uintptr_t toAddr = g_FreeAreaBegin + blockNum * blockSize;
        uintptr_t fromAddr = g_HeapPtr + (2 * blockNum - 1) * blockSize;
        WaitHeapState(fromAddr, (2 * blockNum - 1) * blockSize);
        MapDiscontiguousArea(
                fromAddr, (2 * blockNum - 1) * blockSize,
                toAddr, blockNum * blockSize, blockNum);

        uintptr_t pointerAddr = toAddr;
        uintptr_t recvListAddr = toAddr + blockSize * 2;

        // 受信バッファの準備
        {
            char* recvListBuffer = reinterpret_cast<char*>(recvListAddr);
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 3));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::ReceiveListEntry(recvListBuffer, blockSize * 2));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 3);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
                nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);
                ASSERT_TRUE(pointer.GetPointerIndex() == 0);
                ASSERT_TRUE(pointer.GetPointerSize() == blockSize * 2);
                ASSERT_TRUE(recvListAddr <= pointer.GetPointerAddress());
                ASSERT_TRUE(pointer.GetPointerAddress() < recvListAddr + blockSize * 2);
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), reinterpret_cast<char*>(nnMain), pointer.GetPointerSize()) == 0);
            }
        }

        // 送信データを作成
        {
            // 全ての要求を送信
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 1, 0, 0, 0, 0, 0));

            // ポインタ転送
            char* buffer = reinterpret_cast<char*>(pointerAddr);
            std::memcpy(buffer, reinterpret_cast<char*>(nnMain) + 0x100, blockSize * 2);
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(buffer, blockSize * 2, 0));
        }

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        UnmapDiscontiguousArea(
                fromAddr, (2 * blockNum - 1) * blockSize,
                toAddr, blockNum * blockSize, blockNum);
    }
    if (x == IpcClient)
    {
        {
            uintptr_t recvListAddr = g_FreeAreaBegin + blockSize * 2;

            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);

                nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);

                ASSERT_TRUE(pointer.GetPointerIndex() == 0);
                ASSERT_TRUE(pointer.GetPointerSize() == blockSize * 2);
                ASSERT_TRUE(recvListAddr <= pointer.GetPointerAddress());
                ASSERT_TRUE(pointer.GetPointerAddress() < recvListAddr + blockSize * 2);
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), reinterpret_cast<char*>(nnMain) + 0x100, blockSize * 2) == 0);
            }
        }

        UnmapDiscontiguousArea(
                g_HeapPtr, (2 * blockNum - 1) * blockSize, g_FreeAreaBegin, blockNum * blockSize, blockNum);
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

nn::svc::MemoryInfo g_BackupState;
void MapDataAlignTestImpl(
        const bool x, nn::svc::Handle handle,
        uint64_t addr, uint64_t dataSize,
        uint64_t blockAddr, uint64_t blockSize,
        MapDataType dataType, int mapType, bool userBuffer, bool async)
{
    nn::Result result;
    nn::svc::MemoryInfo info;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    size_t headBufSize = 0;
    size_t tailBufSize = 0;

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    int sendNum = 0;
    int recvNum = 0;
    int exchNum = 0;

    switch(dataType)
    {
    case MapDataType_Send:
        sendNum = 1;
        break;
    case MapDataType_Receive:
        recvNum = 1;
        break;
    case MapDataType_Exchange:
        exchNum = 1;
        break;
    default: FAIL();
    }

    if (addr - blockAddr > 0x1000)
    {
        blockSize -= addr - blockAddr - 0x1000;
        blockAddr = addr - 0x1000;
    }

    if (x == IpcClient)
    {
        GetMemoryInfo(&info, addr);
        g_BackupState = info;
        headBufSize = addr - blockAddr;
        ::std::memcpy(g_XferBuffer, reinterpret_cast<void*>(blockAddr), headBufSize);
        if (dataType != MapDataType_Send)
        {
            ::std::memcpy(reinterpret_cast<void*>(addr), reinterpret_cast<void*>(nnMain), dataSize);
        }
        tailBufSize = (blockAddr + blockSize) - (addr + dataSize);
        if (tailBufSize > 0x1000)
        {
            tailBufSize = 0x1000;
        }
        ::std::memcpy(&g_XferBuffer[0x1000], reinterpret_cast<void*>(addr + dataSize), tailBufSize);

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        // ヘッダ
        nn::svc::ipc::MessageBuffer::MessageHeader
            ipcHeader(0x0001, 0, 0, sendNum, recvNum, exchNum, 0, 0);
        int offset = ipcMsg.Set(ipcHeader);

        nn::svc::ipc::MessageBuffer::MapData mapData(reinterpret_cast<void*>(addr), dataSize);
        offset = ipcMsg.Set(offset, mapData);
        pMsgBuffer[offset - 1] |= mapType;

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(
                            &event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(
                            reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        if (!result.IsSuccess())
        {
            GetMemoryInfo(&info, addr);
        }
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.SetNull();
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(
                        &index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize,
                        handles, sizeof(handles) / sizeof(*handles),
                        nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(
                        &index, handles, sizeof(handles) / sizeof(*handles),
                        nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        GetMemoryInfo(&info, addr);
#ifdef SUPPORT_IPC_LOCKED
        if ((addr & 0xfff) > 0 || dataSize < 0x1000)
        {
            // 端数部分は属性が付かない
            ASSERT_TRUE(info.attribute == g_BackupState.attribute);
            ASSERT_TRUE(info.ipcCount == g_BackupState.ipcCount);
        }
        else
        {
            ASSERT_TRUE((info.attribute & nn::svc::MemoryAttribute_IpcLocked) == nn::svc::MemoryAttribute_IpcLocked);
            ASSERT_TRUE(info.ipcCount == g_BackupState.ipcCount + 1);
        }
#endif

        uintptr_t buf;
        uintptr_t size;

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == sendNum);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == recvNum);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == exchNum);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::MapData mapData(offset, ipcMsg);
            ASSERT_TRUE(mapData.GetDataSize() == dataSize);
            ASSERT_TRUE(mapData.GetAttribute() == static_cast<nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute>(mapType));

            switch (dataType)
            {
                case MapDataType_Send:
                    ASSERT_TRUE(::std::memcmp(
                                    reinterpret_cast<void*>(mapData.GetDataAddress()),
                                    reinterpret_cast<void*>(addr),
                                    mapData.GetDataSize())
                                == 0);
                    break;
                case MapDataType_Receive:
                    break;
                case MapDataType_Exchange:
                    ASSERT_TRUE(::std::memcmp(
                                    reinterpret_cast<void*>(mapData.GetDataAddress()),
                                    reinterpret_cast<void*>(nnMain),
                                    mapData.GetDataSize())
                                == 0);
                    break;
                default:
                    FAIL();
            }

            buf = mapData.GetDataAddress();
            size = mapData.GetDataSize();

            GetMemoryInfo(&info, buf);
            switch(mapType)
            {
                case 0:
                    ASSERT_TRUE(info.state == nn::svc::MemoryState_Ipc);
                    break;
                case 1:
                    ASSERT_TRUE(info.state == nn::svc::MemoryState_NonSecureIpc);
                    break;
                case 3:
                    ASSERT_TRUE(info.state == nn::svc::MemoryState_NonDeviceIpc);
                    break;
                default:
                    FAIL();
            }
            ASSERT_TRUE(info.size >= size);
            if (dataType == MapDataType_Send)
            {
                ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_Read);
            }
            else
            {
                ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadWrite);
            }
            ASSERT_TRUE(info.attribute == 0);
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            nn::svc::ipc::MessageBuffer::MessageHeader
                ipcHeader (0x0002, 0, 0, 0, 0, 0, 0, 0);
            ipcMsg.Set(ipcHeader);

            if (dataType != MapDataType_Send)
            {
                ::std::memcpy(
                        reinterpret_cast<void*>(buf),
                        reinterpret_cast<char*>(nnMain) + 0x100,
                        size);
            }
        }

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(
                        &index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize,
                        handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        GetMemoryInfo(&info, buf);
        ASSERT_TRUE(info.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(info.attribute == 0);
    }
    if (x == IpcClient)
    {
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            if (dataType != MapDataType_Send)
            {
                ASSERT_TRUE(::std::memcmp(
                                reinterpret_cast<void*>(addr),
                                reinterpret_cast<char*>(nnMain) + 0x100,
                                dataSize)
                            == 0);
            }

            GetMemoryInfo(&info, addr);
            ASSERT_TRUE(info.state == g_BackupState.state);
            ASSERT_TRUE(info.permission == g_BackupState.permission);
            ASSERT_TRUE(info.attribute == g_BackupState.attribute);

            // 前後のバッファの内容が変わっていない
            ASSERT_TRUE(std::memcmp(g_XferBuffer, reinterpret_cast<void*>(blockAddr), headBufSize) == 0);
            ASSERT_TRUE(std::memcmp(&g_XferBuffer[0x1000], reinterpret_cast<void*>(addr + dataSize), tailBufSize) == 0);
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void CallMapAlignTest(const bool x, nn::svc::Handle handle, uintptr_t blockAddr, size_t blockSize, MapDataType dataType, int mapType, bool userBuffer, bool async)
{
    ASSERT_TRUE(blockSize >= 0x5000);

    // 前後領域を調べるため、0x1000 を空けておく
    uintptr_t addr = blockAddr + 0x1000;

    // 1 page
    MapDataAlignTestImpl(x, handle, addr, 0x1000, blockAddr, 0x5000, dataType, mapType, userBuffer, async);

    // TEST 901-1 から 901-3
    // 前方は揃っているけど、サイズが揃っていない
    MapDataAlignTestImpl(x, handle, addr, 0x100, blockAddr, 0x5000, dataType, mapType, userBuffer, async);

    // TEST 901-4 から 901-6
    // 前方は揃っていないが、後方は揃っている
    MapDataAlignTestImpl(x, handle, addr + 0x100, 0xF00, blockAddr, 0x5000, dataType, mapType, userBuffer, async);

    // TEST 901-7 から 901-9
    // 前方、後方共にページ境界をまたいでいる
    MapDataAlignTestImpl(x, handle, addr + 0x100, 0x1000, blockAddr, 0x5000, dataType, mapType, userBuffer, async);

    // TEST 901-10 から 901-12
    // 間にページを挟んで前方、後方共にページ境界をまたいでいる
    MapDataAlignTestImpl(x, handle, addr + 0x100, 0x2000, blockAddr, 0x5000, dataType, mapType, userBuffer, async);

    // TEST 901-13 から 901-15
    // 間にページを挟んで前方だけページ境界をまたいでいる
    MapDataAlignTestImpl(x, handle, addr + 0x100, 0x2F00, blockAddr, 0x5000, dataType, mapType, userBuffer, async);

    // TEST 901-16 から 901-18
    // 間にページを挟んで後方だけページ境界をまたいでいる
    MapDataAlignTestImpl(x, handle, addr, 0x2100, blockAddr, 0x5000, dataType, mapType, userBuffer, async);
}

void ReceiveListAlignTestImpl(
        const bool x, nn::svc::Handle handle,
        uint64_t clientAddr, uint64_t serverAddr, size_t dataSize,
        uint64_t clientBlockAddr, size_t clientBlockSize,
        uint64_t serverBlockAddr, size_t serverBlockSize,
        bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    size_t headBlockSize = 0;
    size_t tailBlockSize = 0;

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        headBlockSize = clientAddr - clientBlockAddr;
        if (headBlockSize > 0x1000)
        {
            clientBlockSize -= headBlockSize - 0x1000;
            headBlockSize = 0x1000;
            clientBlockAddr = clientAddr - 0x1000;
        }
        tailBlockSize = (clientBlockAddr + clientBlockSize) - (clientAddr + dataSize);
        if (tailBlockSize > 0x1000)
        {
            tailBlockSize = 0x1000;
        }
        std::memcpy(g_XferBuffer, reinterpret_cast<void*>(clientBlockAddr), headBlockSize);
        std::memcpy(&g_XferBuffer[0x1000], reinterpret_cast<void*>(clientAddr + dataSize), tailBlockSize);

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        // ヘッダ
        nn::svc::ipc::MessageBuffer::MessageHeader
            ipcHeader(0x0001, 0, 1, 0, 0, 0, 0, 3);
        int offset = ipcMsg.Set(ipcHeader);

        // ポインタデータ
        nn::svc::ipc::MessageBuffer::PointerData
            pointerData(reinterpret_cast<char*>(nnMain), dataSize, 0);

        offset = ipcMsg.Set(offset, pointerData);

        // 受信指定リスト
        nn::svc::ipc::MessageBuffer::ReceiveListEntry
            receiveEntry(reinterpret_cast<char*>(clientAddr), dataSize);
        offset = ipcMsg.Set(offset, receiveEntry);

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(
                            &event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(
                            reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        {
            headBlockSize = serverAddr - serverBlockAddr;
            if (headBlockSize > 0x1000)
            {
                serverBlockSize -= headBlockSize - 0x1000;
                headBlockSize = 0x1000;
                serverBlockAddr = serverAddr - 0x1000;
            }
            tailBlockSize = (serverBlockAddr + serverBlockSize) - (serverAddr + dataSize);
            if (tailBlockSize > 0x1000)
            {
                tailBlockSize = 0x1000;
            }
            std::memcpy(&g_XferBuffer[0x2000], reinterpret_cast<void*>(serverBlockAddr), headBlockSize);
            std::memcpy(&g_XferBuffer[0x3000], reinterpret_cast<void*>(serverAddr + dataSize), tailBlockSize);
        }
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader
                ipcHeader(0, 0, 0, 0, 0, 0, 0, 3);
            int offset = ipcMsg.Set(ipcHeader);

            // 受信指定リスト
            nn::svc::ipc::MessageBuffer::ReceiveListEntry
                receiveEntry(reinterpret_cast<char*>(serverAddr), dataSize);
            offset = ipcMsg.Set(offset, receiveEntry);
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(
                        &index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize,
                        handles, sizeof(handles) / sizeof(*handles),
                        nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(
                        &index, handles, sizeof(handles) / sizeof(*handles),
                        nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 3);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
                nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);
                ASSERT_TRUE(pointer.GetPointerIndex() == 0);
                ASSERT_TRUE(pointer.GetPointerSize() == dataSize);
                ASSERT_TRUE(serverAddr <= pointer.GetPointerAddress());
                ASSERT_TRUE(pointer.GetPointerAddress() < serverAddr + dataSize);
                ASSERT_TRUE(::std::memcmp(
                                reinterpret_cast<void*>(pointer.GetPointerAddress()),
                                reinterpret_cast<char*>(nnMain),
                                pointer.GetPointerSize())
                            == 0);
            }

            {
                ASSERT_TRUE(::std::memcmp(
                                reinterpret_cast<void*>(serverBlockAddr),
                                &g_XferBuffer[0x2000],
                                headBlockSize)
                            == 0);
                ASSERT_TRUE(::std::memcmp(
                                reinterpret_cast<void*>(serverAddr + dataSize),
                                &g_XferBuffer[0x3000],
                                tailBlockSize)
                            == 0);
            }
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            nn::svc::ipc::MessageBuffer::MessageHeader
                ipcHeader (0x0002, 0, 1, 0, 0, 0, 0, 0);
            int offset = ipcMsg.Set(ipcHeader);

            // ポインタデータ
            nn::svc::ipc::MessageBuffer::PointerData
                pointerData(reinterpret_cast<char*>(nnMain) + 0x100, dataSize, 0);

            offset = ipcMsg.Set(offset, pointerData);
        }

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(
                        &index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize,
                        handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);
            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == dataSize);
            ASSERT_TRUE(clientAddr <= pointer.GetPointerAddress());
            ASSERT_TRUE(pointer.GetPointerAddress() < clientAddr + dataSize);
            ASSERT_TRUE(::std::memcmp(
                            reinterpret_cast<void*>(pointer.GetPointerAddress()),
                            reinterpret_cast<char*>(nnMain) + 0x100,
                            pointer.GetPointerSize())
                        == 0);

            {
                ASSERT_TRUE(::std::memcmp(
                                reinterpret_cast<void*>(clientBlockAddr),
                                &g_XferBuffer,
                                headBlockSize)
                            == 0);
                ASSERT_TRUE(::std::memcmp(
                                reinterpret_cast<void*>(clientAddr + dataSize),
                                &g_XferBuffer[0x1000],
                                tailBlockSize)
                            == 0);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)


void CallReceiveListAlignTest(
        const bool x, nn::svc::Handle handle,
        uint64_t clientBlockAddr, size_t clientBlockSize,
        uint64_t serverBlockAddr, size_t serverBlockSize,
        bool userBuffer, bool async)
{
    uintptr_t clientAddr = clientBlockAddr + 0x1000;
    uintptr_t serverAddr = serverBlockAddr + 0x1000;

    // Success Check
    {
        size_t dataSize = clientBlockSize < serverBlockSize ? clientBlockSize : serverBlockSize;
        if (dataSize > 0x1000)
        {
            dataSize = 0x1000;
        }
        ReceiveListAlignTestImpl(
                x, handle, clientBlockAddr, serverBlockAddr, dataSize,
                clientBlockAddr, clientBlockSize, serverBlockAddr, serverBlockSize, userBuffer, async);
    }

    if (clientBlockSize < 0x5000 ||
        serverBlockSize < 0x5000)
    {
        return;
    }

    // TEST 901-21
    // アラインが前後揃っている
    ReceiveListAlignTestImpl(
            x, handle, clientAddr, serverAddr, 0x1000,
            clientBlockAddr, clientBlockSize, serverBlockAddr, serverBlockSize, userBuffer, async);

    // TEST 901-22
    // アラインが前後共に揃っていない
    ReceiveListAlignTestImpl(
            x, handle, clientAddr + 0x100, serverAddr + 0x100, 0x1100,
            clientBlockAddr, clientBlockSize, serverBlockAddr, serverBlockSize, userBuffer, async);

    // TEST 901-23
    // データがページの内部に収まっている
    ReceiveListAlignTestImpl(
            x, handle, clientAddr + 0x100, serverAddr + 0x100, 0x100,
            clientBlockAddr, clientBlockSize, serverBlockAddr, serverBlockSize, userBuffer, async);

    // TEST 901-24
    // 前方は揃っているけど、後方が揃っていない
    ReceiveListAlignTestImpl(
            x, handle, clientAddr, serverAddr, 0x1100,
            clientBlockAddr, clientBlockSize, serverBlockAddr, serverBlockSize, userBuffer, async);

    // TEST 901-25
    // 前方は揃っていないが、後方は揃っている
    ReceiveListAlignTestImpl(
            x, handle, clientAddr + 0x100, serverAddr + 0x100, 0x1F00,
            clientBlockAddr, clientBlockSize, serverBlockAddr, serverBlockSize, userBuffer, async);

    // TEST 901-26
    // 前方、後方共にページ境界をまたいでいる
    ReceiveListAlignTestImpl(
            x, handle, clientAddr + 0x100, serverAddr + 0x100, 0x2100,
            clientBlockAddr, clientBlockSize, serverBlockAddr, serverBlockSize, userBuffer, async);

    // TEST 901-27
    // 間にページを挟んで前方、後方共にページ境界をまたいでいる
    ReceiveListAlignTestImpl(
            x, handle, clientAddr + 0x100, serverAddr + 0x100, 0x3100,
            clientBlockAddr, clientBlockSize, serverBlockAddr, serverBlockSize, userBuffer, async);

    // TEST 901-28
    // 間にページを挟んで前方だけページ境界をまたいでいる
    ReceiveListAlignTestImpl(
            x, handle, clientAddr + 0x100, serverAddr + 0x100, 0x2F00,
            clientBlockAddr, clientBlockSize, serverBlockAddr, serverBlockSize, userBuffer, async);

    // TEST 901-29
    // 間にページを挟んで後方だけページ境界をまたいでいる
    ReceiveListAlignTestImpl(
            x, handle, clientAddr, serverAddr, 0x2100,
            clientBlockAddr, clientBlockSize, serverBlockAddr, serverBlockSize, userBuffer, async);
}

#if (defined NN_BUILD_CONFIG_CPU_ARM_V8A && defined NN_BUILD_CONFIG_ABI_LP64)
void Pointer64bitDataTestIn64BitEnv(
        const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    uint16_t dataSize = 0x1000;

    if (x == IpcClient)
    {
        SCOPED_TRACE(__func__);
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        // ヘッダ
        nn::svc::ipc::MessageBuffer::MessageHeader
            ipcHeader(0x0001, 0, 1, 0, 0, 0, 0, 3);
        int offset = ipcMsg.Set(ipcHeader);

        ASSERT_TRUE(g_HeapPtr > 0x100000000);
        WaitHeapState(g_HeapPtr, dataSize);

        ::std::memcpy(reinterpret_cast<void*>(g_HeapPtr), reinterpret_cast<void*>(nnMain), dataSize);

        // ポインタデータ
        nn::svc::ipc::MessageBuffer::PointerData
            pointerData(reinterpret_cast<void*>(g_HeapPtr), dataSize, 0);

        offset = ipcMsg.Set(offset, pointerData);

        // 受信指定リスト
        nn::svc::ipc::MessageBuffer::ReceiveListEntry
            receiveEntry(g_RecvBuffer0, dataSize);
        offset = ipcMsg.Set(offset, receiveEntry);

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(
                            &event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(
                            reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader
                ipcHeader(0, 0, 0, 0, 0, 0, 0, 3);
            int offset = ipcMsg.Set(ipcHeader);

            // 受信指定リスト
            nn::svc::ipc::MessageBuffer::ReceiveListEntry
                receiveEntry(g_RecvBuffer1, dataSize);
            offset = ipcMsg.Set(offset, receiveEntry);
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(
                        &index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize,
                        handles, sizeof(handles) / sizeof(*handles),
                        nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(
                        &index, handles, sizeof(handles) / sizeof(*handles),
                        nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 3);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // ポインタ転送
            {
                int offset;

                offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
                nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);
                ASSERT_TRUE(pointer.GetPointerIndex() == 0);
                ASSERT_TRUE(pointer.GetPointerSize() == dataSize);
                ASSERT_TRUE(::std::memcmp(
                                reinterpret_cast<void*>(pointer.GetPointerAddress()),
                                reinterpret_cast<char*>(nnMain),
                                pointer.GetPointerSize())
                            == 0);
            }
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            nn::svc::ipc::MessageBuffer::MessageHeader
                ipcHeader (0x0002, 0, 1, 0, 0, 0, 0, 0);
            int offset = ipcMsg.Set(ipcHeader);

            ::std::memcpy(
                    reinterpret_cast<void*>(g_HeapPtr + dataSize),
                    reinterpret_cast<void*>(nnMain),
                    dataSize);

            // ポインタデータ
            nn::svc::ipc::MessageBuffer::PointerData
                pointerData(reinterpret_cast<void*>(g_HeapPtr + dataSize), dataSize, 0);

            offset = ipcMsg.Set(offset, pointerData);
        }

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(
                        &index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize,
                        handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);
            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == dataSize);
            ASSERT_TRUE(::std::memcmp(
                            reinterpret_cast<void*>(pointer.GetPointerAddress()),
                            reinterpret_cast<char*>(nnMain),
                            pointer.GetPointerSize())
                        == 0);

        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)
#endif

void Pointer64bitDataTest(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
#if (defined NN_BUILD_CONFIG_CPU_ARM_V7A) \
    || ((defined NN_BUILD_CONFIG_CPU_ARM_V8A) && (defined NN_BUILD_CONFIG_ABI_ILP32))
    // TEST 901-30, 31
    // 32ビット 環境では64 bit アドレスを指定すると失敗する
    // アドレスの上位 4 bit は無視されるのが仕様なので、廃止します。
    NN_UNUSED(x);
    NN_UNUSED(handle);
    NN_UNUSED(userBuffer);
    NN_UNUSED(async);
#else
    // TEST 901-33
    // ARMv8 lp64 環境では64 bit アドレスを指定することが出来る
    Pointer64bitDataTestIn64BitEnv(x, handle, userBuffer, async);
#endif
}

#if (defined NN_BUILD_CONFIG_CPU_ARM_V8A && defined NN_BUILD_CONFIG_ABI_LP64)
void MapData64bitTestIn64bitEnvImpl(
        const bool x, nn::svc::Handle handle, uint64_t dataSize,
        MapDataType dataType, bool userBuffer, bool async)
{
    nn::Result result;
    nn::svc::MemoryInfo info;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE(__func__);
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    int sendNum = 0;
    int recvNum = 0;
    int exchNum = 0;

    switch(dataType)
    {
    case MapDataType_Send:
        sendNum = 1;
        break;
    case MapDataType_Receive:
        recvNum = 1;
        break;
    case MapDataType_Exchange:
        exchNum = 1;
        break;
    default: FAIL();
    }

    if (x == IpcClient)
    {
        ASSERT_TRUE(g_HeapPtr > 0x100000000);
        WaitHeapState(g_HeapPtr, dataSize);

        ::std::memcpy(
                reinterpret_cast<void*>(g_HeapPtr), reinterpret_cast<void*>(nnMain), dataSize);

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        // ヘッダ
        nn::svc::ipc::MessageBuffer::MessageHeader
            ipcHeader(0x0001, 0, 0, sendNum, recvNum, exchNum, 0, 0);
        int offset = ipcMsg.Set(ipcHeader);

        nn::svc::ipc::MessageBuffer::MapData
            mapData(reinterpret_cast<void*>(static_cast<uintptr_t>(g_HeapPtr)), dataSize);
        offset = ipcMsg.Set(offset, mapData);

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(
                            &event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(
                            reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.SetNull();
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(
                        &index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize,
                        handles, sizeof(handles) / sizeof(*handles),
                        nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(
                        &index, handles, sizeof(handles) / sizeof(*handles),
                        nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        GetMemoryInfo(&info, g_HeapPtr);
#ifdef SUPPORT_IPC_LOCKED
        ASSERT_TRUE(info.attribute == nn::svc::MemoryAttribute_IpcLocked);
#endif

        uintptr_t buf;
        uintptr_t size;

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == sendNum);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == recvNum);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == exchNum);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::MapData mapData(offset, ipcMsg);
            ASSERT_TRUE(mapData.GetDataSize() == dataSize);
            ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
            if (dataType != MapDataType_Receive)
            {
                ASSERT_TRUE(::std::memcmp(
                                reinterpret_cast<void*>(mapData.GetDataAddress()),
                                reinterpret_cast<void*>(nnMain),
                                mapData.GetDataSize())
                            == 0);
            }
            buf = mapData.GetDataAddress();
            size = mapData.GetDataSize();

            GetMemoryInfo(&info, buf);
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Ipc);
            ASSERT_TRUE(info.size >= size);
            if (dataType == MapDataType_Send)
            {
                ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_Read);
            }
            else
            {
                ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadWrite);
            }
            ASSERT_TRUE(info.attribute == 0);


        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            nn::svc::ipc::MessageBuffer::MessageHeader
                ipcHeader (0x0002, 0, 0, 0, 0, 0, 0, 0);
            ipcMsg.Set(ipcHeader);

            if (dataType != MapDataType_Send)
            {
                ::std::memcpy(
                        reinterpret_cast<void*>(buf),
                        reinterpret_cast<char*>(nnMain) + 0x100,
                        size);
            }
        }

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(
                        &index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize,
                        handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        GetMemoryInfo(&info, buf);
        ASSERT_TRUE(info.state == nn::svc::MemoryState_Free);
        ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_None);
        ASSERT_TRUE(info.attribute == 0);
    }
    if (x == IpcClient)
    {
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            if (dataType != MapDataType_Send)
            {
                ASSERT_TRUE(::std::memcmp(
                                reinterpret_cast<void*>(g_HeapPtr),
                                reinterpret_cast<char*>(nnMain) + 0x100,
                                dataSize)
                            == 0);
            }
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)
#endif

void MapData64bitTestImpl(
        const bool x, nn::svc::Handle handle, uint64_t dataSize,
        MapDataType dataType, bool userBuffer, bool async)
{
#if (defined NN_BUILD_CONFIG_CPU_ARM_V7A) \
    || ((defined NN_BUILD_CONFIG_CPU_ARM_V8A) && (defined NN_BUILD_CONFIG_ABI_ILP32))
    // 901-30, 31
    // 64 bit アドレスを指定すると無視される
    // 上位 4 bit は無視されるのが仕様なので廃止します。
    NN_UNUSED(x);
    NN_UNUSED(handle);
    NN_UNUSED(dataSize);
    NN_UNUSED(dataType);
    NN_UNUSED(userBuffer);
    NN_UNUSED(async);
#else
    // 901-33
    // 64 bit アドレスを指定することが出来る
    MapData64bitTestIn64bitEnvImpl(x, handle, dataSize, dataType, userBuffer, async);
#endif
}

void MapArea64bitTest(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    MapData64bitTestImpl(x, handle, 0x1000, MapDataType_Send, userBuffer, async);
    MapData64bitTestImpl(x, handle, 0x1000, MapDataType_Receive, userBuffer, async);
    MapData64bitTestImpl(x, handle, 0x1000, MapDataType_Exchange, userBuffer, async);
}

void CheckSuccessPointerMemoryFromServer(
        TestMemoryInfo** array, int numArray, nn::svc::Handle serverSession, bool userBuffer)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(IpcServer, &pMsgBuffer, &msgSize, userBuffer);
    TestIpcAssertObject assertObj(serverSession, pMsgBuffer, msgSize);

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 1, 0, 0, 0, 0, 0);

    ipcMsg.SetNull();
    result = ReceiveIpc(&serverSession, 1, pMsgBuffer, msgSize, userBuffer);
    ASSERT_RESULT_SUCCESS(result);
    ipcMsg.Set(IpcSuccessHeader);
    result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // サーバー側がポインタ転送に用いるバッファのメモリ属性をチェック
    for(int i = 0; i < numArray; i++)
    {
        if (array[i]->GetSize() == 0)
        {
            array[i]->Close();
            continue;
        }
        if (array[i]->GetPermission() == nn::svc::MemoryPermission_None)
        {
            array[i]->Close();
            continue;
        }

        array[i]->SetHeap(g_HeapPtr, g_HeapSize);

        array[i]->Initialize();
        uintptr_t addr = array[i]->GetAddress();
        size_t size = (array[i]->GetSize() > 0x10) ? 0x10 : array[i]->GetSize();

        ipcMsg.SetNull();
        result = ReceiveIpc(&serverSession, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_SUCCESS(result);

        int offset = ipcMsg.Set(ipcHeader);
        nn::svc::ipc::MessageBuffer::PointerData ptrData(reinterpret_cast<char*>(addr), size, 0);
        offset = ipcMsg.Set(offset, ptrData);

        result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        array[i]->Close();
    }
    assertObj.Cancel();
}

void CheckInvalidPointerMemoryFromServer(
        TestMemoryInfo** array, int numArray, nn::svc::Handle serverSession, bool userBuffer)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(IpcServer, &pMsgBuffer, &msgSize, userBuffer);

    TestIpcAssertObject assertObj(serverSession, pMsgBuffer, msgSize);

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 1, 0, 0, 0, 0, 0);

    // サーバー側がポインタ転送に用いるバッファのメモリ属性をチェック
    for(int i = 0; i < numArray; i++)
    {
        if (array[i]->GetSize() == 0)
        {
            array[i]->Close();
            continue;
        }

        array[i]->SetHeap(g_HeapPtr, g_HeapSize);
        array[i]->Initialize();
        uintptr_t addr = array[i]->GetAddress();
        size_t size = (array[i]->GetSize() > 0x10) ? 0x10 : array[i]->GetSize();

        if (static_cast<uint64_t>(addr) >= 0x1000000000ULL)
        {
            array[i]->Close();
            continue;
        }

        ipcMsg.SetNull();
        result = ReceiveIpc(&serverSession, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_SUCCESS(result);

        int offset = ipcMsg.Set(ipcHeader);
        nn::svc::ipc::MessageBuffer::PointerData ptrData(reinterpret_cast<char*>(addr), size, 0);
        offset = ipcMsg.Set(offset, ptrData);

        result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        array[i]->Close();
    }
    assertObj.Cancel();
}

void CheckSuccessPointerMemoryFromClient(
        TestMemoryInfo** array, int numArray, nn::svc::Handle clientSession,
        bool userBuffer, bool async)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(IpcClient, &pMsgBuffer, &msgSize, userBuffer);

    TestIpcAssertObject assertObj(clientSession, pMsgBuffer, msgSize);

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 1, 0, 0, 0, 0, 0);

    // クライアント側がポインタ転送に用いるバッファのメモリ属性をチェック
    for(int i = 0; i < numArray; i++)
    {
        if (array[i]->GetSize() == 0)
        {
            array[i]->Close();
            continue;
        }
        if (array[i]->GetPermission() == nn::svc::MemoryPermission_None)
        {
            array[i]->Close();
            continue;
        }

        array[i]->SetHeap(g_HeapPtr, g_HeapSize);
        array[i]->Initialize();
        uintptr_t addr = array[i]->GetAddress();
        size_t size = (array[i]->GetSize() > 0x10) ? 0x10 : array[i]->GetSize();

        int offset = ipcMsg.Set(ipcHeader);
        nn::svc::ipc::MessageBuffer::PointerData ptrData(reinterpret_cast<char*>(addr), size, 0);
        offset = ipcMsg.Set(offset, ptrData);

        result = SendIpc(clientSession, pMsgBuffer, msgSize, userBuffer, async);
        ASSERT_RESULT_SUCCESS(result);

        array[i]->Close();
    }
    assertObj.Cancel();
}

void CheckInvalidPointerMemoryFromClient(
        TestMemoryInfo** array, int numArray, nn::svc::Handle clientSession,
        bool userBuffer, bool async)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(IpcClient, &pMsgBuffer, &msgSize, userBuffer);

    TestIpcAssertObject assertObj(clientSession, pMsgBuffer, msgSize);

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 1, 0, 0, 0, 0, 0);

    ipcMsg.Set(IpcFaildHeader);
    result = SendIpc(clientSession, pMsgBuffer, msgSize, userBuffer, async);
    ASSERT_RESULT_SUCCESS(result);

    // クライアント側がポインタ転送に用いるバッファのメモリ属性をチェック
    for(int i = 0; i < numArray; i++)
    {
        if (array[i]->GetSize() == 0)
        {
            array[i]->Close();
            continue;
        }

        array[i]->SetHeap(g_HeapPtr, g_HeapSize);
        array[i]->Initialize();
        uintptr_t addr = array[i]->GetAddress();
        size_t size = (array[i]->GetSize() > 0x10) ? 0x10 : array[i]->GetSize();

        if (static_cast<uint64_t>(addr) >= 0x1000000000ULL)
        {
            array[i]->Close();
            continue;
        }

        int offset = ipcMsg.Set(ipcHeader);
        nn::svc::ipc::MessageBuffer::PointerData ptrData(reinterpret_cast<char*>(addr), size, 0);
        offset = ipcMsg.Set(offset, ptrData);

        result = SendIpc(clientSession, pMsgBuffer, msgSize, userBuffer, async);
        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }

        array[i]->Close();
    }
    assertObj.Cancel();
}

void PointerMemoryAreaTestServer(nn::svc::Handle serverSession, bool userBuffer)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(IpcServer, &pMsgBuffer, &msgSize, userBuffer);
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcReplyHeader(0x0002, 0, 0, 0, 0, 0, 0, 0);

    TestIpcAssertObject assertObj(serverSession, pMsgBuffer, msgSize);

    // クライアント側がポインタ転送に用いるバッファのメモリ属性のチェックをしている
    for(;;)
    {
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0002, 0, 0, 0, 0, 0, 0, 1);
        ipcMsg.Set(ipcHeader);

        result = ReceiveIpc(&serverSession, 1, pMsgBuffer, msgSize, userBuffer);
        if (result.IsSuccess())
        {
            if (GetIpcTag(pMsgBuffer) == IpcEndHeaderTag)
            {
                break;
            }
            ipcMsg.Set(ipcReplyHeader);
            result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
        else
        {
            ASSERT_TRUE(
                    result <= nn::svc::ResultReceptionSpecificationListBroken() ||
                    result <= nn::svc::ResultTimeout());
        }
    }

    ipcMsg.Set(ipcReplyHeader);
    result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);

    // メモリ状態
    nn::svc::MemoryState invalidState[] = {
        nn::svc::MemoryState_Io,
#ifndef SUPPORT_IPC_WITH_STATIC
        nn::svc::MemoryState_Static,
#endif // SUPPORT_IPC_WITH_STATIC
        nn::svc::MemoryState_Free, // Permision=None
        nn::svc::MemoryState_Inaccessible,
    };
    const int NumInvalidState = sizeof(invalidState) / sizeof(nn::svc::MemoryState);

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    TestMemoryInfo* invalidArea[NumInvalidState];
    header->GetTestListWithStates(invalidArea, invalidState, NumInvalidState);

    const int NumSuccessState = NumTestMemoryState - NumInvalidState;
    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListExceptStates(successArea, NumSuccessState, invalidState, NumInvalidState);

    CheckSuccessPointerMemoryFromServer(successArea, NumSuccessState, serverSession, userBuffer);

    // Free だけ確認する
    TestFreeMemoryState testFreeState;
    TestMemoryInfo* invalidFreeArea[1] = { &testFreeState };
    CheckInvalidPointerMemoryFromServer(invalidFreeArea, 1, serverSession, userBuffer);

    // メモリ属性
    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    uint32_t successAttr[] = {
        0,
        nn::svc::MemoryAttribute_Locked,
        nn::svc::MemoryAttribute_DeviceShared,
#ifdef SUPPORT_IPC_LOCKED
        nn::svc::MemoryAttribute_IpcLocked,
        nn::svc::MemoryAttribute_IpcLocked | nn::svc::MemoryAttribute_DeviceShared,
#endif
    };
    const int NumSuccessAttr = sizeof(successAttr) / sizeof(nn::svc::MemoryAttribute);

    TestMemoryInfo* successAttrArea[NumSuccessAttr];
    header->GetTestListWithAttributes(successAttrArea, successAttr, NumSuccessAttr);

    const int NumInvalidAttr = NumTestMemoryAttribute - NumSuccessAttr;
    TestMemoryInfo* invalidAttrArea[NumInvalidAttr];
    header->GetTestListExceptAttributes(
            invalidAttrArea, NumInvalidAttr, successAttr, NumSuccessAttr);

    CheckSuccessPointerMemoryFromServer(successAttrArea, NumSuccessAttr, serverSession, userBuffer);
    // サーバー側は invalid のチェックしない

    // AliasCode のバリエーション
    {
        nn::svc::MemoryPermission perms[] = {
            nn::svc::MemoryPermission_None,
#ifdef SVC_AGING_TEST
            nn::svc::MemoryPermission_Read,
            nn::svc::MemoryPermission_ReadExecute,
#endif
            nn::svc::MemoryPermission_ReadWrite,
        };
        const int NumPerms = sizeof(perms) / sizeof(nn::svc::MemoryPermission);

        for (int i = 0; i < NumPerms; i++)
        {
            TestAliasCodeMemoryState aliasCode;
            aliasCode.SetPermission(perms[i]);
            TestMemoryInfo* memoryInfo = &aliasCode;
            // サーバー側は invalid のチェックしない
            if (perms[i] != nn::svc::MemoryPermission_None)
            {
                CheckSuccessPointerMemoryFromServer(
                        &memoryInfo, 1, serverSession, userBuffer);
            }
        }
    }

    ipcMsg.SetNull();
    result = ReceiveIpc(&serverSession, 1, pMsgBuffer, msgSize, userBuffer);
    ASSERT_RESULT_SUCCESS(result);
    ipcMsg.Set(IpcEndHeader);
    result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    assertObj.Cancel();
}

void PointerMemoryAreaTestClient(nn::svc::Handle clientSession, bool userBuffer, bool async)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(IpcClient, &pMsgBuffer, &msgSize, userBuffer);
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

    TestIpcAssertObject assertObj(clientSession, pMsgBuffer, msgSize);

    // メモリ状態
    nn::svc::MemoryState invalidState[] = {
        nn::svc::MemoryState_Io,
#ifndef SUPPORT_IPC_WITH_STATIC
        nn::svc::MemoryState_Static,
#endif // SUPPORT_IPC_WITH_STATIC
        nn::svc::MemoryState_Free, // Permision=None
        nn::svc::MemoryState_Inaccessible,
    };

    const int NumInvalidState = sizeof(invalidState) / sizeof(nn::svc::MemoryState);

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    TestMemoryInfo* invalidArea[NumInvalidState];
    header->GetTestListWithStates(invalidArea, invalidState, NumInvalidState);

    const int NumSuccessState = NumTestMemoryState - NumInvalidState;
    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListExceptStates(successArea, NumSuccessState, invalidState, NumInvalidState);

    CheckSuccessPointerMemoryFromClient(
            successArea, NumSuccessState, clientSession, userBuffer, async);
    CheckInvalidPointerMemoryFromClient(
            invalidArea, NumInvalidState, clientSession, userBuffer, async);

    // メモリ属性
    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    uint32_t successAttr[] = {
        0,
        nn::svc::MemoryAttribute_Locked,
        nn::svc::MemoryAttribute_DeviceShared,
#ifdef SUPPORT_IPC_LOCKED
        nn::svc::MemoryAttribute_IpcLocked,
        nn::svc::MemoryAttribute_DeviceShared | nn::svc::MemoryAttribute_IpcLocked,
#endif
    };
    const int NumSuccessAttr = sizeof(successAttr) / sizeof(nn::svc::MemoryAttribute);

    TestMemoryInfo* successAttrArea[NumSuccessAttr];
    header->GetTestListWithAttributes(successAttrArea, successAttr, NumSuccessAttr);

    const int NumInvalidAttr = NumTestMemoryAttribute - NumSuccessAttr;
    TestMemoryInfo* invalidAttrArea[NumInvalidAttr];
    header->GetTestListExceptAttributes(
            invalidAttrArea, NumInvalidAttr, successAttr, NumSuccessAttr);

    CheckSuccessPointerMemoryFromClient(
            successAttrArea, NumSuccessAttr, clientSession, userBuffer, async);
    CheckInvalidPointerMemoryFromClient(
            invalidAttrArea, NumInvalidAttr, clientSession, userBuffer, async);

    // AliasCode のバリエーション
    {
        nn::svc::MemoryPermission perms[] = {
            nn::svc::MemoryPermission_None,
#ifdef SVC_AGING_TEST
            nn::svc::MemoryPermission_Read,
            nn::svc::MemoryPermission_ReadExecute,
#endif
            nn::svc::MemoryPermission_ReadWrite,
        };
        const int NumPerms = sizeof(perms) / sizeof(nn::svc::MemoryPermission);

        for (int i = 0; i < NumPerms; i++)
        {
            TestAliasCodeMemoryState aliasCode;
            aliasCode.SetPermission(perms[i]);
            TestMemoryInfo* memoryInfo = &aliasCode;
            if (perms[i] == nn::svc::MemoryPermission_None)
            {
                CheckInvalidPointerMemoryFromClient(
                        &memoryInfo, 1, clientSession, userBuffer, async);
            }
            else
            {
                CheckSuccessPointerMemoryFromClient(
                        &memoryInfo, 1, clientSession, userBuffer, async);
            }
        }
    }

    ipcMsg.Set(IpcEndHeader);
    result = SendIpc(clientSession, pMsgBuffer, msgSize, userBuffer, async);
    ASSERT_RESULT_SUCCESS(result);

    // サーバー側がポインタ転送に用いるバッファのメモリ状態をチェック
    bool checkResult = false;
    for(;;)
    {
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 0, 0, 0, 0, 0, 1);
        ipcMsg.Set(ipcHeader);
        result = SendIpc(clientSession, pMsgBuffer, msgSize, userBuffer, async);

        if (result.IsSuccess())
        {
            if (GetIpcTag(pMsgBuffer) == IpcEndHeaderTag)
            {
                break;
            }
            else if (GetIpcTag(pMsgBuffer) == IpcSuccessHeaderTag)
            {
                checkResult = true;
            }
            else if (GetIpcTag(pMsgBuffer) == IpcFaildHeaderTag)
            {
                checkResult = false;
            }
            else if (checkResult)
            {
                ASSERT_RESULT_SUCCESS(result);
            }
            else
            {
                FAIL();
            }
        }
        else
        {
            if (!checkResult)
            {
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
            }
        }
    }
    assertObj.Cancel();
} // NOLINT(readability/fn_size)

void PointerMemoryAreaTest(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    if (x == IpcClient)
    {
        SCOPED_TRACE(__func__);
        NN_LOG("[%s]\n", __func__);
    }

    if (x == IpcServer)
    {
        PointerMemoryAreaTestServer(handle, userBuffer);
    }
    else
    {
        PointerMemoryAreaTestClient(handle, userBuffer, async);
    }
}

void CheckSuccessReceiveListMemoryFromServer(
        TestMemoryInfo** array, int numArray, nn::svc::Handle serverSession, bool userBuffer)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(IpcServer, &pMsgBuffer, &msgSize, userBuffer);
    TestIpcAssertObject assertObj(serverSession, pMsgBuffer, msgSize);

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 0, 0, 0, 0, 0, 3);

    // 開始要求
    {
        nn::svc::ipc::MessageBuffer::MessageHeader tmpIpcHeader(0x0001, 0, 0, 0, 0, 0, 0, 1);
        ipcMsg.Set(tmpIpcHeader);
        result = ReceiveIpc(&serverSession, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_SUCCESS(result);
        ipcMsg.Set(IpcSuccessHeader);
        result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }

    // サーバー側がポインタ転送に用いるバッファのメモリ属性をチェック
    for(int i = 0; i < numArray; i++)
    {
        if (array[i]->GetSize() == 0)
        {
            array[i]->Close();
            continue;
        }

        array[i]->SetHeap(g_HeapPtr, g_HeapSize);
        array[i]->Initialize();

        uintptr_t addr = array[i]->GetAddress();
        size_t size = array[i]->GetSize();

        if (size > 0x5000)
        {
            size = 0x5000;
        }

        if (array[i]->GetState() == nn::svc::MemoryState_ThreadLocal)
        {
            addr = addr + size - 0x10;
            size = 0x10;
        }

        ServerSendData(serverSession, &size, sizeof(size));

        CallReceiveListAlignTest(
                IpcServer, serverSession,
                -1, -1, // client 側は使わない
                addr, size,
                userBuffer, false);

        array[i]->Close();
    }

    // 終了通知
    size_t size = 0;
    ServerSendData(serverSession, &size, sizeof(size));
    assertObj.Cancel();
}

void CheckInvalidReceiveListMemoryFromServer(
        TestMemoryInfo** array, int numArray, nn::svc::Handle serverSession, bool userBuffer)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(IpcServer, &pMsgBuffer, &msgSize, userBuffer);
    TestIpcAssertObject assertObj(serverSession, pMsgBuffer, msgSize);

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 0, 0, 0, 0, 0, 3);

    {
        nn::svc::ipc::MessageBuffer::MessageHeader tmpIpcHeader(0x0001, 0, 0, 0, 0, 0, 0, 1);
        ipcMsg.Set(tmpIpcHeader);
        result = ReceiveIpc(&serverSession, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_SUCCESS(result);
        ipcMsg.Set(IpcFaildHeader);
        result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }

    // サーバー側がポインタ転送に用いるバッファのメモリ属性をチェック
    for(int i = 0; i < numArray; i++)
    {
        if (array[i]->GetSize() == 0)
        {
            array[i]->Close();
            continue;
        }

        array[i]->SetHeap(g_HeapPtr, g_HeapSize);
        array[i]->Initialize();
        uintptr_t addr = array[i]->GetAddress();
        size_t size = (array[i]->GetSize() > 0x10) ? 0x10 : array[i]->GetSize();

        if (static_cast<uint64_t>(addr) >= 0x1000000000ULL)
        {
            array[i]->Close();
            continue;
        }

        int offset = ipcMsg.Set(ipcHeader);

        nn::svc::ipc::MessageBuffer::ReceiveListEntry
            recvList(reinterpret_cast<char*>(addr), size);
        offset = ipcMsg.Set(offset, recvList);

        result = ReceiveIpc(&serverSession, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_TRUE(
                result.IsSuccess() ||
                result <= nn::svc::ResultReceptionSpecificationListBroken());

        if (result.IsSuccess())
        {
            ipcMsg.SetNull();
            result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }

        array[i]->Close();
    }

    nn::svc::ipc::MessageBuffer::MessageHeader ipcTmpHeader(0x0001, 0, 0, 0, 0, 0, 0, 1);
    ipcMsg.Set(ipcTmpHeader);

    result = ReceiveIpc(&serverSession, 1, pMsgBuffer, msgSize, userBuffer);
    ASSERT_RESULT_SUCCESS(result);
    ipcMsg.Set(IpcEndHeader);
    result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
    assertObj.Cancel();
}

void CheckSuccessReceiveListMemoryFromClient(
        TestMemoryInfo** array, int numArray, nn::svc::Handle clientSession,
        bool userBuffer, bool async)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(IpcClient, &pMsgBuffer, &msgSize, userBuffer);
    TestIpcAssertObject assertObj(clientSession, pMsgBuffer, msgSize);

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 0, 0, 0, 0, 0, 3);

    // 開始要求
    {
        ipcMsg.Set(IpcSuccessHeader);
        result = SendIpc(clientSession, pMsgBuffer, msgSize, userBuffer, async);
        ASSERT_RESULT_SUCCESS(result);
    }

    // クライアント側が受信リストに用いるバッファのメモリ属性をチェック
    for(int i = 0; i < numArray; i++)
    {
        if (array[i]->GetSize() == 0)
        {
            array[i]->Close();
            continue;
        }

        array[i]->SetHeap(g_HeapPtr, g_HeapSize);
        array[i]->Initialize();

        uintptr_t addr = array[i]->GetAddress();
        size_t size = array[i]->GetSize();

        if (size > 0x5000)
        {
            size = 0x5000;
        }

        if (array[i]->GetState() == nn::svc::MemoryState_ThreadLocal)
        {
            addr = addr + size - 0x10;
            size = 0x10;
        }

        ClientSendData(clientSession, &size, sizeof(size));

        CallReceiveListAlignTest(
                IpcClient, clientSession,
                addr, size,
                -1, -1, // サーバー側は使わない
                userBuffer, async);

        array[i]->Close();
    }

    // 終了通知
    size_t size = 0;
    ClientSendData(clientSession, &size, sizeof(size));
    assertObj.Cancel();
}

void CheckInvalidReceiveListMemoryFromClient(
        TestMemoryInfo** array, int numArray, nn::svc::Handle clientSession,
        bool userBuffer, bool async)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(IpcClient, &pMsgBuffer, &msgSize, userBuffer);
    TestIpcAssertObject assertObj(clientSession, pMsgBuffer, msgSize);

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 0, 0, 0, 0, 0, 3);

    // 開始通知
    {
        ipcMsg.Set(IpcFaildHeader);
        result = SendIpc(clientSession, pMsgBuffer, msgSize, userBuffer, async);
        ASSERT_RESULT_SUCCESS(result);
    }

    // クライアント側がポインタ転送に用いるバッファのメモリ属性をチェック
    for(int i = 0; i < numArray; i++)
    {
        if (array[i]->GetSize() == 0)
        {
            array[i]->Close();
            continue;
        }

        array[i]->SetHeap(g_HeapPtr, g_HeapSize);
        array[i]->Initialize();
        uintptr_t addr = array[i]->GetAddress();
        size_t size = (array[i]->GetSize() > 0x10) ? 0x10 : array[i]->GetSize();

        if (static_cast<uint64_t>(addr) >= 0x1000000000ULL)
        {
            array[i]->Close();
            continue;
        }

        int offset = ipcMsg.Set(ipcHeader);

        nn::svc::ipc::MessageBuffer::ReceiveListEntry
            recvList(reinterpret_cast<char*>(addr), size);
        offset = ipcMsg.Set(offset, recvList);

        result = SendIpc(clientSession, pMsgBuffer, msgSize, userBuffer, async);
        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }

        array[i]->Close();
    }

    ipcMsg.Set(IpcEndHeader);
    result = SendIpc(clientSession, pMsgBuffer, msgSize, userBuffer, async);
    ASSERT_RESULT_SUCCESS(result);
    assertObj.Cancel();
}

void CheckReceiveListFromClient(nn::svc::Handle serverSession, nn::Bit32* pMsgBuffer, size_t msgSize, bool userBuffer)
{
    nn::Result result;

    TestIpcAssertObject assertObj(serverSession, pMsgBuffer, msgSize);

    bool isEnd = false;
    size_t size = 0;
    const size_t dataSize = 0x10;
    int offset = 0;

    while(!isEnd)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 0, 0, 0, 0, 0, 1);
        nn::svc::ipc::MessageBuffer::MessageHeader ipcReplyHeader(0x0002, 0, 1, 0, 0, 0, 0, 0);
        nn::svc::ipc::MessageBuffer::PointerData
            ptrData(reinterpret_cast<char*>(nnMain), dataSize, 0);

        ipcMsg.Set(ipcHeader);
        result = ReceiveIpc(&serverSession, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_SUCCESS(result);

        switch(GetIpcTag(pMsgBuffer))
        {
            case IpcEndHeaderTag:
                isEnd = true;

                ipcMsg.SetNull();
                result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
                break;
            case IpcSuccessHeaderTag:
                ipcMsg.SetNull();
                result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

                for (;;)
                {
                    ServerReceiveData(&size, sizeof(size), serverSession);
                    if (size == 0)
                    {
                        break;
                    }
                    CallReceiveListAlignTest(
                            IpcServer, serverSession,
                            -1, -1, // クライアント側は使わない
                            g_HeapPtr + g_HeapSize - 0x5000, size,
                            userBuffer, false);
                }
                break;
            case IpcFaildHeaderTag:
                ipcMsg.SetNull();
                result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

                for(;;)
                {
                    ipcMsg.Set(ipcHeader);
                    result = ReceiveIpc(&serverSession, 1, pMsgBuffer, msgSize, userBuffer);
                    ASSERT_RESULT_SUCCESS(result);
                    if (GetIpcTag(pMsgBuffer) == IpcEndHeaderTag)
                    {
                        ipcMsg.SetNull();
                        result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
                        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
                        break;
                    }

                    offset = ipcMsg.Set(ipcReplyHeader);
                    offset = ipcMsg.Set(offset, ptrData);
                    result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
                    ASSERT_TRUE(
                            // result.IsSuccess() ||
                            result <= nn::svc::ResultTimeout() ||
                            result <= nn::svc::ResultReceptionSpecificationListBroken());
                }
                break;
            default:
                FAIL();
        }

    }
    assertObj.Cancel();
}

void ReceiveListMemoryAreaTestServer(nn::svc::Handle serverSession, bool userBuffer)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(IpcServer, &pMsgBuffer, &msgSize, userBuffer);
    TestIpcAssertObject assertObj(serverSession, pMsgBuffer, msgSize);
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcReplyHeader(0x0002, 0, 1, 0, 0, 0, 0, 0);

    // クライアント側のチェック
    CheckReceiveListFromClient(serverSession, pMsgBuffer, msgSize, userBuffer);

    // メモリ状態
    nn::svc::MemoryState invalidState[] = {
        nn::svc::MemoryState_Io,
#ifndef SUPPORT_IPC_WITH_STATIC
        nn::svc::MemoryState_Static,
#endif // SUPPORT_IPC_WITH_STATIC
        nn::svc::MemoryState_Free, // Permision=None
        nn::svc::MemoryState_Code, // Permision=ReadExecute
        nn::svc::MemoryState_AliasCode, // Permision!=ReadWrite
        nn::svc::MemoryState_Inaccessible,
    };
    const int NumInvalidState = sizeof(invalidState) / sizeof(nn::svc::MemoryState);

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    TestMemoryInfo* invalidArea[NumInvalidState];
    header->GetTestListWithStates(invalidArea, invalidState, NumInvalidState);

    const int NumSuccessState = NumTestMemoryState - NumInvalidState;
    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListExceptStates(successArea, NumSuccessState, invalidState, NumInvalidState);

    CheckSuccessReceiveListMemoryFromServer(successArea, NumSuccessState, serverSession, userBuffer);

    // Free だけ確認
    TestFreeMemoryState testFreeState;
    TestMemoryInfo* invalidFreeArea[1] = { &testFreeState };
    CheckInvalidReceiveListMemoryFromServer(invalidFreeArea, 1, serverSession, userBuffer);

    // メモリ属性
    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    uint32_t successAttr[] = {
        0,
        nn::svc::MemoryAttribute_Locked,
        nn::svc::MemoryAttribute_DeviceShared,
    };
    const int NumSuccessAttr = sizeof(successAttr) / sizeof(nn::svc::MemoryAttribute);

    TestMemoryInfo* successAttrArea[NumSuccessAttr];
    header->GetTestListWithAttributes(successAttrArea, successAttr, NumSuccessAttr);

    const int NumInvalidAttr = NumTestMemoryAttribute - NumSuccessAttr;
    TestMemoryInfo* invalidAttrArea[NumInvalidAttr];
    header->GetTestListExceptAttributes(
            invalidAttrArea, NumInvalidAttr, successAttr, NumSuccessAttr);

    CheckSuccessReceiveListMemoryFromServer(successAttrArea, NumSuccessAttr, serverSession, userBuffer);
    // サーバーは invalid をチェックしない

    // AliasCode のバリエーション
    {
        nn::svc::MemoryPermission perms[] = {
            nn::svc::MemoryPermission_None,
#ifdef SVC_AGING_TEST
            nn::svc::MemoryPermission_Read,
            nn::svc::MemoryPermission_ReadExecute,
#endif
            nn::svc::MemoryPermission_ReadWrite,
        };
        const int NumPerms = sizeof(perms) / sizeof(nn::svc::MemoryPermission);

        for (int i = 0; i < NumPerms; i++)
        {
            TestAliasCodeMemoryState aliasCode;
            aliasCode.SetPermission(perms[i]);
            TestMemoryInfo* memoryInfo = &aliasCode;
            if (perms[i] == nn::svc::MemoryPermission_ReadWrite)
            {
                CheckSuccessReceiveListMemoryFromServer(
                        &memoryInfo, 1, serverSession, userBuffer);
            }
            // サーバーは invalid をチェックしない
        }
    }

    result = ReceiveIpc(&serverSession, 1, pMsgBuffer, msgSize, userBuffer);
    ASSERT_RESULT_SUCCESS(result);
    ipcMsg.Set(IpcEndHeader);
    result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
    assertObj.Cancel();
} // NOLINT(readability/fn_size)

void CheckReceiveListFromServer(nn::svc::Handle clientSession, size_t dataSize, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    size_t size = 0;

    // サーバー側がポインタ転送に用いるバッファのメモリ状態をチェック
    GetIpcBuffer(IpcClient, &pMsgBuffer, &msgSize, userBuffer);
    TestIpcAssertObject assertObj(clientSession, pMsgBuffer, msgSize);

    bool isEnd = false;
    while (!isEnd)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
        nn::svc::ipc::MessageBuffer::MessageHeader tmpHeader(0x0001, 0, 0, 0, 0, 0, 0, 0);
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 1, 0, 0, 0, 0, 1);
        nn::svc::ipc::MessageBuffer::PointerData
            ptrData(reinterpret_cast<char*>(nnMain), dataSize, 0);

        int offset = 0;

        ipcMsg.Set(tmpHeader);
        nn::Result result = SendIpc(clientSession, pMsgBuffer, msgSize, userBuffer, async);
        ASSERT_RESULT_SUCCESS(result);

        switch(GetIpcTag(pMsgBuffer))
        {
            case IpcEndHeaderTag:
                isEnd = true;
                break;
            case IpcSuccessHeaderTag:
                for (;;)
                {
                    ClientReceiveData(&size, sizeof(size), clientSession);
                    if (size == 0)
                    {
                        break;
                    }
                    CallReceiveListAlignTest(
                            IpcClient, clientSession,
                            g_HeapPtr + g_HeapSize - 0x5000, size,
                            -1, -1, // サーバー側は使わない
                            userBuffer, async);
                }
                break;
            case IpcFaildHeaderTag:
                for (;;)
                {
                    offset = ipcMsg.Set(ipcHeader);
                    offset = ipcMsg.Set(offset, ptrData);
                    result = SendIpc(clientSession, pMsgBuffer, msgSize, userBuffer, async);
                    if (result.IsSuccess())
                    {
                        if(GetIpcTag(pMsgBuffer) == IpcEndHeaderTag)
                        {
                            break;
                        }
                        else
                        {
                            if (async)
                            {
                                CheckAsyncFailure(pMsgBuffer);
                                continue;
                            }
                            else
                            {
                                FAIL();
                            }
                        }
                    }
                    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
                }
                break;
            default:
                FAIL();
        }
    }
    assertObj.Cancel();
}

void ReceiveListMemoryAreaTestClient(nn::svc::Handle clientSession, bool userBuffer, bool async)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    size_t dataSize = 0x10;
    GetIpcBuffer(IpcClient, &pMsgBuffer, &msgSize, userBuffer);
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

    TestIpcAssertObject assertObj(clientSession, pMsgBuffer, msgSize);

    // メモリ状態
    nn::svc::MemoryState invalidState[] = {
        nn::svc::MemoryState_Io,
#ifndef SUPPORT_IPC_WITH_STATIC
        nn::svc::MemoryState_Static,
#endif // SUPPORT_IPC_WITH_STATIC
        nn::svc::MemoryState_Free, // Permision=None
        nn::svc::MemoryState_Code, // Permision=ReadExecute
        nn::svc::MemoryState_AliasCode, // Permision!=ReadWrite
        nn::svc::MemoryState_Inaccessible,
    };

    const int NumInvalidState = sizeof(invalidState) / sizeof(nn::svc::MemoryState);

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    TestMemoryInfo* invalidArea[NumInvalidState];
    header->GetTestListWithStates(invalidArea, invalidState, NumInvalidState);

    const int NumSuccessState = NumTestMemoryState - NumInvalidState;
    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListExceptStates(successArea, NumSuccessState, invalidState, NumInvalidState);

    CheckSuccessReceiveListMemoryFromClient(
            successArea, NumSuccessState, clientSession, userBuffer, async);
    CheckInvalidReceiveListMemoryFromClient(
            invalidArea, NumInvalidState, clientSession, userBuffer, async);

    // メモリ属性
    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    uint32_t successAttr[] = {
        0,
        nn::svc::MemoryAttribute_Locked,
        nn::svc::MemoryAttribute_DeviceShared,
    };
    const int NumSuccessAttr = sizeof(successAttr) / sizeof(nn::svc::MemoryAttribute);

    TestMemoryInfo* successAttrArea[NumSuccessAttr];
    header->GetTestListWithAttributes(successAttrArea, successAttr, NumSuccessAttr);

    const int NumInvalidAttr = NumTestMemoryAttribute - NumSuccessAttr;
    TestMemoryInfo* invalidAttrArea[NumInvalidAttr];
    header->GetTestListExceptAttributes(
            invalidAttrArea, NumInvalidAttr, successAttr, NumSuccessAttr);

    CheckSuccessReceiveListMemoryFromClient(
            successAttrArea, NumSuccessAttr, clientSession, userBuffer, async);
    CheckInvalidReceiveListMemoryFromClient(
            invalidAttrArea, NumInvalidAttr, clientSession, userBuffer, async);

    // AliasCode のバリエーション
    {
        nn::svc::MemoryPermission perms[] = {
            nn::svc::MemoryPermission_None,
#ifdef SVC_AGING_TEST
            nn::svc::MemoryPermission_Read,
            nn::svc::MemoryPermission_ReadExecute,
#endif
            nn::svc::MemoryPermission_ReadWrite,
        };
        const int NumPerms = sizeof(perms) / sizeof(nn::svc::MemoryPermission);

        for (int i = 0; i < NumPerms; i++)
        {
            TestAliasCodeMemoryState aliasCode;
            aliasCode.SetPermission(perms[i]);
            TestMemoryInfo* memoryInfo = &aliasCode;
            if (perms[i] == nn::svc::MemoryPermission_ReadWrite)
            {
                CheckSuccessReceiveListMemoryFromClient(
                        &memoryInfo, 1, clientSession, userBuffer, async);
            }
            else
            {
                CheckInvalidReceiveListMemoryFromClient(
                        &memoryInfo, 1, clientSession, userBuffer, async);
            }
        }
    }

    ipcMsg.Set(IpcEndHeader);
    result = SendIpc(clientSession, pMsgBuffer, msgSize, userBuffer, async);
    ASSERT_RESULT_SUCCESS(result);

    CheckReceiveListFromServer(clientSession, dataSize, userBuffer, async);
    assertObj.Cancel();
} // NOLINT(readability/fn_size)

void ReceiveListMemoryAreaTest(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    if (x == IpcClient)
    {
        SCOPED_TRACE(__func__);
        NN_LOG("[%s]\n", __func__);
    }

    if (x == IpcServer)
    {
        ReceiveListMemoryAreaTestServer(handle, userBuffer);
    }
    else
    {
        ReceiveListMemoryAreaTestClient(handle, userBuffer, async);
    }
}

struct MapAreaTestData
{
    uintptr_t addr;
    size_t size;
    MapDataType dataType;
    int mapType;
};

void CheckSuccessMapArea(
        TestMemoryInfo** array, int numArray, nn::svc::Handle clientSession,
        bool userBuffer, bool async, int mapAttr)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(IpcClient, &pMsgBuffer, &msgSize, userBuffer);

    TestIpcAssertObject assertObj(clientSession, pMsgBuffer, msgSize);

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

    // 開始要求
    {
        ipcMsg.Set(IpcSuccessHeader);
        result = SendIpc(clientSession, pMsgBuffer, msgSize, userBuffer, async);
        ASSERT_RESULT_SUCCESS(result);
    }

    MapAreaTestData data;

    for(int i = 0; i < numArray; i++)
    {
        if (array[i]->GetSize() == 0)
        {
            array[i]->Close();
            continue;
        }

        array[i]->SetHeap(g_HeapPtr, g_HeapSize);
        array[i]->Initialize();

        const MapDataType DataTypes[] = { MapDataType_Send, MapDataType_Receive, MapDataType_Exchange };
        for (int j = 0; j < sizeof(DataTypes) / sizeof(DataTypes[0]); j++)
        {
            if (array[i]->GetPermission() != nn::svc::MemoryPermission_ReadWrite && DataTypes[j] != MapDataType_Send)
            {
                continue;
            }
            data.addr = array[i]->GetAddress();
            data.size = array[i]->GetSize();
            data.dataType = DataTypes[j];
            data.mapType = mapAttr;
            ClientSendData(clientSession, &data, sizeof(data));
            CallMapAlignTest(IpcClient, clientSession, array[i]->GetAddress(), array[i]->GetSize(), DataTypes[j], mapAttr, userBuffer, async);
        }

        array[i]->Close();
    }

    // 終了通知
    data.addr = 0;
    data.size = 0;
    ClientSendData(clientSession, &data, sizeof(data));
    assertObj.Cancel();
}

void CheckInvalidMapArea(
        TestMemoryInfo** array, int numArray, nn::svc::Handle clientSession,
        bool userBuffer, bool async, int mapAttr)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(IpcClient, &pMsgBuffer, &msgSize, userBuffer);

    TestIpcAssertObject assertObj(clientSession, pMsgBuffer, msgSize);

    // 開始の合図
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    {
        ipcMsg.Set(IpcFaildHeader);
        result = SendIpc(clientSession, pMsgBuffer, msgSize, userBuffer, async);
        ASSERT_RESULT_SUCCESS(result);
    }

    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 0, 1, 0, 0, 0, 0);

    // クライアント側がポインタ転送に用いるバッファのメモリ属性をチェック
    for(int i = 0; i < numArray; i++)
    {
        if (array[i]->GetSize() == 0)
        {
            array[i]->Close();
            continue;
        }

        array[i]->SetHeap(g_HeapPtr, g_HeapSize);
        array[i]->Initialize();
        uintptr_t addr = array[i]->GetAddress();
        size_t size = 0x10;

        int offset;

        // 4kb に沿っていない場合
        offset = ipcMsg.Set(ipcHeader);

        nn::svc::ipc::MessageBuffer::MapData mapData(reinterpret_cast<void*>(addr), size);
        offset = ipcMsg.Set(offset, mapData);
        pMsgBuffer[offset - 1] |= mapAttr;

        array[i]->CheckDefaultState();
        result = SendIpc(clientSession, pMsgBuffer, msgSize, userBuffer, async);
        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }

        array[i]->CheckDefaultState();

        // 4kb に沿っている場合
        bool check = false;
        if (array[i]->GetSize() >= 0x1000)
        {
            check = true;
            size = 0x1000;
            if ((addr & 0xfff) > 0)
            {
                addr = (addr + 0xfff) & ~(0xfff);
                if ((size + (addr - array[i]->GetAddress())) <= array[i]->GetSize())
                {
                    check = false;
                }

            }
        }
        if (check)
        {
            offset = ipcMsg.Set(ipcHeader);

            nn::svc::ipc::MessageBuffer::MapData mapData(reinterpret_cast<void*>(addr), size);
            offset = ipcMsg.Set(offset, mapData);
            pMsgBuffer[offset - 1] |= mapAttr;

            result = SendIpc(clientSession, pMsgBuffer, msgSize, userBuffer, async);
            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
            }

            array[i]->CheckDefaultState();
        }

        array[i]->Close();
    }

    // 終了の通知
    {
        ipcMsg.Set(IpcEndHeader);
        result = SendIpc(clientSession, pMsgBuffer, msgSize, userBuffer, async);
        ASSERT_RESULT_SUCCESS(result);
    }
    assertObj.Cancel();
}

void MapMemoryAreaTestServer(nn::svc::Handle serverSession, bool userBuffer)
{
    nn::Result result;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(IpcServer, &pMsgBuffer, &msgSize, userBuffer);
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

    TestIpcAssertObject assertObj(serverSession, pMsgBuffer, msgSize);

    for(;;)
    {
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 0, 0, 0, 0, 0, 0);
        ipcMsg.Set(ipcHeader);

        result = ReceiveIpc(&serverSession, 1, pMsgBuffer, msgSize, userBuffer);
        if (result.IsSuccess())
        {
            if (GetIpcTag(pMsgBuffer) == IpcSuccessHeaderTag)
            {
                ipcMsg.SetNull();
                result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
                MapAreaTestData data;

                for (;;)
                {
                    ServerReceiveData(&data, sizeof(data), serverSession);
                    if (data.addr == 0 && data.size == 0)
                    {
                        break;
                    }
                    CallMapAlignTest(IpcServer, serverSession, data.addr, data.size, data.dataType, data.mapType, userBuffer, false);
                }
            }
            else if (GetIpcTag(pMsgBuffer) == IpcFaildHeaderTag)
            {
                ipcMsg.SetNull();
                result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

                bool isEnd = false;
                while (!isEnd)
                {
                    ipcMsg.SetNull();
                    result = ReceiveIpc(&serverSession, 1, pMsgBuffer, msgSize, userBuffer);

                    ASSERT_TRUE(result.IsSuccess() ||
                                result <= nn::svc::ResultReceptionSpecificationListBroken());

                    if (result.IsSuccess())
                    {
                        if (GetIpcTag(pMsgBuffer) == IpcEndHeaderTag)
                        {
                            isEnd = true;
                        }
                        ipcMsg.SetNull();
                        result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
                        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
                    }
                }
            }
            else
            {
                if (GetIpcTag(pMsgBuffer) == IpcEndHeaderTag)
                {
                    break;
                }
                ipcMsg.Set(ipcHeader);
                result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
            }
        }
    }

    ipcMsg.SetNull();
    result = ReplyIpc(serverSession, pMsgBuffer, msgSize, userBuffer);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    assertObj.Cancel();
}

void MapMemoryAreaTestClient(nn::svc::Handle clientSession, bool userBuffer, bool async)
{
    nn::Result result;
    const int mapAttr = 0;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(IpcClient, &pMsgBuffer, &msgSize, userBuffer);
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

    TestIpcAssertObject assertObj(clientSession, pMsgBuffer, msgSize);

    // メモリ状態
    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_AliasCode,
#ifdef SUPPORT_ALIAS_CODE_DATA
        nn::svc::MemoryState_AliasCodeData,
#endif
        nn::svc::MemoryState_Code,
        nn::svc::MemoryState_CodeData,
        nn::svc::MemoryState_Ipc,
        nn::svc::MemoryState_Normal,
        nn::svc::MemoryState_Stack,
        nn::svc::MemoryState_Transfered,
#ifndef FIX_IPC_SHARED_CODE
        nn::svc::MemoryState_SharedCode,
#endif // FIX_IPC_SHARED_CODE
    };
    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListWithStates(successArea, successState, NumSuccessState);

    const int NumInvalid = NumTestMemoryState - NumSuccessState;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successState, NumSuccessState);

    CheckSuccessMapArea(
            successArea, NumSuccessState, clientSession, userBuffer, async, mapAttr);
    CheckInvalidMapArea(
            invalidArea, NumInvalid, clientSession, userBuffer, async, mapAttr);

    // メモリ属性
    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    uint32_t successAttr[] = {
        0,
#ifdef SUPPORT_IPC_LOCKED
        nn::svc::MemoryAttribute_IpcLocked,
#endif
    };
    const int NumSuccessAttr = sizeof(successAttr) / sizeof(nn::svc::MemoryAttribute);

    TestMemoryInfo* successAttrArea[NumSuccessAttr];
    header->GetTestListWithAttributes(successAttrArea, successAttr, NumSuccessAttr);

    const int NumInvalidAttr = NumTestMemoryAttribute - NumSuccessAttr;
    TestMemoryInfo* invalidAttrArea[NumInvalidAttr];
    header->GetTestListExceptAttributes(
            invalidAttrArea, NumInvalidAttr, successAttr, NumSuccessAttr);

    CheckSuccessMapArea(successAttrArea, NumSuccessAttr, clientSession, userBuffer, async, mapAttr);
    CheckInvalidMapArea(invalidAttrArea, NumInvalidAttr, clientSession, userBuffer, async, mapAttr);

    // AliasCode のバリエーション
    {
        nn::svc::MemoryPermission perms[] = {
            nn::svc::MemoryPermission_None,
#ifdef SVC_AGING_TEST
            nn::svc::MemoryPermission_Read,
            nn::svc::MemoryPermission_ReadExecute,
#endif
            nn::svc::MemoryPermission_ReadWrite,
        };
        const int NumPerms = sizeof(perms) / sizeof(nn::svc::MemoryPermission);

        for (int i = 0; i < NumPerms; i++)
        {
            TestAliasCodeMemoryState aliasCode;
            aliasCode.SetPermission(perms[i]);
            TestMemoryInfo* memoryInfo = &aliasCode;
            if (perms[i] == nn::svc::MemoryPermission_None)
            {
                CheckInvalidMapArea(
                        &memoryInfo, 1, clientSession, userBuffer, async, mapAttr);
            }
            else
            {
                CheckSuccessMapArea(
                        &memoryInfo, 1, clientSession, userBuffer, async, mapAttr);
            }
        }
    }
    assertObj.Cancel();
}

void MapMemoryAreaTestClientNonSecure(nn::svc::Handle clientSession, bool userBuffer, bool async)
{
    nn::Result result;
    const int mapAttr = 1;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(IpcClient, &pMsgBuffer, &msgSize, userBuffer);
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

    TestIpcAssertObject assertObj(clientSession, pMsgBuffer, msgSize);

    // メモリ状態
    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_AliasCode,
#ifdef SUPPORT_ALIAS_CODE_DATA
        nn::svc::MemoryState_AliasCodeData,
#endif
        nn::svc::MemoryState_Code,
        nn::svc::MemoryState_CodeData,
        nn::svc::MemoryState_Ipc,
        nn::svc::MemoryState_Normal,
        nn::svc::MemoryState_Stack,
        nn::svc::MemoryState_Transfered,
        nn::svc::MemoryState_SharedCode,
        nn::svc::MemoryState_SharedTransfered,
        nn::svc::MemoryState_NonSecureIpc,
    };
    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListWithStates(successArea, successState, NumSuccessState);

    const int NumInvalid = NumTestMemoryState - NumSuccessState;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successState, NumSuccessState);

    CheckSuccessMapArea(
            successArea, NumSuccessState, clientSession, userBuffer, async, mapAttr);
    CheckInvalidMapArea(
            invalidArea, NumInvalid, clientSession, userBuffer, async, mapAttr);

    // メモリ属性
    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    uint32_t successAttr[] = {
        0,
#ifdef SUPPORT_IPC_LOCKED
        nn::svc::MemoryAttribute_IpcLocked,
        nn::svc::MemoryAttribute_IpcLocked | nn::svc::MemoryAttribute_DeviceShared,
#endif
        nn::svc::MemoryAttribute_DeviceShared,
    };
    const int NumSuccessAttr = sizeof(successAttr) / sizeof(nn::svc::MemoryAttribute);

    TestMemoryInfo* successAttrArea[NumSuccessAttr];
    header->GetTestListWithAttributes(successAttrArea, successAttr, NumSuccessAttr);

    const int NumInvalidAttr = NumTestMemoryAttribute - NumSuccessAttr;
    TestMemoryInfo* invalidAttrArea[NumInvalidAttr];
    header->GetTestListExceptAttributes(
            invalidAttrArea, NumInvalidAttr, successAttr, NumSuccessAttr);

    CheckSuccessMapArea(successAttrArea, NumSuccessAttr, clientSession, userBuffer, async, mapAttr);
    CheckInvalidMapArea(invalidAttrArea, NumInvalidAttr, clientSession, userBuffer, async, mapAttr);

    // AliasCode のバリエーション
    {
        nn::svc::MemoryPermission perms[] = {
            nn::svc::MemoryPermission_None,
#ifdef SVC_AGING_TEST
            nn::svc::MemoryPermission_Read,
            nn::svc::MemoryPermission_ReadExecute,
#endif
            nn::svc::MemoryPermission_ReadWrite,
        };
        const int NumPerms = sizeof(perms) / sizeof(nn::svc::MemoryPermission);

        for (int i = 0; i < NumPerms; i++)
        {
            TestAliasCodeMemoryState aliasCode;
            aliasCode.SetPermission(perms[i]);
            TestMemoryInfo* memoryInfo = &aliasCode;
            if (perms[i] == nn::svc::MemoryPermission_None)
            {
                CheckInvalidMapArea(
                        &memoryInfo, 1, clientSession, userBuffer, async, mapAttr);
            }
            else
            {
                CheckSuccessMapArea(
                        &memoryInfo, 1, clientSession, userBuffer, async, mapAttr);
            }
        }
    }
    assertObj.Cancel();
}

void MapMemoryAreaTestClientNonDeviceIpc(nn::svc::Handle clientSession, bool userBuffer, bool async)
{
    nn::Result result;
    const int mapAttr = 3;

    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(IpcClient, &pMsgBuffer, &msgSize, userBuffer);
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

    TestIpcAssertObject assertObj(clientSession, pMsgBuffer, msgSize);

    // メモリ状態
    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_AliasCode,
#ifdef SUPPORT_ALIAS_CODE_DATA
        nn::svc::MemoryState_AliasCodeData,
#endif
        nn::svc::MemoryState_Code,
        nn::svc::MemoryState_CodeData,
        nn::svc::MemoryState_Ipc,
        nn::svc::MemoryState_Normal,
        nn::svc::MemoryState_Stack,
        nn::svc::MemoryState_Transfered,
        nn::svc::MemoryState_SharedCode,
        nn::svc::MemoryState_SharedTransfered,
        nn::svc::MemoryState_NonSecureIpc,
        nn::svc::MemoryState_Alias,
        nn::svc::MemoryState_NonDeviceIpc,
    };
    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListWithStates(successArea, successState, NumSuccessState);

    const int NumInvalid = NumTestMemoryState - NumSuccessState;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successState, NumSuccessState);

    CheckSuccessMapArea(
            successArea, NumSuccessState, clientSession, userBuffer, async, mapAttr);
    CheckInvalidMapArea(
            invalidArea, NumInvalid, clientSession, userBuffer, async, mapAttr);

    // メモリ属性
    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));

    uint32_t successAttr[] = {
        0,
#ifdef SUPPORT_IPC_LOCKED
        nn::svc::MemoryAttribute_IpcLocked,
        nn::svc::MemoryAttribute_IpcLocked | nn::svc::MemoryAttribute_DeviceShared,
#endif
        nn::svc::MemoryAttribute_DeviceShared,
    };
    const int NumSuccessAttr = sizeof(successAttr) / sizeof(nn::svc::MemoryAttribute);

    TestMemoryInfo* successAttrArea[NumSuccessAttr];
    header->GetTestListWithAttributes(successAttrArea, successAttr, NumSuccessAttr);

    const int NumInvalidAttr = NumTestMemoryAttribute - NumSuccessAttr;
    TestMemoryInfo* invalidAttrArea[NumInvalidAttr];
    header->GetTestListExceptAttributes(
            invalidAttrArea, NumInvalidAttr, successAttr, NumSuccessAttr);

    CheckSuccessMapArea(successAttrArea, NumSuccessAttr, clientSession, userBuffer, async, mapAttr);
    CheckInvalidMapArea(invalidAttrArea, NumInvalidAttr, clientSession, userBuffer, async, mapAttr);

    // AliasCode のバリエーション
    {
        nn::svc::MemoryPermission perms[] = {
            nn::svc::MemoryPermission_None,
#ifdef SVC_AGING_TEST
            nn::svc::MemoryPermission_Read,
            nn::svc::MemoryPermission_ReadExecute,
#endif
            nn::svc::MemoryPermission_ReadWrite,
        };
        const int NumPerms = sizeof(perms) / sizeof(nn::svc::MemoryPermission);

        for (int i = 0; i < NumPerms; i++)
        {
            TestAliasCodeMemoryState aliasCode;
            aliasCode.SetPermission(perms[i]);
            TestMemoryInfo* memoryInfo = &aliasCode;
            if (perms[i] == nn::svc::MemoryPermission_None)
            {
                CheckInvalidMapArea(
                        &memoryInfo, 1, clientSession, userBuffer, async, mapAttr);
            }
            else
            {
                CheckSuccessMapArea(
                        &memoryInfo, 1, clientSession, userBuffer, async, mapAttr);
            }
        }
    }
    assertObj.Cancel();
}

void MapMemoryAreaTest(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    if (x == IpcClient)
    {
        SCOPED_TRACE(__func__);
        NN_LOG("[%s]\n", __func__);
    }

    if (x == IpcServer)
    {
        MapMemoryAreaTestServer(handle, userBuffer);
    }
    else
    {
        MapMemoryAreaTestClient(handle, userBuffer, async);
        MapMemoryAreaTestClientNonSecure(handle, userBuffer, async);
        MapMemoryAreaTestClientNonDeviceIpc(handle, userBuffer, async);

        // 終了通知
        {
            nn::Bit32* pMsgBuffer;
            size_t msgSize;
            GetIpcBuffer(IpcClient, &pMsgBuffer, &msgSize, userBuffer);
            TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            ipcMsg.Set(IpcEndHeader);
            nn::Result result = SendIpc(handle, pMsgBuffer, msgSize, userBuffer, async);
            ASSERT_RESULT_SUCCESS(result);
            assertObj.Cancel();
        }
    }
}

void DiscontinuousMapAreaTest(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    if (x == IpcClient)
    {
        SCOPED_TRACE(__func__);
        NN_LOG("[%s]\n", __func__);
    }

    const size_t blockSize = 0x1000;
    const int blockNum = 5;

    // 未使用領域を作るために、0x1000 分ずらす
    uintptr_t addr = g_FreeAreaBegin + 0x1000;

    if (x == IpcClient)
    {
        WaitHeapState(g_HeapPtr, (2 * blockNum - 1) * blockSize);
        MapDiscontiguousArea(
                g_HeapPtr, (2 * blockNum - 1) * blockSize,
                g_FreeAreaBegin, blockNum * blockSize, blockNum);
    }

    MapDataType dataTypes[] = { MapDataType_Send, MapDataType_Receive, MapDataType_Exchange };
    int mapType[] = { 0, 1, 3 };
    for (int i = 0; i < sizeof(dataTypes) / sizeof(dataTypes[0]); i++)
    {
        for (int j = 0; j < sizeof(mapType) / sizeof(mapType[0]); j++)
        {
            // 端数内にマップサイズが収まる
            MapDataAlignTestImpl(x, handle, addr + 0xF00, 0x200, g_FreeAreaBegin, 0x5000, dataTypes[i], mapType[j], userBuffer, async);

            // 前方の端数部分に非連続部分が来る
            MapDataAlignTestImpl(x, handle, addr + 0xF00, 0x1100, g_FreeAreaBegin, 0x5000, dataTypes[i], mapType[j], userBuffer, async);

            // 後方の端数部分に非連続部分が来る
            MapDataAlignTestImpl(x, handle, addr, 0x1100, g_FreeAreaBegin, 0x5000, dataTypes[i], mapType[j], userBuffer, async);

            // ページの切れ目に非連続部分が来る
            MapDataAlignTestImpl(x, handle, addr, 0x2000, g_FreeAreaBegin, 0x5000, dataTypes[i], mapType[j], userBuffer, async);

            // 前後にはみ出る
            MapDataAlignTestImpl(x, handle, addr + 0x100, 0x2000, g_FreeAreaBegin, 0x5000, dataTypes[i], mapType[j], userBuffer, async);
        }
    }

    if (x == IpcClient)
    {
        UnmapDiscontiguousArea(
                g_HeapPtr, (2 * blockNum - 1) * blockSize,
                g_FreeAreaBegin, blockNum * blockSize, blockNum);
    }
}

// 受信指定リストのオフセットがデータと連続していなくても、ポインタデータを受信できる
void ReceiveListOffsetTest(
        const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(x, &pMsgBuffer, &msgSize, userBuffer);
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;

    if (x == IpcClient)
    {
        SCOPED_TRACE(__func__);
        NN_LOG("[%s]\n", __func__);
    }

    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 1, 0, 0, 0, 0, 2);
            offset = ipcMsg.Set(ipcHeader);

            nn::svc::ipc::MessageBuffer::PointerData
                pointerData(reinterpret_cast<char*>(nnMain), 0x10, 0);
            offset = ipcMsg.Set(offset, pointerData);

            offset += 2;
            pMsgBuffer[1] |= offset << 20;
            nn::svc::ipc::MessageBuffer::ReceiveListEntry
                recvList(g_RecvBuffer1, sizeof(g_RecvBuffer1));
            offset = ipcMsg.Set(offset, recvList);

            result = SendIpc(handle, pMsgBuffer, msgSize, userBuffer, async);
            ASSERT_RESULT_SUCCESS(result);
        }

    }

    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 2);
            int offset = ipcMsg.Set(ipcHeader);

            offset += 2;
            pMsgBuffer[1] |= offset << 20;
            nn::svc::ipc::MessageBuffer::ReceiveListEntry
                recvList(g_RecvBuffer0, sizeof(g_RecvBuffer0));
            offset = ipcMsg.Set(offset, recvList);
        }
        result = ReceiveIpc(&handle, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_SUCCESS(result);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0x1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 2);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
#ifdef SUPPORT_RECEIVELIST_OFFSET
            ASSERT_TRUE(offset > 2);
#endif
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);

            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == 0x10);
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer0) <= pointer.GetPointerAddress());
            ASSERT_TRUE(
                    pointer.GetPointerAddress()
                    < reinterpret_cast<uintptr_t>(g_RecvBuffer0) + sizeof(g_RecvBuffer0));
            ASSERT_TRUE(
                    ::std::memcmp(
                        reinterpret_cast<void*>(pointer.GetPointerAddress()),
                        reinterpret_cast<void*>(nnMain),
                        pointer.GetPointerSize())
                    == 0);
        }

        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0002, 0, 1, 0, 0, 0, 0, 0);
            int offset = ipcMsg.Set(ipcHeader);

            nn::svc::ipc::MessageBuffer::PointerData
                pointerData(reinterpret_cast<char*>(nnMain) + 0x10, 0x10, 0);
            offset = ipcMsg.Set(offset, pointerData);
        }

        result = ReplyIpc(handle, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }

    if (x == IpcClient)
    {
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0x1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
#ifdef SUPPORT_RECEIVELIST_OFFSET
            ASSERT_TRUE(offset > 2);
#endif
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);

            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == 0x10);
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer1) <= pointer.GetPointerAddress());
            ASSERT_TRUE(
                    pointer.GetPointerAddress()
                    < reinterpret_cast<uintptr_t>(g_RecvBuffer1) + sizeof(g_RecvBuffer1));
            ASSERT_TRUE(
                    ::std::memcmp(
                        reinterpret_cast<void*>(pointer.GetPointerAddress()),
                        reinterpret_cast<char*>(nnMain) + 0x10,
                        pointer.GetPointerSize())
                    == 0);
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// 受信指定リストのオフセットがメッセージバッファに収まっていないと失敗する
// Send時
void InvalidReceiveListOffsetTestOnSend(
        const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(x, &pMsgBuffer, &msgSize, userBuffer);
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;

    if (x == IpcClient)
    {
        SCOPED_TRACE(__func__);
        NN_LOG("[%s]\n", __func__);
    }

    if (msgSize > 0x1000)
    {
        msgSize = 0x1000;
    }

    // Send 時の受信指定リスト
    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 1, 0, 0, 0, 0, 2);
            offset = ipcMsg.Set(ipcHeader);

            nn::svc::ipc::MessageBuffer::PointerData
                pointerData(reinterpret_cast<char*>(nnMain), 0x10, 0);
            offset = ipcMsg.Set(offset, pointerData);

            offset = msgSize / sizeof(nn::Bit32);
            pMsgBuffer[1] |= offset << 20;

            nn::svc::ipc::MessageBuffer::MessageHeader toCheckIpcHeader(ipcMsg);
            ASSERT_TRUE(toCheckIpcHeader.GetReceiveListOffset() == offset);

            result = SendIpc(handle, pMsgBuffer, msgSize, userBuffer, async);
            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
            }
        }

    }

    // サーバー側を動かすためのメッセージ
    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0002, 0, 1, 0, 0, 0, 0, 2);
            offset = ipcMsg.Set(ipcHeader);

            nn::svc::ipc::MessageBuffer::PointerData
                pointerData(reinterpret_cast<char*>(nnMain), 0x10, 0);
            offset = ipcMsg.Set(offset, pointerData);

            nn::svc::ipc::MessageBuffer::ReceiveListEntry
                recvList(g_RecvBuffer1, sizeof(g_RecvBuffer1));
            offset = ipcMsg.Set(offset, recvList);

            result = SendIpc(handle, pMsgBuffer, msgSize, userBuffer, async);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 2);
            int offset = ipcMsg.Set(ipcHeader);

            nn::svc::ipc::MessageBuffer::ReceiveListEntry
                recvList(g_RecvBuffer0, sizeof(g_RecvBuffer0));
            offset = ipcMsg.Set(offset, recvList);
        }
        result = ReceiveIpc(&handle, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_SUCCESS(result);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 2);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);

            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == 0x10);
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer0) <= pointer.GetPointerAddress());
            ASSERT_TRUE(
                    pointer.GetPointerAddress()
                    < reinterpret_cast<uintptr_t>(g_RecvBuffer0) + sizeof(g_RecvBuffer0));
            ASSERT_TRUE(
                    ::std::memcmp(
                        reinterpret_cast<void*>(pointer.GetPointerAddress()),
                        reinterpret_cast<void*>(nnMain),
                        pointer.GetPointerSize())
                    == 0);
        }

        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0003, 0, 1, 0, 0, 0, 0, 0);
            int offset = ipcMsg.Set(ipcHeader);

            nn::svc::ipc::MessageBuffer::PointerData
                pointerData(reinterpret_cast<char*>(nnMain) + 0x10, 0x10, 0);
            offset = ipcMsg.Set(offset, pointerData);
        }

        result = ReplyIpc(handle, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }

    if (x == IpcClient)
    {
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0003);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);

            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == 0x10);
            ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer1) <= pointer.GetPointerAddress());
            ASSERT_TRUE(
                    pointer.GetPointerAddress()
                    < reinterpret_cast<uintptr_t>(g_RecvBuffer1) + sizeof(g_RecvBuffer1));
            ASSERT_TRUE(
                    ::std::memcmp(
                        reinterpret_cast<void*>(pointer.GetPointerAddress()),
                        reinterpret_cast<char*>(nnMain) + 0x10,
                        pointer.GetPointerSize())
                    == 0);
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// 受信指定リストのオフセットがメッセージバッファに収まっていないと失敗する
// Recieve時
void InvalidReceiveListOffsetTestOnRecieve(
        const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(x, &pMsgBuffer, &msgSize, userBuffer);
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;

    if (x == IpcClient)
    {
        SCOPED_TRACE(__func__);
        NN_LOG("[%s]\n", __func__);
    }

    if (msgSize > 0x1000)
    {
        msgSize = 0x1000;
    }

    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.SetNull();
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 1, 0, 0, 0, 0, 1);
            offset = ipcMsg.Set(ipcHeader);

            nn::svc::ipc::MessageBuffer::PointerData
                pointerData(reinterpret_cast<char*>(nnMain), 0x10, 0);
            offset = ipcMsg.Set(offset, pointerData);

            nn::svc::ipc::MessageBuffer::ReceiveListEntry
                recvList(g_RecvBuffer1, sizeof(g_RecvBuffer1));
            offset = ipcMsg.Set(offset, recvList);

            result = SendIpc(handle, pMsgBuffer, msgSize, userBuffer, async);

            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
            }
        }
    }

    // サーバーを動かす
    if (x == IpcClient)
    {
        {
            result = nn::svc::CancelSynchronization(g_ServerHandle);
            ASSERT_RESULT_SUCCESS(result);
            // 必ずサーバーに移ることを保証する
            nn::svc::SleepThread(100 * 1000 * 1000);
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0000, 0, 0, 0, 0, 0, 0, 2);
            int offset = ipcMsg.Set(ipcHeader);

            offset = msgSize / sizeof(nn::Bit32);
            pMsgBuffer[1] |= offset << 20;
            nn::svc::ipc::MessageBuffer::MessageHeader toCheckIpcHeader(ipcMsg);
            ASSERT_TRUE(toCheckIpcHeader.GetReceiveListOffset() == offset);
        }

        result = ReceiveIpc(&handle, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultCancelled());
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void InvalidPointerIndex(
        const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Bit32* pMsgBuffer;
    size_t msgSize;
    GetIpcBuffer(x, &pMsgBuffer, &msgSize, userBuffer);
    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    nn::Result result;

    if (x == IpcClient)
    {
        SCOPED_TRACE(__func__);
        NN_LOG("[%s]\n", __func__);
    }

    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 2, 0, 0, 0, 0, 3);
            offset = ipcMsg.Set(ipcHeader);

            // インデックスを不正な値にする
            nn::svc::ipc::MessageBuffer::PointerData
                pointerData1(reinterpret_cast<char*>(nnMain), 0x10, 0);
            offset = ipcMsg.Set(offset, pointerData1);

            nn::svc::ipc::MessageBuffer::PointerData
                pointerData2(reinterpret_cast<char*>(nnMain) + 0x10, 0x10, 1);
            offset = ipcMsg.Set(offset, pointerData2);

            nn::svc::ipc::MessageBuffer::ReceiveListEntry
                recvList(g_RecvBuffer1, sizeof(g_RecvBuffer1));
            offset = ipcMsg.Set(offset, recvList);

            result = SendIpc(handle, pMsgBuffer, msgSize, userBuffer, async);
            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            }
        }
    }

    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 3);
            int offset = ipcMsg.Set(ipcHeader);

            nn::svc::ipc::MessageBuffer::ReceiveListEntry
                recvList(g_RecvBuffer0, sizeof(g_RecvBuffer0));
            offset = ipcMsg.Set(offset, recvList);
        }
        result = ReceiveIpc(&handle, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_TRUE(result.IsSuccess() ||
                result <= nn::svc::ResultTimeout() ||
                result <= nn::svc::ResultReceptionSpecificationListBroken());
    }

    if (x == IpcClient)
    {
        {
            int offset;
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0002, 0, 2, 0, 0, 0, 0, 3);
            offset = ipcMsg.Set(ipcHeader);

            nn::svc::ipc::MessageBuffer::PointerData
                pointerData1(reinterpret_cast<char*>(nnMain), 0x10, 0);
            offset = ipcMsg.Set(offset, pointerData1);

            nn::svc::ipc::MessageBuffer::PointerData
                pointerData2(reinterpret_cast<char*>(nnMain) + 0x10, 0x10, 1);
            offset = ipcMsg.Set(offset, pointerData2);


            nn::svc::ipc::MessageBuffer::ReceiveListEntry
                recvList(g_RecvBuffer1, sizeof(g_RecvBuffer1));
            offset = ipcMsg.Set(offset, recvList);

            result = SendIpc(handle, pMsgBuffer, msgSize, userBuffer, async);
            if (async)
            {
                ASSERT_RESULT_SUCCESS(result);
                CheckAsyncFailure(pMsgBuffer);
            }
            else
            {
                ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfResource());
            }
        }
    }

    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0, 0, 0, 0, 0, 0, 0, 2);
            int offset = ipcMsg.Set(ipcHeader);

            nn::svc::ipc::MessageBuffer::ReceiveListEntry
                recvList(g_RecvBuffer0, sizeof(g_RecvBuffer0));
            offset = ipcMsg.Set(offset, recvList);
        }
        result = ReceiveIpc(&handle, 1, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_SUCCESS(result);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 2);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 3);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);

            for (int i = 0; i < 2; i++)
            {
                nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                ASSERT_TRUE(pointer.GetPointerIndex() == i);
                ASSERT_TRUE(pointer.GetPointerSize() == 0x10);
                ASSERT_TRUE(reinterpret_cast<uintptr_t>(g_RecvBuffer0) <= pointer.GetPointerAddress());
                ASSERT_TRUE(
                        pointer.GetPointerAddress()
                        < reinterpret_cast<uintptr_t>(g_RecvBuffer0) + sizeof(g_RecvBuffer0));
                ASSERT_TRUE(
                        ::std::memcmp(
                            reinterpret_cast<void*>(pointer.GetPointerAddress()),
                            reinterpret_cast<char*>(nnMain) + 0x10 * i,
                            pointer.GetPointerSize())
                        == 0);
            }
        }

        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0003, 0, 2, 0, 0, 0, 0, 0);
            int offset = ipcMsg.Set(ipcHeader);

            // インデックスを不正な値にする
            nn::svc::ipc::MessageBuffer::PointerData
                pointerData1(reinterpret_cast<char*>(nnMain) + 0x10, 0x10, 0);
            offset = ipcMsg.Set(offset, pointerData1);

            nn::svc::ipc::MessageBuffer::PointerData
                pointerData2(reinterpret_cast<char*>(nnMain) + 0x20, 0x10, 1);
            offset = ipcMsg.Set(offset, pointerData2);
        }

        result = ReplyIpc(handle, pMsgBuffer, msgSize, userBuffer);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// Reply 時に Send データを送ることが出来ない
void CannotReplySendData(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("SendDataMaxNum");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0x1, 0, 0, 0, 0));

        // Send 転送
        uintptr_t addr = reinterpret_cast<uintptr_t>(g_XferBuffer);
        CheckMemory(addr, nn::svc::MemoryState_CodeData, nn::svc::MemoryPermission_ReadWrite, 0);

        ::std::memcpy(g_XferBuffer, reinterpret_cast<void*>(nnMain), sizeof(g_XferBuffer));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(addr), 0x1000));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);

        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        uintptr_t buf;
        size_t size;
        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 1);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Send
            {
                int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
                nn::svc::ipc::MessageBuffer::MapData mapData0(offset, ipcMsg);
                ASSERT_TRUE(mapData0.GetDataSize() == 0x1000);
                ASSERT_TRUE(mapData0.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(nnMain), reinterpret_cast<void*>(mapData0.GetDataAddress()), mapData0.GetDataSize()) == 0);
                buf = mapData0.GetDataAddress();
                size = mapData0.GetDataSize();
            }
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 1, 0, 0, 0, 0));

            // Send
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(buf), size));
        }

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // 受信に失敗しているのでデータのチェックはしない
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// Reply 時に Receive データを送ることが出来ない
void CannotReplyReceiveData(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("SendDataMaxNum");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 1, 0, 0, 0));

        nn::svc::SleepThread(100 * 1000 * 1000);
        // Receive 転送
        uintptr_t addr = reinterpret_cast<uintptr_t>(g_XferBuffer);
        nn::svc::MemoryInfo info;
        GetMemoryInfo(&info, addr);
        CheckMemory(addr, nn::svc::MemoryState_CodeData, nn::svc::MemoryPermission_ReadWrite, 0);

        ::std::memset(g_XferBuffer, 0, sizeof(g_XferBuffer));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(addr), 0x1000));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);

        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        uintptr_t buf;
        size_t size;
        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 1);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Receive
            {
                int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
                nn::svc::ipc::MessageBuffer::MapData mapData0(offset, ipcMsg);
                ASSERT_TRUE(mapData0.GetDataSize() == 0x1000);
                ASSERT_TRUE(mapData0.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                buf = mapData0.GetDataAddress();
                size = mapData0.GetDataSize();
            }
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 1, 0, 0, 0));

            ::std::memcpy(reinterpret_cast<char*>(buf), reinterpret_cast<char*>(nnMain), size);
            // Receive
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(buf), size));
        }

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // 受信に失敗しているのでデータのチェックはしない
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// Reply 時に Exchange データを送ることが出来ない
void CannotReplyExchangeData(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE("SendDataMaxNum");
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 1, 0, 0));

        // Exchange 転送
        uintptr_t addr = reinterpret_cast<uintptr_t>(g_XferBuffer);
        CheckMemory(addr, nn::svc::MemoryState_CodeData, nn::svc::MemoryPermission_ReadWrite, 0);

        ::std::memcpy(g_XferBuffer, reinterpret_cast<char*>(nnMain), sizeof(g_XferBuffer));
        offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(addr), 0x1000));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);

        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCombination());
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        uintptr_t buf;
        size_t size;
        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 1);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Exchange
            {
                int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
                nn::svc::ipc::MessageBuffer::MapData mapData0(offset, ipcMsg);
                ASSERT_TRUE(mapData0.GetDataSize() == 0x1000);
                ASSERT_TRUE(mapData0.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                ASSERT_TRUE(std::memcmp(reinterpret_cast<char*>(mapData0.GetDataAddress()), reinterpret_cast<char*>(nnMain), mapData0.GetSize()) == 0);
                buf = mapData0.GetDataAddress();
                size = mapData0.GetDataSize();
            }
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 1, 0, 0));

            ::std::memcpy(reinterpret_cast<char*>(buf), reinterpret_cast<char*>(nnMain) + 0x100, size);

            // Exchange
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(buf), size));
        }

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // 受信に失敗しているのでデータのチェックはしない
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void SendCodeDataTest(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE(__func__);
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0x1, 0, 0, 0, 0));

        // Send 転送
        uintptr_t addr = reinterpret_cast<uintptr_t>(NotBool);
        nn::svc::MemoryInfo info;
        GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(addr));
        ASSERT_TRUE(info.state == nn::svc::MemoryState_Code);
        ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadExecute);
        ASSERT_TRUE(info.attribute == 0);
        ASSERT_TRUE(info.ipcCount == 0);

        ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(info.baseAddress), info.size));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // TEST 67-48, 68-95, 68-160
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0x1);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Send
            {
                int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
                uintptr_t addr = reinterpret_cast<uintptr_t>(NotBool);
                nn::svc::MemoryInfo info;
                GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(addr));
                addr = info.baseAddress;
                nn::svc::ipc::MessageBuffer::MapData mapData(offset, ipcMsg);
                ASSERT_TRUE(mapData.GetDataSize() == info.size);
                ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(addr), reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);
                ASSERT_TRUE(info.state == nn::svc::MemoryState_Code);
                ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadExecute);
                ASSERT_TRUE(info.attribute == nn::svc::MemoryAttribute_IpcLocked);

                // 関数が呼べる
                ASSERT_TRUE(NotBool(false));
            }
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
        }

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // 元に戻っているか
            uintptr_t addr = reinterpret_cast<uintptr_t>(NotBool);
            nn::svc::MemoryInfo info;
            GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(addr));
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Code);
            ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadExecute);
            ASSERT_TRUE(info.attribute == 0);
            ASSERT_TRUE(info.ipcCount == 0);
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void ReceiveCodeDataTest(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE(__func__);
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 1, 0, 0, 0));

        // Recv 転送
        uintptr_t addr = reinterpret_cast<uintptr_t>(NotBool);
        nn::svc::MemoryInfo info;
        GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(addr));
        ASSERT_TRUE(info.state == nn::svc::MemoryState_Code);
        ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadExecute);
        ASSERT_TRUE(info.attribute == 0);
        ASSERT_TRUE(info.ipcCount == 0);

        ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(info.baseAddress), info.size));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);

        // 失敗する
        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }

        // 元に戻っていることの確認
        addr = reinterpret_cast<uintptr_t>(NotBool);
        GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(addr));
        ASSERT_TRUE(info.state == nn::svc::MemoryState_Code);
        ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadExecute);
        ASSERT_TRUE(info.attribute == 0);
        ASSERT_TRUE(info.ipcCount == 0);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);

        ASSERT_TRUE(result.IsSuccess() ||
                    result <= nn::svc::ResultReceptionSpecificationListBroken());

        if (result.IsSuccess())
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));

            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

void ExchangeCodeDataTest(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE(__func__);
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0, 0, 1, 0, 0));

        // Exchange 転送
        uintptr_t addr = reinterpret_cast<uintptr_t>(NotBool);
        nn::svc::MemoryInfo info;
        GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(addr));
        ASSERT_TRUE(info.state == nn::svc::MemoryState_Code);
        ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadExecute);
        ASSERT_TRUE(info.attribute == 0);
        ASSERT_TRUE(info.ipcCount == 0);

        ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(info.baseAddress), info.size));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);

        // 失敗する
        if (async)
        {
            ASSERT_RESULT_SUCCESS(result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }

        // 元に戻っていることの確認
        addr = reinterpret_cast<uintptr_t>(NotBool);
        GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(addr));
        ASSERT_TRUE(info.state == nn::svc::MemoryState_Code);
        ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadExecute);
        ASSERT_TRUE(info.attribute == 0);
        ASSERT_TRUE(info.ipcCount == 0);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // TEST 67-48, 68-95, 68-160
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);

        ASSERT_TRUE(result.IsSuccess() ||
                    result <= nn::svc::ResultReceptionSpecificationListBroken());

        if (result.IsSuccess())
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));

            BeforeServerIpc(pMsgBuffer, msgSize);
            if (userBuffer)
            {
                result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
            }
            else
            {
                result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
            }
            AfterServerIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

nn::svc::MemoryInfo g_CodeBlock;

void SendExAndRoDataTest(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize;

    if (x == IpcClient)
    {
        SCOPED_TRACE(__func__);
        NN_LOG("[%s]\n", __func__);
    }

    if (userBuffer)
    {
        pMsgBuffer = (x == IpcClient? g_UserBuffer0: g_UserBuffer1);
        msgSize = (x == IpcClient)? sizeof(g_UserBuffer0): sizeof(g_UserBuffer1);
    }
    else
    {
        pMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        msgSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
    }

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);

    if (x == IpcClient)
    {
        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // ヘッダ
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0001, 0, 0, 0x1, 0, 0, 0, 0));

        // Send 転送
        uintptr_t addr = reinterpret_cast<uintptr_t>(NotBool);
        nn::svc::MemoryInfo info;
        GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(addr));
        ASSERT_TRUE(info.state == nn::svc::MemoryState_Code);
        ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadExecute);
        ASSERT_TRUE(info.attribute == 0);
        ASSERT_TRUE(info.ipcCount == 0);

        addr = info.baseAddress + info.size - 0x1000;
        g_CodeBlock = info;

        GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(addr + 0x1000));
        ASSERT_TRUE(info.state == nn::svc::MemoryState_Code);
        ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_Read);
        ASSERT_TRUE(info.attribute == 0);
        ASSERT_TRUE(info.ipcCount == 0);

        ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::MapData(reinterpret_cast<void*>(addr), 0x2000));

        BeforeClientIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            if (async)
            {
                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {
                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
        }
        else
        {
            result = nn::svc::SendSyncRequest(handle);
        }
        AfterClientIpc(pMsgBuffer, msgSize);

        // 成功する
        ASSERT_RESULT_SUCCESS(result);
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader());
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // TEST 67-48, 68-95, 68-160
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        }
        AfterServerIpc(pMsgBuffer, msgSize);

        ASSERT_RESULT_SUCCESS(result);

        // 受信データの確認
        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0001);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0x1);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // Send
            {
                int offset = nn::svc::ipc::MessageBuffer::GetMapDataOffset(ipcHeader, ipcSpecial);
                uintptr_t addr = g_CodeBlock.baseAddress + g_CodeBlock.size - 0x1000;
                nn::svc::ipc::MessageBuffer::MapData mapData(offset, ipcMsg);
                ASSERT_TRUE(mapData.GetDataSize() == 0x2000);
                ASSERT_TRUE(mapData.GetAttribute() == nn::svc::ipc::MessageBuffer::MapData::MapTransferAttribute_Ipc);
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(addr), reinterpret_cast<void*>(mapData.GetDataAddress()), mapData.GetDataSize()) == 0);

                nn::svc::MemoryInfo info;
                GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(addr));
                ASSERT_TRUE(info.state == nn::svc::MemoryState_Code);
                ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadExecute);
                ASSERT_TRUE(info.attribute == nn::svc::MemoryAttribute_IpcLocked);

                GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(addr + 0x1000));
                ASSERT_TRUE(info.state == nn::svc::MemoryState_Code);
                ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_Read);
                ASSERT_TRUE(info.attribute == nn::svc::MemoryAttribute_IpcLocked);
            }
        }

        // 送信データを作成
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

            // ヘッダ
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0002, 0, 0, 0, 0, 0, 0, 0));
        }

        // TEST 67-49, 68-96, 68-161
        // 最大個数の Send 結果を送信することが出来る
        BeforeServerIpc(pMsgBuffer, msgSize);
        if (userBuffer)
        {
            result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        }
        else
        {
            result = nn::svc::ReplyAndReceive(&index, handles, 0, handle, 0);
        }
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        // TEST 28-73, 29-86, 30-96
        // 最大個数の Send 結果を受信することが出来る
        {
            // ヘッダ情報が更新されている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0002);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            // 元に戻っているか
            uintptr_t addr = reinterpret_cast<uintptr_t>(NotBool);
            nn::svc::MemoryInfo info;
            GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(addr));
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Code);
            ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_ReadExecute);
            ASSERT_TRUE(info.attribute == 0);
            ASSERT_TRUE(info.ipcCount == 0);

            addr = info.baseAddress + info.size - 0x1000;

            GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(addr + 0x1000));
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Code);
            ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_Read);
            ASSERT_TRUE(info.attribute == 0);
            ASSERT_TRUE(info.ipcCount == 0);
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

/* Only Using UserBuffer Test */

void SendMaxPointerDataWithBuffer(const bool x, nn::svc::Handle handle, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize = 0x10000 * 0x10;

    pMsgBuffer = reinterpret_cast<nn::Bit32*>(x == IpcClient ? g_HeapPtr : g_HeapPtr + msgSize);

    if (x == IpcClient)
    {
        SCOPED_TRACE(__func__);
        NN_LOG("[%s]\n", __func__);
    }


    WaitHeapState(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize);

    TestIpcAssertObject assertObj(handle, pMsgBuffer, msgSize);
    if (x == IpcClient)
    {
        {
            int offset;
            // ポインタ転送にメッセージバッファを利用することが出来る
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0xFFFF, 0, 0xF, 0, 0, 0, 0, 1));
            for (int i = 0; i < 0xF; i++)
            {
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(reinterpret_cast<char*>(nnMain), 0xFFFF, i));
            }

            BeforeClientIpc(pMsgBuffer, msgSize);
            if (async)
            {

                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {

                result = nn::svc::SendSyncRequestWithUserBuffer(reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handle);
            }
            AfterClientIpc(pMsgBuffer, msgSize);
            ASSERT_RESULT_SUCCESS(result);
        }
    }
    if (x == IpcServer)
    {
        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 1));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // SendAsyncRequestWithUserBuffer から要求を受信できる

        // SendSyncRequestWithUserBuffer から要求を受信できる
        BeforeServerIpc(pMsgBuffer, msgSize);
        result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0xFFFF);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0xF);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            for (int i = 0; i < 0xF; i++)
            {
                nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                ASSERT_TRUE(pointer.GetPointerIndex() == i);
                ASSERT_TRUE(pointer.GetPointerSize() == 0xFFFF);
                ASSERT_TRUE(reinterpret_cast<uintptr_t>(pMsgBuffer) < pointer.GetPointerAddress());
                ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(pMsgBuffer) + msgSize);
                ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), reinterpret_cast<void*>(nnMain), pointer.GetPointerSize()) == 0);
            }
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0000, 0, 0xF, 0, 0, 0, 0, 0));

            for (int i = 0; i < 0xF; i++)
            {
                offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(reinterpret_cast<char*>(reinterpret_cast<uintptr_t>(nnMain) + 0x100), 0xFFFF, i));
            }
        }


        // 相手に reply を返すことが出来る
        BeforeServerIpc(pMsgBuffer, msgSize);
        result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(pMsgBuffer), msgSize, handles, 0, handle, 0);
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());
    }
    if (x == IpcClient)
    {
        {

            // 結果を受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0000);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 0xF);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送でデータを受信できている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            for (int i = 0; i < 0xF; i++)
            {
                nn::svc::ipc::MessageBuffer::PointerData pointer(offset + i * 2, ipcMsg);

                ASSERT_TRUE(pointer.GetPointerIndex() == i);
                ASSERT_TRUE(pointer.GetPointerSize() == 0xFFFF);
                ASSERT_TRUE(reinterpret_cast<uintptr_t>(pMsgBuffer) < pointer.GetPointerAddress());
                ASSERT_TRUE(pointer.GetPointerAddress() < reinterpret_cast<uintptr_t>(pMsgBuffer) + msgSize);
                ASSERT_TRUE(::std::memcmp(
                    reinterpret_cast<void*>(pointer.GetPointerAddress()),
                    reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(nnMain) + 0x100),
                    pointer.GetPointerSize()) == 0);
            }

            nn::svc::MemoryInfo info;
            GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(pMsgBuffer));
            ASSERT_TRUE(info.attribute != nn::svc::MemoryAttribute_Locked);
        }
    }
    assertObj.Cancel();
} // NOLINT (readability/fn_size)

// TEST 902-20
// ユーザーバッファに物理不連続な領域を指定することが出来る
volatile int g_Sequence = 0;
void UseDiscontinuoudUserBuffer(const bool x, nn::svc::Handle handle, bool async)
{
    nn::Result result;
    nn::Bit32* pMsgBuffer;
    size_t msgSize = 0x1000;
    size_t bufSize = 0x2000;
    uintptr_t addr;
    TestAliasCodeDataMemoryState prevBlock;
    TestAliasCodeDataMemoryState nextBlock;

    if (x == IpcClient)
    {
        SCOPED_TRACE(__func__);
        NN_LOG("[%s]\n", __func__);
    }

    if (x == IpcClient)
    {
        WaitHeapState(g_HeapPtr, 0x3000);

        prevBlock.SetSize(0x1000);
        prevBlock.SetHeap(g_HeapPtr, 0x1000);
        prevBlock.SetFree(g_FreeAreaBegin, 0x1000);
        prevBlock.Initialize();

        nextBlock.SetSize(0x1000);
        nextBlock.SetHeap(g_HeapPtr + 0x2000, 0x1000);
        nextBlock.SetFree(g_FreeAreaBegin + 0x1000, 0x1000);
        nextBlock.Initialize();

        g_Sequence = 1;

        addr = g_FreeAreaBegin;

        nn::svc::PhysicalMemoryInfo info;
        result = nn::svc::QueryPhysicalAddress(&info, addr);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t beforeArea = info.physicalAddress;

        result = nn::svc::QueryPhysicalAddress(&info, addr + 0x1000);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t afterArea = info.physicalAddress;

        ASSERT_TRUE(beforeArea != afterArea);

        pMsgBuffer = reinterpret_cast<nn::Bit32*>(addr);
        TestIpcAssertObject assertObj(handle, pMsgBuffer, bufSize);

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        {
            int offset;
            // ポインタ転送にメッセージバッファを利用することが出来る
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0xFFFF, 0, 1, 0, 0, 0, 0, 1));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(reinterpret_cast<char*>(nnMain), msgSize, 0));

            BeforeClientIpc(pMsgBuffer, bufSize);
            if (async)
            {

                nn::svc::Handle event;
                result = nn::svc::SendAsyncRequestWithUserBuffer(&event, addr, bufSize, handle);
                ASSERT_RESULT_SUCCESS(result);
                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &event, 1, -1);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(event);
            }
            else
            {

                result = nn::svc::SendSyncRequestWithUserBuffer(addr, bufSize, handle);
            }
            AfterClientIpc(pMsgBuffer, bufSize);
            ASSERT_RESULT_SUCCESS(result);
        }

        assertObj.Cancel();
    }
    if (x == IpcServer)
    {
        const uintptr_t BlockOffset = 0x5000;
        WaitHeapState(g_HeapPtr + BlockOffset, 0x3000);

        while(g_Sequence != 1)
        {
            nn::svc::SleepThread(SleepTime);
        }
        prevBlock.SetSize(0x1000);
        prevBlock.SetHeap(g_HeapPtr + BlockOffset, 0x1000);
        prevBlock.SetFree(g_FreeAreaBegin + BlockOffset, 0x1000);
        prevBlock.Initialize();

        nextBlock.SetSize(0x1000);
        nextBlock.SetHeap(g_HeapPtr + BlockOffset + 0x2000, 0x1000);
        nextBlock.SetFree(g_FreeAreaBegin + BlockOffset + 0x1000, 0x1000);
        nextBlock.Initialize();
        g_Sequence = 0;

        addr = g_FreeAreaBegin + BlockOffset;

        nn::svc::PhysicalMemoryInfo info;
        result = nn::svc::QueryPhysicalAddress(&info, addr);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t beforeArea = info.physicalAddress;

        result = nn::svc::QueryPhysicalAddress(&info, addr + 0x1000);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t afterArea = info.physicalAddress;

        ASSERT_TRUE(beforeArea != afterArea);

        pMsgBuffer = reinterpret_cast<nn::Bit32*>(addr);
        TestIpcAssertObject assertObj(handle, pMsgBuffer, bufSize);

        nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);

        // 受信バッファの準備
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0, 0, 0, 0, 0, 0, 0, 1));
        }
        int32_t index;
        nn::svc::Handle handles[1] = { handle };

        // SendAsyncRequestWithUserBuffer から要求を受信できる
        // SendSyncRequestWithUserBuffer から要求を受信できる
        BeforeServerIpc(pMsgBuffer, bufSize);
        result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, addr, bufSize, handles, sizeof(handles) / sizeof(*handles), nn::svc::INVALID_HANDLE_VALUE, -1);
        AfterServerIpc(pMsgBuffer, bufSize);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(index == 0);

        {
            // IPC ヘッダが受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0xFFFF);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送のデータが受け取れている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);

            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == msgSize);
            ASSERT_TRUE(addr < pointer.GetPointerAddress());
            ASSERT_TRUE(pointer.GetPointerAddress() < addr + bufSize);
            ASSERT_TRUE(::std::memcmp(reinterpret_cast<void*>(pointer.GetPointerAddress()), reinterpret_cast<void*>(nnMain), pointer.GetPointerSize()) == 0);
        }
        {
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::MessageHeader(0x0000, 0, 1, 0, 0, 0, 0, 0));
            offset = ipcMsg.Set(offset, nn::svc::ipc::MessageBuffer::PointerData(reinterpret_cast<char*>(reinterpret_cast<uintptr_t>(nnMain) + 0x100), msgSize, 0));
        }

        // 相手に reply を返すことが出来る
        BeforeServerIpc(pMsgBuffer, msgSize);
        result = nn::svc::ReplyAndReceiveWithUserBuffer(&index, addr, msgSize, handles, 0, handle, 0);
        AfterServerIpc(pMsgBuffer, msgSize);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

        assertObj.Cancel();
        prevBlock.Close();
        nextBlock.Close();
    }
    if (x == IpcClient)
    {
        TestIpcAssertObject assertObj(handle, pMsgBuffer, bufSize);
        {
            // 結果を受信できている
            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(ipcMsg);
            nn::svc::ipc::MessageBuffer::SpecialHeader ipcSpecial(ipcMsg, ipcHeader);
            ASSERT_TRUE(ipcHeader.GetTag() == 0x0000);
            ASSERT_TRUE(ipcHeader.GetPointerNum() == 1);
            ASSERT_TRUE(ipcHeader.GetSendNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveNum() == 0);
            ASSERT_TRUE(ipcHeader.GetExchangeNum() == 0);
            ASSERT_TRUE(ipcHeader.GetRawNum() == 0);
            ASSERT_TRUE(ipcHeader.GetReceiveListNum() == 0);
            ASSERT_TRUE(ipcHeader.GetSpecialNum() == 0);

            int offset;

            // ポインタ転送でデータを受信できている
            offset = nn::svc::ipc::MessageBuffer::GetPointerDataOffset(ipcHeader, ipcSpecial);
            nn::svc::ipc::MessageBuffer::PointerData pointer(offset, ipcMsg);

            ASSERT_TRUE(pointer.GetPointerIndex() == 0);
            ASSERT_TRUE(pointer.GetPointerSize() == msgSize);
            ASSERT_TRUE(addr < pointer.GetPointerAddress());
            ASSERT_TRUE(pointer.GetPointerAddress() < addr + bufSize);
            ASSERT_TRUE(::std::memcmp(
                        reinterpret_cast<void*>(pointer.GetPointerAddress()),
                        reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(nnMain) + 0x100),
                        pointer.GetPointerSize()) == 0);

            nn::svc::MemoryInfo info;
            GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(pMsgBuffer));
            ASSERT_TRUE(info.attribute != nn::svc::MemoryAttribute_Locked);
        }

        assertObj.Cancel();
        prevBlock.Close();
        nextBlock.Close();
    }
} // NOLINT(readability/fn_size)

/* Utility Functions */


void GeneralTest(const bool x, nn::svc::Handle handle, bool userBuffer, bool async)
{
    SpecialDataMaxNum(x, handle, userBuffer, async);
    PointerDataMaxNumWithBuffer(x, handle, userBuffer, async);
    PointerDataMaxNumWithRecieveListBuffer(x, handle, userBuffer, async);
    SendDataMaxNum(x, handle, userBuffer, async);
    RecvDataMaxNum(x, handle, userBuffer, async);
    ExchangeDataMaxNum(x, handle, userBuffer, async);
    RawDataMaxNum(x, handle, userBuffer, async);
    SendAndReceiveIpc(x, handle, userBuffer, async);
    ReceiveAndExchangeIpc(x, handle, userBuffer, async);
    SendAndExchange(x, handle, userBuffer, async);
    SendAll(x, handle, userBuffer, async);
    UnAlignedMapIsCopyWithRecvRequest(x, handle, userBuffer, async);
    UseCopiedHandle(x, handle, userBuffer, async);
    UseMovedHandle(x, handle, userBuffer, async);
    CopyInvalidHandle(x, handle, userBuffer, async);
    MoveInvalidHandle(x, handle, userBuffer, async);
    CopyPseudoThreadHanldle(x, handle, userBuffer, async);
    CopyPseudoProcessHanldle(x, handle, userBuffer, async);
    MovePseudoThreadHanldle(x, handle, userBuffer, async);
    MovePseudoProcessHanldle(x, handle, userBuffer, async);
    ReplyDifferentInfoFromPointerToRaw(x, handle, userBuffer, async);
    ReplyDifferentInfoFromExchangeToPointer(x, handle, userBuffer, async);
    ReplyDifferentInfoFromRawToSpecial(x, handle, userBuffer, async);
    ClientReceiveListWithInvalidMememoryPermmision(x, handle, userBuffer, async);
    ServerReceiveListWithInvalidMememoryPermmision(x, handle, userBuffer, async);
    ClientPointerWithInvalidMememoryPermmision(x, handle, userBuffer, async);
    ServerPointerWithInvalidMememoryPermmision(x, handle, userBuffer, async);
    CannotUseMemoryPermissionNoneForSend(x, handle, userBuffer, async);
    CanUseMemoryPermissionReadForSend(x, handle, userBuffer, async);
    CannotUseMemoryPermissionNoneForRecv(x, handle, userBuffer, async);
    CannotUseMemoryPermissionReadForRecv(x, handle, userBuffer, async);
    CannotUseMemoryPermissionNoneForExchange(x, handle, userBuffer, async);
    CannotUseMemoryPermissionReadForExchange(x, handle, userBuffer, async);
    SendOverSizePointerDataWithIpcBuffer(x, handle, userBuffer, async);
    SendOverSizePointerDataWithOneBuffer(x, handle, userBuffer, async);
    SendOverSizePointerDataWithMultiBuffer(x, handle, userBuffer, async);
    SendOverPointerNumZero(x, handle, userBuffer, async);
    SendOverPointerNum(x, handle, userBuffer, async);
    SendOverSizeRawData(x, handle, userBuffer, async);
    PointerZeroDataWithBuffer(x, handle, userBuffer, async);
    PointerZeroDataWithReceiveListBuffer(x, handle, userBuffer, async);
    PointerZeroDataWithReceiveList(x, handle, userBuffer, async);
    PointeWithZeroSizeReceiveListBuffer(x, handle, userBuffer, async);
    PointerWithZeroSizeReceiveList(x, handle, userBuffer, async);
    SendZeroData(x, handle, userBuffer, async);
    RecvZeroData(x, handle, userBuffer, async);
    ExchangeZeroData(x, handle, userBuffer, async);
    DiscontinuousPointerBuffer(x, handle, userBuffer, async);
    Pointer64bitDataTest(x, handle, userBuffer, async);
    MapArea64bitTest(x, handle, userBuffer, async);
    PointerMemoryAreaTest(x, handle, userBuffer, async);
    ReceiveListMemoryAreaTest(x, handle, userBuffer, async);
    MapMemoryAreaTest(x, handle, userBuffer, async);
    DiscontinuousMapAreaTest(x, handle, userBuffer, async);
    ReceiveListOffsetTest(x, handle, userBuffer, async);
    InvalidReceiveListOffsetTestOnSend(x, handle, userBuffer, async);
    InvalidReceiveListOffsetTestOnRecieve(x, handle, userBuffer, async);
    InvalidPointerIndex(x, handle, userBuffer, async);
    CannotReplySendData(x, handle, userBuffer, async);
    CannotReplyReceiveData(x, handle, userBuffer, async);
    CannotReplyExchangeData(x, handle, userBuffer, async);
    SendCodeDataTest(x, handle, userBuffer, async);
    ReceiveCodeDataTest(x, handle, userBuffer, async);
    ExchangeCodeDataTest(x, handle, userBuffer, async);
    SendExAndRoDataTest(x, handle, userBuffer, async);
}

void UserBufferTest(const bool x, nn::svc::Handle handle, bool async)
{
    SendMaxPointerDataWithBuffer(x, handle, async);
    UseDiscontinuoudUserBuffer(x, handle, async);
}

void CallSync(const bool x, nn::svc::Handle handle)
{
    GeneralTest(x, handle, false, false);
}

void CallSyncWithUserBuffer(const bool x, nn::svc::Handle handle)
{
    GeneralTest(x, handle, true, false);
    UserBufferTest(x, handle, false);
}

void CallAsyncWithUserBuffer(const bool x, nn::svc::Handle handle)
{
    GeneralTest(x, handle, true, true);
    UserBufferTest(x, handle, true);
}

void SessionThread(nn::svc::Handle *handle)
{
    AutoThreadExit autoExit;
    CallSync(IpcServer, *handle);
    CallSyncWithUserBuffer(IpcServer, *handle);
    CallAsyncWithUserBuffer(IpcServer, *handle);
}

} // namespace


TEST(AdditionalIpcTest, NamedPortTest)
{
    nn::Result result;
    nn::svc::Handle portHandle;
    nn::svc::Handle sessionHandle;

    result = nn::svc::SetHeapSize(&g_HeapPtr, g_HeapSize);
    ASSERT_RESULT_SUCCESS(result);

    NamedPortManager namedPortManager(TestPortName, 2);
    portHandle = namedPortManager.GetHandle();

    int32_t curPriority;
    result = nn::svc::GetThreadPriority(&curPriority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(curPriority - 1 < nn::svc::LowestThreadPriority);
    ASSERT_TRUE(curPriority + 1 > nn::svc::HighestThreadPriority);

    for (int32_t i = 0; i < 2; i++)
    {
        int32_t idealCore = (g_ProcessIdealCore + i) % NumCore;
        for (int32_t priority = curPriority - 1; priority <= curPriority + 1; priority++)
        {
            uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
            uintptr_t pc = reinterpret_cast<uintptr_t>(SessionThread);
            nn::svc::Handle serverSessionHandle;
            nn::svc::Handle clientSessionHandle;
            nn::svc::Handle thread;

            result = nn::svc::ConnectToNamedPort(&clientSessionHandle, TestPortName);
            ASSERT_RESULT_SUCCESS(result);

            int32_t index;
            result = nn::svc::WaitSynchronization(&index, &portHandle, 1, -1);
            ASSERT_RESULT_SUCCESS(result);
            result = nn::svc::AcceptSession(&serverSessionHandle, portHandle);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::CreateThread(
                    &thread, pc, reinterpret_cast<uintptr_t>(&serverSessionHandle),
                    sp, priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);
            g_ServerHandle = thread;
            result = nn::svc::StartThread(thread);
            ASSERT_RESULT_SUCCESS(result);

            CallSync(IpcClient, clientSessionHandle);
            CallSyncWithUserBuffer(IpcClient, clientSessionHandle);
            CallAsyncWithUserBuffer(IpcClient, clientSessionHandle);

            result = nn::svc::CloseHandle(clientSessionHandle);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::WaitSynchronization(&index, &thread, 1, -1);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::CloseHandle(thread);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::CloseHandle(serverSessionHandle);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

    result = nn::svc::SetHeapSize(&g_HeapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(AdditionalIpcTest, PortTest)
{
    nn::Result result;
    nn::svc::Handle clientPortHandle;
    nn::svc::Handle serverPortHandle;

    result = nn::svc::SetHeapSize(&g_HeapPtr, g_HeapSize);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::CreatePort(&serverPortHandle, &clientPortHandle, 2, false, 0);
    ASSERT_RESULT_SUCCESS(result);

    int32_t curPriority;
    result = nn::svc::GetThreadPriority(&curPriority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(curPriority - 1 < nn::svc::LowestThreadPriority);
    ASSERT_TRUE(curPriority + 1 > nn::svc::HighestThreadPriority);

    for (int32_t i = 0; i < 2; i++)
    {
        int32_t idealCore = (g_ProcessIdealCore + i) % NumCore;
        for (int32_t priority = curPriority - 1; priority <= curPriority + 1; priority++)
        {
            uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
            uintptr_t pc = reinterpret_cast<uintptr_t>(SessionThread);
            nn::svc::Handle serverSessionHandle;
            nn::svc::Handle clientSessionHandle;
            nn::svc::Handle thread;

            result = nn::svc::ConnectToPort(&clientSessionHandle, clientPortHandle);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::AcceptSession(&serverSessionHandle, serverPortHandle);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::CreateThread(
                    &thread, pc, reinterpret_cast<uintptr_t>(&serverSessionHandle),
                    sp, priority, idealCore);
            ASSERT_RESULT_SUCCESS(result);
            g_ServerHandle = thread;
            result = nn::svc::StartThread(thread);
            ASSERT_RESULT_SUCCESS(result);

            CallSync(IpcClient, clientSessionHandle);
            CallSyncWithUserBuffer(IpcClient, clientSessionHandle);
            CallAsyncWithUserBuffer(IpcClient, clientSessionHandle);

            result = nn::svc::CloseHandle(clientSessionHandle);
            ASSERT_RESULT_SUCCESS(result);

            int32_t index;
            result = nn::svc::WaitSynchronization(&index, &thread, 1, -1);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::CloseHandle(serverSessionHandle);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::CloseHandle(thread);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

    result = nn::svc::CloseHandle(serverPortHandle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(clientPortHandle);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::SetHeapSize(&g_HeapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(AdditionalIpcTest, SessionTest)
{
    nn::Result result;

    result = nn::svc::SetHeapSize(&g_HeapPtr, g_HeapSize);
    ASSERT_RESULT_SUCCESS(result);

    {
        uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
        uintptr_t pc = reinterpret_cast<uintptr_t>(SessionThread);

        int32_t curPriority;
        result = nn::svc::GetThreadPriority(&curPriority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(curPriority - 1 < nn::svc::LowestThreadPriority);
        ASSERT_TRUE(curPriority + 1 > nn::svc::HighestThreadPriority);

        for (int32_t i = 0; i < 2; i++)
        {
            int32_t idealCore = (g_ProcessIdealCore + i) % NumCore;
            for (int32_t priority = curPriority - 1; priority <= curPriority + 1; priority++)
            {
                nn::svc::Handle serverSessionHandle;
                nn::svc::Handle clientSessionHandle;
                result = nn::svc::CreateSession(
                        &serverSessionHandle, &clientSessionHandle, false, 0);
                ASSERT_RESULT_SUCCESS(result);

                nn::svc::Handle thread;
                result = nn::svc::CreateThread(
                        &thread, pc, reinterpret_cast<uintptr_t>(&serverSessionHandle),
                        sp, priority, idealCore);
                ASSERT_RESULT_SUCCESS(result);
                g_ServerHandle = thread;
                result = nn::svc::StartThread(thread);
                ASSERT_RESULT_SUCCESS(result);

                CallSync(IpcClient, clientSessionHandle);
                CallSyncWithUserBuffer(IpcClient, clientSessionHandle);
                CallAsyncWithUserBuffer(IpcClient, clientSessionHandle);

                int32_t index;
                result = nn::svc::WaitSynchronization(&index, &thread, 1, -1);
                ASSERT_RESULT_SUCCESS(result);

                result = nn::svc::CloseHandle(thread);
                ASSERT_RESULT_SUCCESS(result);

                result = nn::svc::CloseHandle(serverSessionHandle);
                ASSERT_RESULT_SUCCESS(result);
                result = nn::svc::CloseHandle(clientSessionHandle);
                ASSERT_RESULT_SUCCESS(result);
            }
        }

    }

    result = nn::svc::SetHeapSize(&g_HeapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);
}

