﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_Timer.h"
#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Abort.h>
#if (defined(NN_DETAIL_SDMMC_USE_NN_UTIL_FOR_TIMER))
    #include <nn/util.h>
#endif

namespace nn { namespace sdmmc1 {
namespace detail {

namespace
{
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_TIMER))
        void SpinWaitMicroseconds(uint32_t us)
        {
            nn::os::Tick timeoutTick = nn::os::GetSystemTick() + nn::os::ConvertToTick(nn::TimeSpan::FromMicroSeconds(us)) + nn::os::Tick(1);
            while (true)
            {
                if (nn::os::GetSystemTick() > timeoutTick)
                {
                    break;
                }
            }
        }
    #endif

    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_TIMER))
        // TODO: nn::dd 関数に置き換え後、削除予定
        static void DataSynchronizationBarrier()
        {
            #if (defined(NN_BUILD_CONFIG_CPU_ARM_V7A))
                asm volatile("dsb");
            #elif (defined(NN_BUILD_CONFIG_CPU_ARM_V8A))
                asm volatile("dsb sy");
            #else
                // なにもしない
            #endif
        }
    #endif

    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_TIMER))
        // TODO: nn::dd 関数に置き換え後、削除予定
        static void InstructionMemoryBarrier()
        {
            #if (defined(NN_BUILD_CONFIG_CPU_ARM_V7A) || defined(NN_BUILD_CONFIG_CPU_ARM_V8A))
                asm volatile("isb");
            #endif
        }
    #endif
}

void WaitMicroseconds(uint32_t us) NN_NOEXCEPT
{
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_TIMER))
        // TODO: nn::dd 関数に置き換え予定
        DataSynchronizationBarrier();
        InstructionMemoryBarrier();
        if (us < 100)
        {
            // パフォーマンスが悪化するため、短い Wait では Sleep しない
            SpinWaitMicroseconds(us);
        }
        else
        {
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(us));
        }
        DataSynchronizationBarrier();
        InstructionMemoryBarrier();
    #elif (defined(NN_DETAIL_SDMMC_USE_NN_UTIL_FOR_TIMER))
        nn::util::WaitMicroSeconds(us);
    #else
        NN_ABORT("WaitMicroseconds() isn't implemented.\n");
    #endif
}

void WaitClocks(uint32_t numClocks, uint32_t clockFrequencyKHz) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(clockFrequencyKHz > 0);
    WaitMicroseconds(nn::util::DivideUp(1000 * numClocks, clockFrequencyKHz));
}

} // namespace detail {
}} // namespace nn { namespace sdmmc1 {
