﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/dd.h>
#include "../i2c_DdUtil.h"

namespace nn {
namespace i2c {
namespace driver {
namespace detail {

/**
 * @brief CLK_RST の、バスに対応するレジスタアドレスおよびビットシフトを保持するための構造体
 */
struct ClkRstRegInfo
{
public:
    void Set(int busIdx) NN_NOEXCEPT
    {
        /**
         * @brief Clock / Reset レジスタのマッピング対象物理アドレス ([Ref1] 2.0)
         */
        static const nn::dd::PhysicalAddress   ClkRstPhysicalAddress  = 0x060006000ull;
        static const size_t                    ClkRstAddressSize      = 0x1000;

        // Clock source register <=> CLK_RST_CONTROLLER_CLK_SOURCE_I2C# ([Ref1] 5.7.52(I2C1), 5.7.68(I2C2), 5.7.73(I2C3), 5.7.126(I2C4), 5.7.53(I2C5), 5.7.248(I2C6), )
        static const uintptr_t ClockSourceRegOffsetTable[] =
        {
            0x0124, // I2C1
            0x0198, // I2C2
            0x01b8, // I2C3
            0x03c4, // I2C4
            0x0128, // I2C5
            0x065c  // I2C6
        };

        // Clock enable register <=> CLK_RST_CONTROLLER_CLK_OUT_ENB_{L,H,U} ([Ref1] 5.7.7 - 5.7.9)
        static const uintptr_t ClockEnableRegOffsetTable[] =
        {
            0x0010, // I2C1
            0x0014, // I2C2
            0x0018, // I2C3
            0x0360, // I2C4
            0x0014, // I2C5
            0x0280  // I2C6
        };
        static const int ClockEnableBitShiftTable[] =
        {
            12,     // I2C1
            22,     // I2C2
            3,      // I2C3
            7,      // I2C4
            15,     // I2C5
            6       // I2C6
        };

        // Reset register <=> CLK_RST_CONTROLLER_RST_DEVICES_{L,H,U} ([Ref1] 5.7.4 - 5.7.6)
        static const uintptr_t ResetRegOffsetTable[] =
        {
            0x0004, // I2C1
            0x0008, // I2C2
            0x000c, // I2C3
            0x0358, // I2C4
            0x0008, // I2C5
            0x028c  // I2C6
        };
        static const int ResetBitShiftTable[] =
        {
            12,     // I2C1
            22,     // I2C2
            3,      // I2C3
            7,      // I2C4
            15,     // I2C5
            6       // I2C6
        };

        NN_SDK_ASSERT(0 <= busIdx && busIdx < static_cast<int>( sizeof(ClockSourceRegOffsetTable) / sizeof(ClockSourceRegOffsetTable[0]) ) );
        // IO マッピングを取得する
        uintptr_t baseAddr  = GetVirtualAddress(ClkRstPhysicalAddress,  ClkRstAddressSize);  // CLK_RST

        pClockSource       = reinterpret_cast<volatile uint32_t*>(baseAddr + ClockSourceRegOffsetTable[busIdx]);
        pClockEnable       = reinterpret_cast<volatile uint32_t*>(baseAddr + ClockEnableRegOffsetTable[busIdx]);
        clockEnableBitMask = (1 << ClockEnableBitShiftTable[busIdx]);
        pReset             = reinterpret_cast<volatile uint32_t*>(baseAddr + ResetRegOffsetTable[busIdx]);
        resetBitMask       = (1 << ResetBitShiftTable[busIdx]);
    }

public:
    volatile uint32_t*  pClockSource;
    volatile uint32_t*  pClockEnable;
    uint32_t            clockEnableBitMask;
    volatile uint32_t*  pReset;
    uint32_t            resetBitMask;
};

} // detail
} // dirver
} // i2c
} // nn
