﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Common.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>

#include "../../include/nn/i2c/driver/i2c.h"

#include "i2c_Util.h"
#include "i2c_TargetSpec.h"

#include "i2c_Session.h"
#include "i2c_Command.h"

namespace nn {
namespace i2c {
namespace driver {
namespace detail {

void Session::Open( int busIdx, Bit16 slaveAddress, AddressingMode addressingMode, SpeedMode speedMode, BusAccessor* pBusAccessor ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_BusAccessorMutex );
    if( !m_IsOpen )
    {
        m_pBusAccessor   = pBusAccessor;
        m_SlaveAddress   = slaveAddress;
        m_AddressingMode = addressingMode;
        m_pBusAccessor->Open(busIdx, speedMode);
        m_IsOpen = true; // m_IsOpen へのアクセスは ResourceManager でシリアライズされている
    }
}

void Session::Close() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_BusAccessorMutex );
    if( m_IsOpen )
    {
        m_pBusAccessor->Close();
        m_pBusAccessor = nullptr;
        m_IsOpen = false; // m_IsOpen へのアクセスは ResourceManager でシリアライズされている
    }
}

bool Session::IsOpened() const NN_NOEXCEPT
{
    return m_IsOpen;
}

Result  Session::ExecuteTransaction( void* pReceiveData, const void* pSendData, size_t dataBytes, TransactionOption inOption, Command command ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock( m_BusAccessorMutex );

    Result    result;

    if ( m_pBusAccessor->CheckBusy() )
    {
        return nn::i2c::ResultBusBusy();
    }

    m_pBusAccessor->StartCondition();

    if ((result = m_pBusAccessor->StartTransaction( m_SlaveAddress, m_AddressingMode, command)).IsSuccess()) // スレーブアドレスの送信
    {
        switch(command)
        {
        case Command_Send:
            result = m_pBusAccessor->Send(reinterpret_cast<const Bit8*>(pSendData), dataBytes, inOption, m_SlaveAddress, m_AddressingMode );
            break;

        case Command_Receive:
            result = m_pBusAccessor->Receive(reinterpret_cast<Bit8*>(pReceiveData), dataBytes, inOption, m_SlaveAddress, m_AddressingMode );
            break;
        default:
            NN_ABORT("Invalid command");
            break;
        }
    }

    m_pBusAccessor->StopCondition();

    return result;

}


} // detail
} // driver
} // i2c
} // nn
