﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/ahid/hdr/hdr.h>
#include <nn/ahid/ahid.h>

#include <nn/stdfu/stdfu.h>
#include <nn/stdfu/stdfu_Result.public.h>

#include "ProControllerBinary.h"
#include "ProControllerBinary0x0211.h"
//#include "ChargingGripBinary.h"

namespace nn {
namespace stdfu {

StDfu g_StDfu;

const size_t StackSize = 64 * 1024;
NN_ALIGNAS(4096) uint8_t g_ProgressThreadStack[StackSize];
bool g_ProgressThreadProcess;
nn::os::ThreadType g_ProgressThread;

//////////////////////////////////////////////////////////////////////////////
static Result GetHdrDevice(nn::ahid::hdr::DeviceParameters *pDeviceParameters, uint16_t vid, uint16_t pid, uint16_t bcd) NN_NOEXCEPT
{
    NN_SDK_LOG("Looking for AHID Device vid: %04x pid:%04x bcd: %04x\n", vid, pid, bcd);

    Result result;
    nn::ahid::hdr::Hdr hdr;

    result = hdr.Initialize();

    if (result.IsSuccess())
    {
        size_t outEntries;
        result = hdr.GetDeviceEntries(&outEntries);

        if (result.IsSuccess())
        {
            size_t                      inEntries = nn::ahid::hdr::AhidDevicesCountMax;
            nn::ahid::hdr::AttachFilter attachFilter;
            nn::ahid::hdr::DeviceHandle deviceHandle[inEntries];

            memset(&attachFilter, 0, sizeof(nn::ahid::hdr::AttachFilter));

            result = hdr.GetDeviceList(&outEntries, inEntries, deviceHandle, &attachFilter);

            if (result.IsSuccess())
            {
                result = ResultAhidDeviceNotFound();

                // for loop does not like to compare size_t in this build system :0
                int entries = static_cast<int>(outEntries);

                for (int i = 0; i < entries; i++)
                {
                    Result result1 = hdr.GetDeviceParameters(pDeviceParameters, deviceHandle[i]);

                    if (result1.IsSuccess())
                    {
                        if (
                                (pDeviceParameters->busId == nn::ahid::hdr::AhidBusIdUsb)   &&
                                (pDeviceParameters->vendorId == vid)                        &&
                                (pDeviceParameters->productId == pid)                       &&
                                (pDeviceParameters->versionNumber == bcd)
                            )
                        {
                            /*
                            NN_SDK_LOG("HDR DeviceHandle:               %08x\n",    deviceHandle[i]);
                            NN_SDK_LOG("    deviceHandle:               %08x\n",    hdrDeviceParameters.deviceHandle);
                            NN_SDK_LOG("    servicePath:                %s\n",      hdrDeviceParameters.servicePath);
                            NN_SDK_LOG("    usagePage:                  %04x\n",    hdrDeviceParameters.usagePage);
                            NN_SDK_LOG("    usageId:                    %04x\n",    hdrDeviceParameters.usageId);
                            NN_SDK_LOG("    busId:                      %d\n",      hdrDeviceParameters.busId);
                            NN_SDK_LOG("    vendorId:                   %08x\n",    hdrDeviceParameters.vendorId);
                            NN_SDK_LOG("    productId:                  %08x\n",    hdrDeviceParameters.productId);
                            NN_SDK_LOG("    versionNumber:              %08x\n",    hdrDeviceParameters.versionNumber);
                                */
                            result = ResultSuccess();

                            break;
                        }
                    }
                }
            }
        }

        hdr.Finalize();
    }

    if (!result.IsSuccess())
    {
        NN_SDK_LOG("AHID device not found\n");
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
void ProgressThread(void *p)
{
    while (g_ProgressThreadProcess)
    {
        nn::stdfu::UpdateStatus updateStatus;

        g_StDfu.GetUpdateStatus(&updateStatus);

        NN_SDK_LOG("UpdateStatus phase %d image %d element %d progress %d%\n", updateStatus.phase, updateStatus.image, updateStatus.element, updateStatus.progress);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    }
}


//////////////////////////////////////////////////////////////////////////////
extern "C" void nnMain()
{
    NN_SDK_LOG("->%s()\n", __FUNCTION__);

    Result result;

    result = g_StDfu.Initialize();

    if (result.IsSuccess())
    {
        // Test update using ahid pointer
        nn::ahid::hdr::DeviceParameters deviceParameters;

        // Somone got this device from HDR, the VID, PID, BCD is already known
        result = GetHdrDevice(&deviceParameters, 0x057e, 0x2009, 0x0200);

        if (result.IsSuccess())
        {
            nn::ahid::Ahid ahid;
            nn::ahid::hdr::DeviceParameters deviceParametersFromAhid;

            // AHID is initialized with the parameters including knowm VID, PID, BCD
            result = ahid.Initialize(&deviceParameters);

            if (result.IsSuccess())
            {
                // We added this API to get parameters from AHID... but it is already known
                ahid.GetDeviceParameters(&deviceParametersFromAhid);

                NN_SDK_LOG("device vendor id    %08x\n", deviceParametersFromAhid.vendorId);
                NN_SDK_LOG("device product id   %08x\n", deviceParametersFromAhid.productId);
                NN_SDK_LOG("device bcd          %08x\n", deviceParametersFromAhid.versionNumber);

                NN_SDK_LOG("Update using ProControllerBinary0x0211 to ahid.\n");

                // Start thread to check progress
                g_ProgressThreadProcess = true;
                nn::os::CreateThread(&g_ProgressThread, ProgressThread, 0, g_ProgressThreadStack, StackSize, nn::os::DefaultThreadPriority);
                nn::os::StartThread(&g_ProgressThread);

                result = g_StDfu.Update(ProControllerBinary0x0211, sizeof(ProControllerBinary0x0211), &ahid);
                NN_SDK_LOG("Update returns %08x.\n", result);

                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                g_ProgressThreadProcess = false;
                nn::os::WaitThread(&g_ProgressThread);
                nn::os::DestroyThread(&g_ProgressThread);

                NN_SDK_LOG("Wait 5 seconds for the new device to enumerate\n");
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

                ahid.Finalize();
            }
        }

        result = GetHdrDevice(&deviceParameters, 0x057e, 0x2009, 0x0211);

        if (result.IsSuccess())
        {
            nn::ahid::Ahid ahid;
            nn::ahid::hdr::DeviceParameters deviceParametersFromAhid;

            // AHID is initialized with the parameters including knowm VID, PID, BCD
            result = ahid.Initialize(&deviceParameters);

            if (result.IsSuccess())
            {
                // We added this API to get parameters from AHID... but it is already known
                ahid.GetDeviceParameters(&deviceParametersFromAhid);

                NN_SDK_LOG("device vendor id    %08x\n", deviceParametersFromAhid.vendorId);
                NN_SDK_LOG("device product id   %08x\n", deviceParametersFromAhid.productId);
                NN_SDK_LOG("device bcd          %08x\n", deviceParametersFromAhid.versionNumber);

                NN_SDK_LOG("Update using ProControllerBinary0x0211 to ahid.\n");
                result = g_StDfu.Update(ProControllerBinary, sizeof(ProControllerBinary), &ahid);

                NN_SDK_LOG("Update returns %08x.\n", result);

                NN_SDK_LOG("Wait 5 seconds for the new device to enumerate\n");
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

                ahid.Finalize();
            }
        }

        // Test update using vid, pid, bcd
        NN_SDK_LOG("Update using ProControllerBinary0x0211, override the pid and bcd.\n");

        // Start thread to check progress
        g_ProgressThreadProcess = true;
        nn::os::CreateThread(&g_ProgressThread, ProgressThread, 0, g_ProgressThreadStack, StackSize, nn::os::DefaultThreadPriority);
        nn::os::StartThread(&g_ProgressThread);

        result = g_StDfu.Update(
                                ProControllerBinary0x0211,                          // pointer to binary
                                sizeof(ProControllerBinary0x0211),                  // binary size in bytes
                                UpdateParameters::UpdateParameters_IdFromBinary,    // target device vid
                                0x2009,                                             // target device pid
                                0x0200                                              // target device bcd
                                );

        NN_SDK_LOG("Update returns %08x.\n", result);

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        g_ProgressThreadProcess = false;
        nn::os::WaitThread(&g_ProgressThread);
        nn::os::DestroyThread(&g_ProgressThread);

        NN_SDK_LOG("Wait 5 seconds for the new device to enumerate\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        NN_SDK_LOG("Update using ProControllerBinary, override the pid and bcd.\n");

        // Start thread to check progress
        g_ProgressThreadProcess = true;
        nn::os::CreateThread(&g_ProgressThread, ProgressThread, 0, g_ProgressThreadStack, StackSize, nn::os::DefaultThreadPriority);
        nn::os::StartThread(&g_ProgressThread);

        result = g_StDfu.Update(
                                ProControllerBinary,                                // pointer to binary
                                sizeof(ProControllerBinary),                        // binary size in bytes
                                UpdateParameters::UpdateParameters_IdFromBinary,    // target device vid
                                0x2009,                                             // target device pid
                                0x0211                                              // target device bcd
                                );

        NN_SDK_LOG("Update returns %08x.\n", result);

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        g_ProgressThreadProcess = false;
        nn::os::WaitThread(&g_ProgressThread);
        nn::os::DestroyThread(&g_ProgressThread);

        g_StDfu.Finalize();
    }

    NN_SDK_LOG("<-%s()\n", __FUNCTION__);
}

} // namespace stdfu
} // namespace nn

