﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <tuple>
#include <queue>

#include <nnt.h>

#include <nn/os.h>
#include <nn/account.h>

#include <nn/srepo.h>
#include <nn/srepo/srepo_ApiAdmin.h>
#include <nn/srepo/detail/srepo_SystemReportGenerator.h>
#include <nn/srepo/detail/service/core/srepo_ReportBuffer.h>

namespace
{
    const size_t ThreadStackSize = 8192;
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ThreadStackSize];
    nn::os::ThreadType g_Thread;

    nn::os::Event g_StopEvent(nn::os::EventClearMode_ManualClear);
}

namespace
{
    std::queue<std::pair<nn::srepo::ReportCategory, nn::account::Uid>> g_Queue;
    nn::os::Mutex g_QueueMutex(false);
    nn::os::ConditionVariable g_QueueCondition;
}

namespace
{
    const char TestEventId[] = "test_srepo_report_buffer";
    const nn::ApplicationId TestApplicationId = { 0x1234567890ABCDEF };
}

namespace
{
    template <typename T>
    T GetRandom() NN_NOEXCEPT
    {
        T value;
        nn::os::GenerateRandomBytes(&value, sizeof (value));
        return value;
    }
}

void PopSystemReport(void*) NN_NOEXCEPT
{
    auto& buffer = nn::srepo::detail::service::core::ReportBuffer::GetInstance();

    while (!g_StopEvent.TryWait())
    {
        nn::srepo::ReportCategory category;
        nn::account::Uid user;
        char eventId[nn::srepo::EventIdLengthMax + 1];
        nn::ApplicationId applicationId;
        NN_FUNCTION_LOCAL_STATIC(nn::Bit8, s_Data, [nn::srepo::ReportBufferSizeMax]);
        size_t dataSize;

        const auto result = buffer.Pop(&category, &user, eventId, &applicationId, &dataSize, s_Data, sizeof (eventId), sizeof (s_Data));
        ASSERT_TRUE(result.IsSuccess() || nn::srepo::ResultBufferEmpty::Includes(result));

        if (!result.IsSuccess())
        {
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(GetRandom<uint32_t>() % 1000 + 1));
            continue;
        }

        {
            NN_UTIL_LOCK_GUARD(g_QueueMutex);

            while (g_Queue.empty())
            {
                g_QueueCondition.Wait(g_QueueMutex);
            }

            ASSERT_EQ(g_Queue.front().first, category);
            ASSERT_EQ(g_Queue.front().second, user);

            g_Queue.pop();
        }

        ASSERT_STREQ(TestEventId, eventId);
        ASSERT_EQ(TestApplicationId, applicationId);
    }
}

TEST(ReportBuffer, Main)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, PopSystemReport, nullptr, g_ThreadStack, ThreadStackSize, nn::os::GetThreadPriority(nn::os::GetCurrentThread()) + 1));
    nn::os::StartThread(&g_Thread);

    NN_UTIL_SCOPE_EXIT
    {
        g_StopEvent.Signal();

        nn::os::WaitThread(&g_Thread);
        nn::os::DestroyThread(&g_Thread);
    };

    nn::Bit8 data[nn::srepo::ReportBufferSizeMax];
    size_t dataSize;

    nn::srepo::detail::SystemReportGenerator::Initialize(&dataSize, data, sizeof (data));

    auto Add = [&](const char* key, auto value) -> nn::Result
    {
        return nn::srepo::detail::SystemReportGenerator::AddKeyValue(
            &dataSize, key, value, data, sizeof (data), dataSize);
    };

    NNT_ASSERT_RESULT_SUCCESS(Add("x", 1.23));
    NNT_ASSERT_RESULT_SUCCESS(Add("y", 4.56));
    NNT_ASSERT_RESULT_SUCCESS(Add("z", 7.89));

    auto& buffer = nn::srepo::detail::service::core::ReportBuffer::GetInstance();

    for (int64_t count = 0; count < 100000; count++)
    {
        const auto category = static_cast<nn::srepo::ReportCategory>(count);
        const auto user = GetRandom<uint64_t>() % 2 == 0
            ? nn::account::InvalidUid
            : GetRandom<nn::account::Uid>();

        if (!buffer.Push(category, user, TestEventId, TestApplicationId, data, dataSize).IsSuccess())
        {
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(GetRandom<uint32_t>() % 1000 + 1));
            continue;
        }

        NN_UTIL_LOCK_GUARD(g_QueueMutex);

        g_Queue.emplace(category, user);

        g_QueueCondition.Signal();
    }
}
