﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.NintendoSdkVsExtension.UnitTest
{
    public static class TestUtil
    {
        public static string GetSdkRoot(TestContext testContext)
        {
            var testPath = new TestUtility.TestPath(testContext);
            return testPath.GetSigloRoot();
        }

        public static void CopyDirectory(string sourceDirectoryPath, string destDirectoryPath)
        {
            CopyDirectory(
                new DirectoryInfo(sourceDirectoryPath),
                new DirectoryInfo(destDirectoryPath));
        }

        public static void CopyDirectory(DirectoryInfo sourceDirectory, DirectoryInfo destDirectory)
        {
            if (!destDirectory.Exists)
            {
                destDirectory.Create();
            }

            foreach (var fi in sourceDirectory.EnumerateFiles())
            {
                fi.CopyTo(Path.Combine(destDirectory.FullName, fi.Name));
            }

            foreach (var di in sourceDirectory.EnumerateDirectories())
            {
                CopyDirectory(di, new DirectoryInfo(Path.Combine(destDirectory.FullName, di.Name)));
            }
        }
    }

    public class TemporaryTestDirectory : IDisposable
    {
        private readonly string _fullName;
        private bool _disposedValue = false;

        public string FullName => _fullName;

        public TemporaryTestDirectory(TestContext testContext)
        {
            this._fullName = Path.Combine(testContext.TestRunDirectory, Path.GetRandomFileName());
        }

        private static string CreateTemporaryDirectory(string hoge)
        {
            while (true)
            {
                var fullName = Path.Combine(hoge, Path.GetRandomFileName());
                if (Directory.Exists(fullName))
                {
                    continue;
                }

                Directory.CreateDirectory(fullName);
                return fullName;
            }
        }

        #region IDisposable Support

        protected virtual void Dispose(bool disposing)
        {
            if (!_disposedValue)
            {
                // ファイナライザでは実施しない
                Debug.Assert(disposing, "must be explicitly disposed");

                if (Directory.Exists(_fullName))
                {
                    try
                    {
                        Directory.Delete(_fullName, true);
                    }
                    // ロックされていて削除できない場合はエラーを無視する
                    catch (IOException)
                    {
                    }
                    catch (UnauthorizedAccessException)
                    {
                    }
                }

                _disposedValue = true;
            }
        }

        public void Dispose()
        {
            Dispose(true);
        }
        #endregion
    }

}
