﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>

#include <nn/aoc.h>
#include <nn/aoc/aoc_SystemApi.h>

#include "../TestApplicationSimple/TestAppSimple_FsUtilities.h"

#include "TestAppSmallSimple_CommonUtil.h"
#include "TestAppSmallSimple_AocInfo.h"

AocInfo::AocInfo() NN_NOEXCEPT
{
    // スモール版アプリは特殊?なため aoc の Initialize 処理が必要な模様。。
    nn::aoc::Initialize();
    this->Init();
}

AocInfo::~AocInfo() NN_NOEXCEPT
{
    nn::aoc::Finalize();
}

void AocInfo::Init() NN_NOEXCEPT
{
    const auto totalCount = nn::aoc::CountAddOnContent();
    if (totalCount <= 0)
    {
        // Aoc が存在しなければ以下の処理は意味が無いので抜ける
        return;
    }

    const int MaxListupCount = 200;
    static nn::aoc::AddOnContentIndex s_AocListupBuffer[MaxListupCount];

    int offset = 0;
    while (offset < totalCount)
    {
        // Aoc のインデックスリストを取得
        const auto count = nn::aoc::ListAddOnContent(s_AocListupBuffer, offset, MaxListupCount);
        if (count == 0)
        {
            // 実質的には入らないパスだと思うが 0 が取得されたら抜ける
            break;
        }

        for (int i = 0; i < count; ++i)
        {
            // 内部のインデックスリストに詰める
            this->m_IndexList.push_back(s_AocListupBuffer[i]);
        }

        offset += count;
    }
}

size_t AocInfo::GetCount() NN_NOEXCEPT
{
    return this->m_IndexList.size();
}

bool AocInfo::CheckAllHash() NN_NOEXCEPT
{
    bool isSuccess = true;
    for (auto& idx : this->m_IndexList)
    {
        const auto isOK = this->CheckHash(idx);
        if (isOK == false)
        {
            // ひとつでも失敗があれば結果としては false としておく
            isSuccess = false;
        }
        NN_LOG("   Index %-4d: %s\n", idx, ((isOK) ? "OK" : "NG"));
    }
    return isSuccess;
}

bool AocInfo::CheckHash(int inIndex) NN_NOEXCEPT
{
    fsutil::ScopedMountAoc mountAoc(AOC_MOUNT_NAME, inIndex);
    auto result = mountAoc.GetLastResult();
    if (result.IsFailure())
    {
        // マウントで失敗
        NN_LOG("[Error] AocInfo::CheckHash() MountProcess Failed : result = 0x%08x\n", result.GetInnerValueForDebug());
        return false;
    }

    const auto dataFilePath = std::string(AOC_MOUNT_NAME) + ":/AddOnContent.dat";
    // そもそも想定しているファイルが存在するかどうかの確認
    if (CommonUtil::IsExistExpectedHashFile(AOC_MOUNT_NAME) == false ||
        fsutil::IsExistPath(dataFilePath.c_str()) == false)
    {
        // 想定データでなければチェックできないので true で返しておく
        NN_LOG("    Not Default Data (Index=%d)\n", inIndex);
        return true;
    }

    const size_t HashBufSize = HASH_STRING_SIZE + 1;
    char expHash[HashBufSize] = { 0 };
    result = CommonUtil::ReadExpectedHashValue(expHash, AOC_MOUNT_NAME);
    if (result.IsFailure())
    {
        NN_LOG("[Error] AocInfo::CheckHash() ReadExpectedHashValue Failed : result = 0x%08x\n", result.GetInnerValueForDebug());
        return false;
    }

    char actHash[HashBufSize] = { 0 };
    result = CommonUtil::GetFileDataHashValue(actHash, dataFilePath);
    if (result.IsFailure())
    {
        NN_LOG("[Error] AocInfo::CheckHash() GetFileDataHashValue Failed : result = 0x%08x\n", result.GetInnerValueForDebug());
        return false;
    }

    if (memcmp(expHash, actHash, HashBufSize) != 0)
    {
        return false;
    }

    return true;
}

