﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <iostream>
#include <iomanip>
#include "TestAppSimple_Utility.h"

// -----------------------------------------------

namespace {
    // 文字列型から任意の数値型へ
    template <typename T>
    T ToNum(const std::string& inNumStr) NN_NOEXCEPT
    {
        if (inNumStr == "")
        {
            return 0;
        }

        std::istringstream iss(inNumStr);
        T num = 0;

        if (inNumStr.find("0x") != std::string::npos)
        {
            iss >> std::hex >> num;
        }
        else
        {
            iss >> num;
        }

        return num;
    }
};

uint32_t Convert::ToUInt32(const std::string& inNumStr) NN_NOEXCEPT
{
    return ToNum<uint32_t>(inNumStr);
}

uint64_t Convert::ToUInt64(const std::string& inNumStr) NN_NOEXCEPT
{
    return ToNum<uint64_t>(inNumStr);
}

uint8_t Convert::ToUInt8(const std::string& inNumStr) NN_NOEXCEPT
{
    // 一旦、通常の数値型(UInt32)で変換し、
    const auto num = Convert::ToUInt32(inNumStr);
    if (num <= 255)
    {
        // uint8_t の範囲内であれば、キャストして返す
        return static_cast<uint8_t>(num);
    }
    // それ以外の値は 0 を返しておく
    return 0;
}

std::string Convert::ToStringFillZero(int value, int digit) NN_NOEXCEPT
{
    std::string numStr = "";
    if (digit < 0)
    {
        return numStr;
    }
    int number = value % static_cast<int>(pow(10, digit));
    if (number < 0)
    {
        number = 0;
    }

    std::ostringstream sout;
    sout << std::setfill('0') << std::setw(digit) << number;
    numStr = sout.str();
    std::cout << numStr << std::endl;

    return numStr;
}

// -----------------------------------------------

void VersionHolder::Convert(const std::string& inRV, const std::string& inPV, const std::string& inUV) NN_NOEXCEPT
{
    // 計算のため数値化しておく
    auto rvNum = Convert::ToUInt32(inRV);
    auto pvNum = Convert::ToUInt32(inPV);
    auto verNum = Convert::ToUInt32(inUV);

    if (inRV != "" || inPV != "")
    {
        // ReleaseVersion または PrivateVersion のどちらかが設定されていればそれらを基準に Version を確定させる
        verNum = (rvNum << 16) | pvNum;
    }
    else
    {
        // ReleaseVersion, PrivateVersion のどちらも設定されていない場合は Version を基準に他の値を確定させる
        rvNum = verNum >> 16;
        pvNum = 0x0000FFFF & verNum;
    }

    this->releaseVer = static_cast<uint16_t>(rvNum);
    this->privateVer = static_cast<uint16_t>(pvNum);
    this->unifiedVer = verNum;
}

// -----------------------------------------------

nn::time::CalendarTime TimeStamp::GetCalendarTime() NN_NOEXCEPT
{
    nn::time::PosixTime posixTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime));
    nn::time::CalendarTime calendarTime;
    nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime));

    return calendarTime;
}

// -----------------------------------------------

ApplicationHeap::ApplicationHeap(size_t size) NN_NOEXCEPT
    : m_pMemoryHeap(nullptr)
{
    this->Initialize(size);
}

ApplicationHeap::~ApplicationHeap() NN_NOEXCEPT
{
    if (m_pMemoryHeap != nullptr)
    {
        this->Finalize();
    }
}

void ApplicationHeap::Initialize(size_t size) NN_NOEXCEPT
{
    m_pMemoryHeap = new nn::Bit8[size];

    m_ApplicationHeapHandle.Initialize(m_pMemoryHeap, size);
}

void ApplicationHeap::Finalize() NN_NOEXCEPT
{
    m_ApplicationHeapHandle.Finalize();

    delete[] reinterpret_cast<nn::Bit8*>(m_pMemoryHeap);
    m_pMemoryHeap = nullptr;
}

void* ApplicationHeap::Allocate(size_t size, size_t alignment) NN_NOEXCEPT
{
    return m_ApplicationHeapHandle.Allocate(size, alignment);
}

void* ApplicationHeap::Allocate(size_t size) NN_NOEXCEPT
{
    return this->Allocate(size, sizeof(void*));
}

void ApplicationHeap::Deallocate(void* ptr) NN_NOEXCEPT
{
    if (ptr != nullptr)
    {
        m_ApplicationHeapHandle.Free(ptr);
    }
}

// -----------------------------------------------
