﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/account.h>
#include <nn/fs.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>

#include "TestAppSimple_SaveDataScene.h"
#include "TestAppSimple_AccountManager.h"

namespace {
    const size_t UniqueDataSize = 20;
    struct SaveDataInfo
    {
        char uniqueData[UniqueDataSize];
        int version;
    };
    SaveDataInfo g_SaveDataArray[nn::account::UserCountMax] = {};

    constexpr auto FileSize = sizeof(SaveDataInfo);
    constexpr auto MountName = "save";
    const auto FilePath = std::string(MountName) + ":/file";

    nn::Result EnsureSaveData(nn::account::Uid uid) NN_NOEXCEPT
    {
        return nn::fs::EnsureSaveData(uid);
    }

    nn::Result CreateSaveData(SaveDataInfo* pOutData, nn::account::Uid uid) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutData);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSaveData(MountName, uid));
        NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(MountName); };

        NN_RESULT_DO(nn::fs::CreateFile(FilePath.c_str(), FileSize));

        {
            nn::fs::FileHandle fileHandle;
            NN_RESULT_DO(nn::fs::OpenFile(&fileHandle, FilePath.c_str(), nn::fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(fileHandle); };

            nn::time::CalendarTime calendarTime = TimeStamp::GetCalendarTime();
            nn::util::SNPrintf(pOutData->uniqueData, UniqueDataSize, "%04d/%02d/%02d_%02d:%02d:%02d",
                calendarTime.year, calendarTime.month, calendarTime.day,
                calendarTime.hour, calendarTime.minute, calendarTime.second);
            pOutData->version = 1;

            NN_RESULT_DO(nn::fs::WriteFile(fileHandle, 0, pOutData, sizeof(*pOutData),
                nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::Commit(MountName));

        NN_RESULT_SUCCESS;
    }

    nn::Result ReadSaveData(SaveDataInfo* pOutData, nn::account::Uid uid) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutData);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSaveData(MountName, uid));
        NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(MountName); };

        {
            nn::fs::FileHandle fileHandle;
            NN_RESULT_DO(nn::fs::OpenFile(&fileHandle, FilePath.c_str(), nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(fileHandle); };

            size_t readSize;
            NN_RESULT_DO(nn::fs::ReadFile(&readSize, fileHandle, 0, pOutData, sizeof(*pOutData)));
            NN_ASSERT_EQUAL(readSize, sizeof(*pOutData));
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::Commit(MountName));

        NN_RESULT_SUCCESS;
    }

    nn::Result UpdateSaveData(SaveDataInfo* pOutData, nn::account::Uid uid) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutData);

        NN_RESULT_DO(ReadSaveData(pOutData, uid));
        ++pOutData->version;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSaveData(MountName, uid));
        NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(MountName); };

        {
            nn::fs::FileHandle fileHandle;
            NN_RESULT_DO(nn::fs::OpenFile(&fileHandle, FilePath.c_str(), nn::fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(fileHandle); };

            NN_RESULT_DO(nn::fs::WriteFile(fileHandle, 0, pOutData, sizeof(SaveDataInfo),
                nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::Commit(MountName));

        NN_RESULT_SUCCESS;
    }

    nn::Result DeleteSaveData(nn::account::Uid uid) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSaveData(MountName, uid));
        NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(MountName); };

        NN_RESULT_DO(nn::fs::DeleteFile(FilePath.c_str()));

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::Commit(MountName));

        NN_RESULT_SUCCESS;
    }

    bool IsCreatedSaveData(nn::account::Uid uid) NN_NOEXCEPT
    {
        if (nn::fs::IsSaveDataExisting(uid) == false)
        {
            return false;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSaveData(MountName, uid));
        NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(MountName); };

        nn::fs::DirectoryEntryType directoryEntryType;
        auto result = nn::fs::GetEntryType(&directoryEntryType, FilePath.c_str());

        return nn::fs::ResultPathNotFound().Includes(result) == false;
    }
}

SaveDataScene::SaveDataScene() NN_NOEXCEPT
    : m_State(State::None)
    , m_CurrentIndex(0)
    , m_Result(nn::ResultSuccess())
    , m_ActionStr("")
    , m_LastResultStr("")
{
}

void SaveDataScene::InternalSetup() NN_NOEXCEPT
{
}

void SaveDataScene::EnterScene() NN_NOEXCEPT
{
    // 他のページでユーザーの追加が可能なのでページを開く際に更新する
    this->UpdateUserInfo();
}

void SaveDataScene::InternalHandleNPad() NN_NOEXCEPT
{
    if (m_State != State::None || m_UserInfoList.empty())
    {
        return;
    }

    auto& info = m_UserInfoList[m_CurrentIndex];
    if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
    {
        if (info.saveState == UserInfo::SaveState::Created)
        {
            m_State = State::Update;
        }
        else if (info.saveState == UserInfo::SaveState::Ensured)
        {
            m_State = State::Create;
        }
        else if (info.saveState == UserInfo::SaveState::None)
        {
            m_State = State::Ensure;
        }
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::X::Mask))
    {
        if (info.saveState == UserInfo::SaveState::Created)
        {
            m_State = State::Delete;
        }
    }
    else if (HasHidControllerAnyButtonsDown(CursorDown))
    {
        m_CurrentIndex = (m_CurrentIndex + 1) % m_UserInfoList.size();
    }
    else if (HasHidControllerAnyButtonsDown(CursorUp))
    {
        auto index = m_CurrentIndex + static_cast<int>(m_UserInfoList.size()) - 1;
        m_CurrentIndex = index % m_UserInfoList.size();
    }
}

void SaveDataScene::InternalHandleTouchScreen() NN_NOEXCEPT
{
    if (m_State != State::None)
    {
        return;
    }

    auto changeState = [&](State state, int index) {
        m_State = state;
        m_CurrentIndex = index;
    };

    for (auto& info : m_UserInfoList)
    {
        if (info.saveButton.range.IsInRange(m_PreviousTouch))
        {
            if (info.saveState == UserInfo::SaveState::Created)
            {
                changeState(State::Update, info.index);
                break;
            }
            else if (info.saveState == UserInfo::SaveState::Ensured)
            {
                changeState(State::Create, info.index);
                break;
            }
            else if (info.saveState == UserInfo::SaveState::None)
            {
                changeState(State::Ensure, info.index);
                break;
            }
        }
        else if (info.deleteButton.range.IsInRange(m_PreviousTouch) &&
            info.saveState == UserInfo::SaveState::Created)
        {
            changeState(State::Delete, info.index);
            break;
        }
    }
}

void SaveDataScene::InternalProcess() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    auto changeStr = [&]() {
        const size_t len = 22;
        char resultStr[len];
        if (m_Result.IsSuccess())
        {
            NN_ABORT_UNLESS(nn::util::SNPrintf(resultStr, len, "Success (0x%08x)", m_Result) < len);
        }
        else
        {
            NN_ABORT_UNLESS(nn::util::SNPrintf(resultStr, len, "Failed (0x%08x)", m_Result) < len);
        }
        m_LastResultStr = resultStr;
    };

    auto& info = m_UserInfoList[m_CurrentIndex];
    if(m_State == State::Ensure)
    {
        m_ActionStr = info.name + " - Ensure : ";
        m_Result = EnsureSaveData(info.id);
        changeStr();

        if (m_Result.IsSuccess())
        {
            // サイズ未指定・0x0指定でも Ensure が成功するため
            info.saveState = nn::fs::IsSaveDataExisting(info.id) ?
                UserInfo::SaveState::Ensured : UserInfo::SaveState::NotSpecifiedSize;
        }
    }
    else if (m_State == State::Create)
    {
        m_ActionStr = info.name + " - Create : ";
        m_Result = CreateSaveData(&g_SaveDataArray[m_CurrentIndex], info.id);
        changeStr();

        if (m_Result.IsSuccess())
        {
            info.saveState = UserInfo::SaveState::Created;
        }
    }
    else if (m_State == State::Update)
    {
        m_ActionStr = info.name + " - Update : ";
        m_Result = UpdateSaveData(&g_SaveDataArray[m_CurrentIndex], info.id);
        changeStr();
    }
    else if (m_State == State::Delete)
    {
        m_ActionStr = info.name + " - Delete : ";
        m_Result = DeleteSaveData(info.id);
        changeStr();

        if (m_Result.IsSuccess())
        {
            info.saveState = UserInfo::SaveState::Ensured;
        }
    }
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

    // State を戻す
    m_State = State::None;
}

void SaveDataScene::InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* pWriter) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pWriter);

    this->SetDefaultScale(pWriter);
    pWriter->SetTextColor(White);
    pWriter->SetCursor(40.0f, 75.0f);
    pWriter->Print("[SaveData]  ");

    if (m_UserInfoList.empty())
    {
        pWriter->SetCursor(ListPosX + 30.0f, ListPosY);
        pWriter->Print("User account is not registered.");
        return;
    }

    pWriter->Print(m_ActionStr.c_str());

    pWriter->SetTextColor(m_Result.IsSuccess() ? Green : Red);
    pWriter->Print(m_LastResultStr.c_str());

    pWriter->SetTextColor(White);
    pWriter->SetCursor(ListPosX + 5.0f, ListPosY + (m_CurrentIndex * ListSpanY));
    pWriter->Print("*");

    for (auto& info : m_UserInfoList)
    {
        pWriter->SetTextColor(White);
        pWriter->SetCursor(ListPosX + 30.0f, ListPosY + ListSpanY * info.index);
        pWriter->Print("%-10s", info.name.c_str());

        pWriter->SetCursor(ListPosX + 30.0f, ListPosY + ListSpanY * info.index + 32.0f);
        if (info.saveState == UserInfo::SaveState::None)
        {
            pWriter->SetTextColor(Red);
            pWriter->Print("    Save data is not ensured.");

            info.saveButton.labelStr = "A: Ensure";
            this->WriteTouchRange(pWriter, &info.saveButton, true);
        }
        else if (info.saveState == UserInfo::SaveState::NotSpecifiedSize)
        {
            pWriter->SetTextColor(Red);
            pWriter->Print("    Save data size is not specified.");
        }
        else if (info.saveState == UserInfo::SaveState::Ensured)
        {
            pWriter->SetTextColor(Yellow);
            pWriter->Print("    Save data file is not created.");

            info.saveButton.labelStr = "A: Create";
            this->WriteTouchRange(pWriter, &info.saveButton, true);
        }
        else
        {
            pWriter->Print("    SaveData : %s ", g_SaveDataArray[info.index].uniqueData);
            pWriter->Print("    Version : %d", g_SaveDataArray[info.index].version);

            info.saveButton.labelStr = "A: Update";
            this->WriteTouchRange(pWriter, &info.saveButton, true);
            this->WriteTouchRange(pWriter, &info.deleteButton, true);
        }
    }
}

void SaveDataScene::UpdateUserInfo() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    int count;
    nn::account::Uid users[nn::account::UserCountMax];
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::account::ListAllUsers(&count, users, nn::account::UserCountMax));

    m_UserInfoList.clear();
    UserInfo info;
    for (int i = 0; i < count; i++)
    {
        info.id = users[i];
        info.index = i;

        nn::account::Nickname name;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNickname(&name, info.id));
        info.name = name.name;

        info.saveButton.pos = { 240.0f, ListPosY + ListSpanY * i };
        info.deleteButton.labelStr = "X: Delete";
        info.deleteButton.pos = { 440.0f, ListPosY + ListSpanY * i };

        g_SaveDataArray[i] = SaveDataInfo();

        if (nn::fs::IsSaveDataExisting(info.id) == false)
        {
            info.saveState = UserInfo::SaveState::None;
        }
        else if (IsCreatedSaveData(info.id) == false)
        {
            info.saveState = UserInfo::SaveState::Ensured;
        }
        else
        {
            info.saveState = UserInfo::SaveState::Created;

            ReadSaveData(&g_SaveDataArray[i], info.id);
        }

        m_UserInfoList.push_back(info);
    }
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
}
