﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <memory>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/http/json/http_JsonPath.h>
#include <nn/http/json/http_JsonErrorMap.h>
#include <nn/http/json/http_RapidJsonApi.h>
#include <nn/http/json/http_RapidJsonInputStream.h>

namespace json {

// ImportJsonByRapidJson で使用する InputStreamType の定義
class MemoriedStreamForRapidJson : public nn::http::json::StringBufferForRapidJson
{
private:
    typedef nn::http::json::StringBufferForRapidJson Base;

public:
    typedef Base::Ch Ch;

public:
    MemoriedStreamForRapidJson(const Ch* json, size_t size) NN_NOEXCEPT
        : m_Json(json)
        , m_Size(size)
        , m_Position(0u)
    {
        NN_SDK_REQUIRES(json && size > 0u);
        NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(json) % NN_ALIGNOF(Ch) == 0);
        NN_SDK_ASSERT(size % sizeof(Ch) == 0);
    }
    ~MemoriedStreamForRapidJson() NN_NOEXCEPT
    {
    }

    // 以降は InputStreamType で定義が必要なメソッド
    nn::Result GetResult() const NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }
    inline Ch Peek() NN_NOEXCEPT
    {
        return (m_Position < m_Size) ? m_Json[m_Position] : '\0';
    }
    inline Ch Take() NN_NOEXCEPT
    {
        auto c = Peek();
        ++m_Position;
        return c;
    }
    inline size_t Tell() const NN_NOEXCEPT
    {
        return m_Position;
    }

private:
    const Ch* m_Json;
    size_t m_Size;
    size_t m_Position;
};

// ImportJsonByRapidJson で使用する JsonAdaptorType の定義
class DebugAdaptor
{
public:
    typedef nn::http::json::JsonPath<16, 256, false> JsonPathType;

    DebugAdaptor() NN_NOEXCEPT
    {
    };

    // 以降は JsonAdaptorType で定義が必要なメソッド
    void Update(const JsonPathType& path, std::nullptr_t) NN_NOEXCEPT
    {
        NN_LOG("[DebugAdaptor Update_nullptr_t] %s nullptr\n", path.ToString());
    }
    void Update(const JsonPathType& path, bool value) NN_NOEXCEPT
    {
        NN_LOG("[DebugAdaptor Update_bool] %s %s \n", path.ToString(), value ? "true" : "false");
    }
    void Update(const JsonPathType& path, int64_t value) NN_NOEXCEPT
    {
        NN_LOG("[DebugAdaptor Update_int64_t] %s 0x%016llx \n", path.ToString(), value);
    }
    void Update(const JsonPathType& path, uint64_t value) NN_NOEXCEPT
    {
        NN_LOG("[DebugAdaptor Update_uint64_t] %s 0x%016llx \n", path.ToString(), value);
    }
    void Update(const JsonPathType& path, double value) NN_NOEXCEPT
    {
        NN_LOG("[DebugAdaptor Update_double] %s %lf \n", path.ToString(), value);
    }
    void Update(const JsonPathType& path, const char* value, int length) NN_NOEXCEPT
    {
        NN_LOG("[DebugAdaptor Update_char(%d)] %s %s \n", length, path.ToString(), value);
    }
    void NotifyObjectBegin(const JsonPathType& path) NN_NOEXCEPT
    {
        NN_LOG("[DebugAdaptor NotifyObjectBegin] %s\n", path.ToString());
    }
    void NotifyObjectEnd(const JsonPathType& path) NN_NOEXCEPT
    {
        NN_LOG("[DebugAdaptor NotifyObjectEnd] %s\n", path.ToString());
    }
};

// ImportJsonByRapidJson で使用する CancellableType の定義
struct Cancellable
{
    bool IsCancelled() const { return false; }
};

template <typename JsonAdaptorType, typename InputStreamType>
nn::Result ImportJsonByRapidJson(JsonAdaptorType* adaptor, InputStreamType& stream) NN_NOEXCEPT
{
    Cancellable cancellable;
    return nn::http::json::ImportJsonByRapidJson<nn::http::json::DefaultJsonErrorMap>(*adaptor, stream, &cancellable);
}

} //! namespace json
