﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nn/nn_Log.h>

#include "TestAppSimple_FsUtilities.h"
#include "TestAppSimple_MetaInfo.h"
#include "TestAppSimple_Utility.h"

char ExpectedHash[Hash::Sha1MaxStrSize + 1] =
{
    "CDAAA9E0F5479DD254B97C698DCD5084A9D13738" // Generic版でハッシュ関連の動作確認したい場合はこの値を適宜変更する
};

MetaInfo MetaInfoList[MetaInfoMaxCount] = {
    { "aaa", "111" },
    { "bbb", "222" },
    { "ccc", "333" },
    { "d", "332" },
    { "e", "34" },
    { "f", "366" },
    { "g", "38467" },
    { "h", "33254352" },
    { "i", "32345234533" },
    { "j", "532534333" },
    { "k", "43534" },
    { "l", "8743657423" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "WWWWWWWWW0WWWWWWWWW0", "WWWWWWWWW0WWWWWWWWW0WWWWWWWWW0" },
    { "q", "26452" },
    { "r", "6265" },
    { "s", "45265472" },
    { "t", "43747" },
    { "", "" }, // 最後の番兵的なデータ
};

// (SIGLO-52835) BCAT が有効かどうかの(グローバル)フラグ定義
bool IsBcatSettingEnabled = false;

static const char* const HashValueFilePath = "Contents:/HashVal.txt";

void ReadHashValueFileData() NN_NOEXCEPT
{
    nn::Result result;
    fsutil::File hashFile;

    result = hashFile.Open(HashValueFilePath, nn::fs::OpenMode_Read);
    if (result.IsFailure())
    {
        return;
    }

    const auto fileSize = hashFile.GetSize();
    if (fileSize <= 0 || fileSize <= (Hash::Sha1MaxStrSize))
    {
        NN_LOG("[Error] ReadHashValueFileData() : fileSize = %lld\n", fileSize);
        return;
    }

    std::unique_ptr<char[]> buf(new char[Hash::Sha1MaxStrSize + 1]);
    *(buf.get() + Hash::Sha1MaxStrSize) = '\0';

    size_t outSize = 0;
    size_t readSize = Hash::Sha1MaxStrSize;
    // UTF8のBOM付きファイルであることを想定(オフセットの3バイトはそのため)
    result = hashFile.Read(&outSize, 3, buf.get(), readSize);
    if (result.IsFailure())
    {
        // ファイル読み込み処理で失敗
        return;
    }

    //NN_LOG("buf = \"%s\"\n", buf);
    memcpy(ExpectedHash, buf.get(), outSize);

    ExpectedHash[Hash::Sha1MaxStrSize] = '\0';
    //NN_LOG("ExpectedHash = \"%s\"\n", ExpectedHash);
}

void ParseMetaInfoList(char* inDocPtr, MetaInfo outList[])
{
    const char* divStr = "\"";

    std::string docStr = inDocPtr;
    const size_t maxSize = docStr.size();

    int i = 0;
    for ( std::string::size_type pos = 0; ; ++i)
    {
        if (pos > maxSize || i >= (MetaInfoMaxCount - 1))
        {
            break;
        }

        pos = docStr.find(divStr, pos);
        if (pos == std::string::npos)
        {
            break;
        }

        MetaInfo& refInfo = outList[i];
        ++pos;
        auto nextPos = docStr.find(divStr, pos);
        refInfo.item = docStr.substr(pos, nextPos - pos);
        //NN_LOG("item[%d] = \"%s\"\n", i, refInfo.item.c_str());

        pos = docStr.find(divStr, nextPos + 1);
        if (pos == std::string::npos)
        {
            break;
        }

        ++pos;
        nextPos = docStr.find(divStr, pos);
        refInfo.value = docStr.substr(pos, nextPos - pos);
        //NN_LOG("value[%d] = \"%s\"\n", i, refInfo.value.c_str());
        pos = nextPos + 1;
    }

    // 終端用のデータ挿入
    MetaInfo info;
    outList[i] = info;
}

static const char* const MetaInfoListFilePath = "Contents:/MetaList.txt";

void ReadMetaInfoListFileData() NN_NOEXCEPT
{
    nn::Result result;
    fsutil::File metaListFile;

    // ファイルを読み込む前に念のため終端データを先頭に入れておく
    {
        MetaInfo info;
        MetaInfoList[0] = info;
    }

    result = metaListFile.Open(MetaInfoListFilePath, nn::fs::OpenMode_Read);
    if (result.IsFailure())
    {
        return;
    }

    const auto fileSize = metaListFile.GetSize();
    if (fileSize <= 0)
    {
        return;
    }

    const size_t readSize = static_cast<size_t>(fileSize);
    std::unique_ptr<char[]> buf(new char[readSize + 1]);
    *(buf.get() + readSize) = '\0';

    size_t outSize = 0;
    result = metaListFile.Read(&outSize, 0, buf.get(), readSize);
    if (result.IsFailure())
    {
        // ファイル読み込み処理で失敗
        return;
    }

    ParseMetaInfoList(buf.get(), MetaInfoList);
}

nn::ApplicationId GetApplicationId() NN_NOEXCEPT
{
    nn::ApplicationId id = nn::ApplicationId::GetInvalidId();
    for (auto& info : MetaInfoList)
    {
        if (info.item == "Core/ApplicationId" || info.item == "App/ApplicationId")
        {
            id.value = Convert::ToUInt64(info.value);
            break;
        }
    }
    return id;
}
