﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nifm.h>
#include <nn/account/account_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include "TestAppSimple_EcServiceConfig.h"
#include "TestAppSimple_EcServiceAccount.h"
#include "TestAppSimple_AccountManager.h"

// ユーザー選択
UserSelector::UserSelector() NN_NOEXCEPT
{
    m_Setting =  nn::account::DefaultUserSelectionSettings;
    m_Uid = nn::account::InvalidUid;
    m_Nickname.name[0] = '\n';
    m_NsaId.id = 0;
}

UserSelector::~UserSelector() NN_NOEXCEPT
{
}

void UserSelector::Prepare(bool isNsaRequired) NN_NOEXCEPT
{
    SetStateToPrepared();
    m_Setting.isNetworkServiceAccountRequired = isNsaRequired;
    m_Uid = nn::account::InvalidUid;
    m_Nickname.name[0] = '\n';
    m_NsaId.id = 0;
}

const nn::account::Uid& UserSelector::GetUid() const NN_NOEXCEPT
{
    NN_ASSERT(IsExecuted());
    return m_Uid;
}

const nn::account::Nickname& UserSelector::GetNickname() const NN_NOEXCEPT
{
    NN_ASSERT(IsExecuted());
    return m_Nickname;
}

const nn::account::NetworkServiceAccountId& UserSelector::GetNsaId() const NN_NOEXCEPT
{
    NN_ASSERT(IsExecuted());
    return m_NsaId;
}

bool UserSelector::Exeute() NN_NOEXCEPT
{
    m_LastResult = SelectUser();
    if (m_LastResult.IsSuccess())
    {
        if (m_Setting.isNetworkServiceAccountRequired)
        {
            m_LastResult = InquireNsaId();
        }
    }
    return m_LastResult.IsSuccess();
}

nn::Result UserSelector::SelectUser() NN_NOEXCEPT
{
#if defined( TESTAPP_ENABLE_EC_ACCESS )
    nn::account::Uid uid;
    auto result = nn::account::ShowUserSelector(&uid, m_Setting);
    if (nn::account::ResultCancelledByUser::Includes(result))
    {
        return result;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::account::Nickname name;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNickname(&name, uid));

    m_Uid = uid;
    m_Nickname = name;
#else // defined( TESTAPP_ENABLE_EC_ACCESS )
    m_Uid._data[0] = 0x0123456789abcdef;
    m_Uid._data[1] = 0x0123456789abcdef;
    NN_ABORT_UNLESS(nn::util::Strlcpy(m_Nickname.name, "UserNickname", sizeof(m_Nickname.name)) < sizeof(m_Nickname.name));
#endif // defined( TESTAPP_ENABLE_EC_ACCESS )
    NN_RESULT_SUCCESS;
}

nn::Result UserSelector::InquireNsaId() NN_NOEXCEPT
{
#if defined( TESTAPP_ENABLE_EC_ACCESS )
    nn::account::UserHandle handle;
    // (SIGLO-79976) アカウント管理に対して Open 済のユーザーであるかどうかの確認
    auto& am = AccountManager::GetInstance();
    AccountManager::UserInfo info;
    const auto isOpened = am.GetOpenedUserInfo(m_Uid, &info);
    if (isOpened)
    {
        // すでに Open 済のユーザーであれば、そのハンドルを利用する
        handle = info.handle;
    }
    else
    {
        // Open していないユーザーであれば従来通り自前で Open 処理を実施
        NN_RESULT_DO(nn::account::OpenUser(&handle, m_Uid));
    }

    NN_UTIL_SCOPE_EXIT
    {
        if (isOpened == false)
        {
            // Open 済だったユーザーであればクローズしない
            nn::account::CloseUser(handle);
        }
    };

    nn::account::NetworkServiceAccountId nsaId;
    NN_RESULT_DO(nn::account::EnsureNetworkServiceAccountAvailable(handle));
    NN_RESULT_DO(nn::account::GetNetworkServiceAccountId(&nsaId, handle));

    m_NsaId = nsaId;
#else // defined( TESTAPP_ENABLE_EC_ACCESS )
    m_NsaId.id = 0x0123456789abcdef;
#endif // defined( TESTAPP_ENABLE_EC_ACCESS )
    NN_RESULT_SUCCESS;
}

// ネットワーク接続
NetworkConnector::NetworkConnector() NN_NOEXCEPT
{
}

NetworkConnector::~NetworkConnector() NN_NOEXCEPT
{
}

void NetworkConnector::Prepare() NN_NOEXCEPT
{
    SetStateToPrepared();
}

bool NetworkConnector::Exeute() NN_NOEXCEPT
{
#if defined( TESTAPP_ENABLE_EC_ACCESS )
    nn::nifm::SubmitNetworkRequest();
    while (nn::nifm::IsNetworkRequestOnHold())
    {
        if (IsCancelRequested())
        {
            nn::nifm::CancelNetworkRequest();
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }
    if (!nn::nifm::IsNetworkAvailable())
    {
        m_LastResult = nn::nifm::HandleNetworkRequestResult();
    }
    return m_LastResult.IsSuccess();
#else // defined( TESTAPP_ENABLE_EC_ACCESS )
    return true;
#endif // defined( TESTAPP_ENABLE_EC_ACCESS )
}
