﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>

// nn::ec 系モジュールは Generic では扱わない
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nn/ec.h>
#include <nn/err/err_ShowUnacceptableAddOnContentVersionErrorApi.h>
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

#include "TestAppSimple_EcScene.h"
#include "TestAppSimple_Utility.h"

namespace {
    const float Label_X_Pos = 60.0f;
    const float Label_Y_Pos = 130.0f;
    const float Label_Y_Width = 90.0f;
};

EcScene::EcScene() NN_NOEXCEPT
    : m_State(State_None), m_SelectedLauchType(Launch_None),
    m_CurrentSelectPos(0), m_CurrentMaxPosNum(MaxItemCount - 1)
{
}

void EcScene::InternalSetup() NN_NOEXCEPT
{
    // 表示するラベルと対応する呼び出し API の enum 値をまとめて設定
    {
        struct labelPair
        {
            const char* label;
            LaunchType  type;
        } labelList[MaxItemCount] = {
            { "ShopApplicationInformation", Launch_ApplicationInfo },
            { "ShopAddOnContentList", Launch_AddOnContentList },
            { "ShopSubscriptionList", Launch_SubscriptionList },
            { "UnacceptableAddOnContentVersionError", Launch_UnacceptableAddOnContentVersionError },
        };

        for (int i = 0; i < MaxItemCount; ++i)
        {
            m_LaunchRangeList[i].range.pos.x = Label_X_Pos;
            m_LaunchRangeList[i].range.pos.y = Label_Y_Pos + (Label_Y_Width * i);
            m_LaunchRangeList[i].range.labelStr = labelList[i].label;
            m_LaunchRangeList[i].type = labelList[i].type;
        }
    }
}

void EcScene::InternalHandleNPad() NN_NOEXCEPT
{
    if (m_State == State_None)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
        {
            m_State = State_Launching;
            m_SelectedLauchType = m_LaunchRangeList[m_CurrentSelectPos].type;
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Down::Mask)
                 || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLDown::Mask))
        {
            if (m_CurrentSelectPos < m_CurrentMaxPosNum)
            {
                ++m_CurrentSelectPos;
            }
            else
            {
                m_CurrentSelectPos = 0;
            }
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Up::Mask)
                 || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLUp::Mask))
        {
            if (m_CurrentSelectPos > 0)
            {
                --m_CurrentSelectPos;
            }
            else
            {
                m_CurrentSelectPos = m_CurrentMaxPosNum;
            }
        }
    }
}

void EcScene::InternalHandleTouchScreen() NN_NOEXCEPT
{
    if (m_State == State_None)
    {
        int count = 0;
        for (auto& lr : m_LaunchRangeList)
        {
            if (lr.range.range.IsInRange(m_PreviousTouch) == true)
            {
                m_State = State_Launching;
                m_SelectedLauchType = lr.type;
                m_CurrentSelectPos = count;
                break;
            }
            ++count;
        }
    }
}

void EcScene::InternalProcess() NN_NOEXCEPT
{
    if (m_State == State_Launching)
    {
        // nn::ec 系モジュールは Generic では扱わない
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        nn::ApplicationId appId;
        // 自分自身の ID を設定する
        appId.value = Convert::ToUInt64(m_ApplicationInfo.id);
        switch (m_SelectedLauchType)
        {
        case Launch_ApplicationInfo:
            nn::ec::ShowShopApplicationInformation(appId);
            break;
        case Launch_AddOnContentList:
            nn::ec::ShowShopAddOnContentList(appId);
            break;
        case Launch_SubscriptionList:
            nn::ec::ShowShopSubscriptionList(appId);
            break;
        case Launch_UnacceptableAddOnContentVersionError:
            nn::err::ShowUnacceptableAddOnContentVersionError();
            break;
        case Launch_None:
        default:
            // ここに来ることはないと思うが何もしない
            break;
        }
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        m_State = State_None;
    }
}

void EcScene::InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetTextColor(White);
    writer->SetCursor(40.0f, 75.0f);
    writer->Print("[E-Commerce]");

    writer->SetScale(1.8f, 1.8f);
    {
        writer->SetTextColor(White);
        writer->SetCursor(Label_X_Pos - 30.0f, Label_Y_Pos + (Label_Y_Width * (m_CurrentSelectPos)));
        writer->Print("*");
    }

    for (auto& lr : m_LaunchRangeList)
    {
        this->WriteTouchRange(writer, &lr.range);
    }

    if (m_State == State_None)
    {
        writer->SetTextColor(White);
        writer->SetCursor(100.0f, 600.0f);
        writer->Print("A: Launch Applet");
    }
}
