﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>

#include <nn/account.h>
#include <nn/bcat.h>
#include <nn/hid.h>
#include <nn/nifm.h>
#include <nn/time.h>

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nn/socket.h>
#include <nn/ssl.h>
#include <nn/ec/ec_ShopServiceAccessor.h>
#include <nn/oe/oe_DebugApis.h>
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

#include "TestAppSimple_Application.h"
#include "TestAppSimple_HidController.h"

#include "TestAppSimple_HeadlineScene.h"
#include "TestAppSimple_MetaInfoScene.h"
#include "TestAppSimple_RomHashScene.h"
#include "TestAppSimple_HtmlDocScene.h"
#include "TestAppSimple_AocInfoScene.h"
#include "TestAppSimple_AccountScene.h"
#include "TestAppSimple_EcScene.h"
#include "TestAppSimple_AbortDebugScene.h"
#include "TestAppSimple_NetworkScene.h"
#include "TestAppSimple_BcatScene.h"
#include "TestAppSimple_MiiScene.h"
#include "TestAppSimple_AlbumScene.h"
#include "TestAppSimple_MultiProgramScene.h"
#include "TestAppSimple_EcOcsiScene.h"
#include "TestAppSimple_EcCatalogScene.h"
#include "TestAppSimple_SaveDataScene.h"

#include "TestAppSimple_MetaInfo.h"
#include "TestAppSimple_GraphicSystem.h"
#include "TestAppSimple_EcServerAccessor.h"

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/nn_Windows.h>
#endif

namespace {

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

    // ソケットのデフォルト構成
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

    // ショップサービスアクセス作業メモリ
    NN_ALIGNAS(nn::os::MemoryPageSize) nn::Bit8 g_WorkMemory[nn::os::MemoryPageSize * 16];

#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

} // namespace <unnamed>

Application::Application() NN_NOEXCEPT
{
}

Application::~Application() NN_NOEXCEPT
{
    if (IsGraphicSystemInitialized())
    {
        FinalizeGraphicSystem();
    }
}

void Application::Initialize() NN_NOEXCEPT
{
    InitializeGraphicSystem();
    InitializeDebugFontWriter(&m_DebugFontWriter);

    m_CurrentScenePtr = new MetaInfoScene();
    m_OperationEnvScenePtr = new OperationEnvScene();
    {
        // Applicationが保持する画面オブジェクトを予め作っておく
        m_ApplicationSceneList.insert(
            std::make_pair(NextAction::MetaInfo, std::unique_ptr<Scene>(m_CurrentScenePtr))
            );
        m_ApplicationSceneList.insert(
            std::make_pair(NextAction::RomHash, std::unique_ptr<Scene>(new RomHashScene()))
            );
        m_ApplicationSceneList.insert(
            std::make_pair(NextAction::HtmlDoc, std::unique_ptr<Scene>(new HtmlDocScene()))
            );
        m_ApplicationSceneList.insert(
            std::make_pair(NextAction::AocInfo, std::unique_ptr<Scene>(new AocInfoScene()))
            );
        // OperationEnvScene は若干特殊・・
        m_ApplicationSceneList.insert(
            std::make_pair(NextAction::OperationEnv, std::unique_ptr<Scene>(m_OperationEnvScenePtr))
            );
        m_ApplicationSceneList.insert(
            std::make_pair(NextAction::Account, std::unique_ptr<Scene>(new AccountScene()))
            );
        m_ApplicationSceneList.insert(
            std::make_pair(NextAction::ECommerce, std::unique_ptr<Scene>(new EcScene()))
            );
        m_ApplicationSceneList.insert(
            std::make_pair(NextAction::AbortDebug, std::unique_ptr<Scene>(new AbortDebugScene()))
            );
        m_ApplicationSceneList.insert(
            std::make_pair(NextAction::Network, std::unique_ptr<Scene>(new NetworkScene()))
            );
        m_ApplicationSceneList.insert(
            std::make_pair(NextAction::Bcat, std::unique_ptr<Scene>(new BcatScene()))
            );
        m_ApplicationSceneList.insert(
            std::make_pair(NextAction::Mii, std::unique_ptr<Scene>(new MiiScene()))
        );
        m_ApplicationSceneList.insert(
            std::make_pair(NextAction::Album, std::unique_ptr<Scene>(new AlbumScene()))
        );
        m_ApplicationSceneList.insert(
            std::make_pair(NextAction::MultiProgram, std::unique_ptr<Scene>(new MultiProgramScene()))
        );
        m_ApplicationSceneList.insert(
            std::make_pair(NextAction::EcOcsi, std::unique_ptr<Scene>(new EcOcsiScene()))
        );
        m_ApplicationSceneList.insert(
            std::make_pair(NextAction::EcCatalog, std::unique_ptr<Scene>(new EcCatalogScene()))
        );
        m_ApplicationSceneList.insert(
            std::make_pair(NextAction::SaveData, std::unique_ptr<Scene>(new SaveDataScene()))
        );
        m_ApplicationSceneList.insert(
            std::make_pair(NextAction::Headline, std::unique_ptr<Scene>(new HeadlineScene()))
        );
    }

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    nn::oe::Initialize();

    // (SIGLO-48495) 必要なライブラリの初期化処理
    nn::nifm::Initialize();
    nn::account::Initialize();

    // 通信関連で必要なライブラリの初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfigWithMemory));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ssl::Initialize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

    // RomFs 内に保存されているはずのメタ情報一覧ファイルを読み込ませる
    ReadMetaInfoListFileData();

    // EC サーバーアクセス機能の初期化
    EcServerAccessor::GetInstance().Initialize();

    this->SetStartupOperationEnvironmentValue();

    // Joy-Con操作に対応する
    InitializeHidController();
    // タッチスクリーンも使用できるようにする
    nn::hid::InitializeTouchScreen();

    // 画面オブジェクトの初期化処理
    nn::hid::TouchScreenState<1> touchState = { 0 };
    for (auto& scene : m_ApplicationSceneList)
    {
        scene.second->Setup(&m_DebugFontWriter, touchState, scene.first);
    }

    // (SIGLO-52835) BCAT の設定が有効な場合は bcat の初期化処理を実行
    // (MetaInfoScene の Setup が終わらないと有効であるかどうか判定できないためこの位置で初期化処理を実行する)
    if (IsBcatSettingEnabled == true)
    {
        nn::bcat::Initialize();
    }

    // TIME ライブラリの初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());

    // トップページを Headline に変える
    auto iter = m_ApplicationSceneList.find(NextAction::Headline);
    m_CurrentScenePtr->ResetState();
    m_CurrentScenePtr->LeaveScene();
    m_CurrentScenePtr = iter->second.get();
    m_CurrentScenePtr->EnterScene();
}

void Application::Finalize() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // 通信関連で必要なライブラリの終了
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ec::FinalizeForShopServiceAccessors());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ssl::Finalize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Finalize());
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

    // EC サーバーアクセス機能の終了
    EcServerAccessor::GetInstance().Finalize();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());

    FinalizeHidController();
    FinalizeDebugFontWriter(&m_DebugFontWriter);
    FinalizeGraphicSystem();
}

void Application::SetStartupOperationEnvironmentValue() NN_NOEXCEPT
{
    // (全画面共通)起動ストレージ情報を設定しておく
    LaunchStorageInfo lsi;
    this->GetLaunchStorageInfo(&lsi);
    m_CurrentScenePtr->SetLaunchStorageInfo(std::move(lsi));

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // 監視対象の Mode を取得・設定しておく
    m_OperationEnvScenePtr->SetOperationModeValue(nn::oe::GetOperationMode());
    m_OperationEnvScenePtr->SetPerformanceModeValue(nn::oe::GetPerformanceMode());
    m_OperationEnvScenePtr->SetFocusStateValue(nn::oe::GetCurrentFocusState());
    // 各モードが変化したことを通知する設定にしておく
    nn::oe::SetOperationModeChangedNotificationEnabled(true);
    nn::oe::SetPerformanceModeChangedNotificationEnabled(true);
#else // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // Generic 版は固定値を設定しておく
    m_OperationEnvScenePtr->SetOperationModeValue(nn::oe::OperationMode_Handheld);
    m_OperationEnvScenePtr->SetPerformanceModeValue(nn::oe::PerformanceMode_Invalid);
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
}

void Application::GetLaunchStorageInfo(LaunchStorageInfo* outStorageInfo) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    using namespace nn::oe;

    // 起動ストレージ情報を取得
    LaunchStorageInfoForDebug lsi;
    GetLaunchStorageInfoForDebug(&lsi);
    //NN_LOG("GetLaunchStorageInfo Launch : %d\n", lsi.launchStorage);
    //NN_LOG("GetLaunchStorageInfo Patch  : %d\n", lsi.patchStorage);

    // 表示用の文字列へ変換する
    struct StorageConvPair
    {
        StorageTypeForDebug num;
        const char* str;
    } storageConvList[] = {
        { StorageTypeForDebug_Unknown,        "Unknown" },
        { StorageTypeForDebug_None,           "None" },
        { StorageTypeForDebug_Host,           "Host" },
        { StorageTypeForDebug_GameCard,       "GameCard" },
        { StorageTypeForDebug_BuiltInStorage, "BuiltIn" },
        { StorageTypeForDebug_SdCard,         "SdCard" },
    };

    // 文字列変換のラムダ式定義
    auto convAction = [&] (StorageTypeForDebug inEnumType)
    {
        for (auto& conv : storageConvList)
        {
            if (inEnumType == conv.num)
            {
                return conv.str;
            }
        }
        // ここに来ることはないと思うが、一応 Unknown を返しておく
        return "Unknown";
    };

    outStorageInfo->launch = convAction(lsi.launchStorage);
    outStorageInfo->patch = convAction(lsi.patchStorage);
#else // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // Generic 版は適当な文字列を設定しておく
    outStorageInfo->launch = "TestLaunch";
    outStorageInfo->patch = "TestPatch";
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
}

void Application::PollOperationModeChangeMessage() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // Operation関連のモードが変化したかどうかを監視する
    nn::oe::Message msg;
    const auto isChanged = nn::oe::TryPopNotificationMessage(&msg);
    if (isChanged == false)
    {
        // 変化がなければすぐに返る
        return;
    }

    // 変化があれば、OperationEnvironment 画面に設定しておく
    if (msg == nn::oe::MessageOperationModeChanged)
    {
        //NN_LOG("PollOperationModeChangeMessage() MessageOperationModeChanged\n");
        m_OperationEnvScenePtr->SetOperationModeValue(nn::oe::GetOperationMode());
    }
    else if (msg == nn::oe::MessagePerformanceModeChanged)
    {
        //NN_LOG("PollOperationModeChangeMessage() MessagePerformanceModeChanged\n");
        m_OperationEnvScenePtr->SetPerformanceModeValue(nn::oe::GetPerformanceMode());
    }
    else if (msg == nn::oe::MessageFocusStateChanged)
    {
        //NN_LOG("PollOperationModeChangeMessage() MessageFocusStateChanged\n");
        m_OperationEnvScenePtr->SetFocusStateValue(nn::oe::GetCurrentFocusState());
    }
#else // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // Generic 版ではひとまず何もしない
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
}

void Application::Run() NN_NOEXCEPT
{
    for(;;)
    {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        // WindowsMessege の処理
        {
            MSG  msg;
            if ( PeekMessage( &msg, NULL, 0, 0, PM_REMOVE) )
            {
                TranslateMessage(&msg);

                if ( msg.message == WM_QUIT)
                {
                    break;
                }

                DispatchMessage(&msg);
            }
        }
#endif // defined(NN_BUILD_CONFIG_OS_WIN32)

        nn::hid::TouchScreenState<1> touchState = { 0 };
        nn::hid::GetTouchScreenState(&touchState);

        UpdateHidController();

        m_CurrentScenePtr->HandleNPad();
        m_CurrentScenePtr->HandleTouchScreen(touchState);

        // オペレーション環境モードの監視処理
        this->PollOperationModeChangeMessage();

        const auto nextAction = m_CurrentScenePtr->Process();
        if (nextAction == NextAction::Exit)
        {
            // Exitの場合はループ処理を抜ける
            return;
        }
        if (EcServerAccessor::GetInstance().Update())
        {
            EcServiceSceneCommon::UpdaetAccount();
            for (auto& scene : m_ApplicationSceneList)
            {
                scene.second->UpdateContents();
            }
        }

        if (nextAction != NextAction::None)
        {
            // Noneでなければ現在の画面を指定の画面に変える
            auto iter = m_ApplicationSceneList.find( nextAction );
            if (iter != std::end(m_ApplicationSceneList))
            {
                m_CurrentScenePtr->ResetState();
                m_CurrentScenePtr->LeaveScene();
                m_CurrentScenePtr = iter->second.get();
                m_CurrentScenePtr->EnterScene();
            }
        }

        // 画面描画処理
        m_CurrentScenePtr->DrawDebugText(&m_DebugFontWriter);
        ProcFrame(&m_DebugFontWriter);
    }
}
