﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>

// nn::album 系モジュールは Generic では扱わない
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nn/album.h>
#include <nn/album/album_Result.h>
#include <nn/album/album_SaveScreenshot.h>
#include <nn/album/album_Applet.h>
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

#include "TestAppSimple_AlbumScene.h"
#include "TestAppSimple_DotText.h"

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
// 保存する画面写真のサイズ。
// サポートされる画面写真のサイズはファームウェアにより決められています。
const nn::album::ImageSize ScreenShotSize = nn::album::ImageSize_1280x720;

// 保存結果の通知設定。
// 画像をアルバムに保存した結果をユーザーに自動的に通知する条件を選ぶことができます。
// 自動的に通知しなかった場合でも API の戻り値によってアプリケーションが独自の通知を行うことができます。
const nn::album::AlbumReportOption ReportOption = nn::album::AlbumReportOption_ReportAlways;
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

namespace {
    const float Label_X_Pos = 60.0f;
    const float Label_Y_Pos = 130.0f;
    const float Label_Y_Width = 90.0f;

    const int PixelSize = 4;
    // 画面写真として保存する画像データを生成します。
    // 画像データの色フォーマットは R8G8B8A8 ですが、アルファチャンネルは無視されます。
    uint8_t* CreateImageData(size_t* pOutSize, int width, int height) NN_NOEXCEPT
    {
        size_t size = PixelSize * width * height;
        uint8_t* pImage = reinterpret_cast<uint8_t*>(malloc(size));
        NN_ASSERT_NOT_NULL(pImage);

        for (int y = 0; y < height; y++)
        {
            uint8_t* pLine = pImage + PixelSize * width * y;
            for (int x = 0; x < width; x++)
            {
                uint8_t* pPixel = pLine + PixelSize * x;

                pPixel[0] = (static_cast<uint8_t>(127 * (static_cast<float>(y) / height)));   // R
                pPixel[1] = (static_cast<uint8_t>(127 * (static_cast<float>(x) / width)));   // G
                pPixel[2] = 127; // B
                pPixel[3] = 255; // A （無視される）
            }
        }

        *pOutSize = size;
        return pImage;
    }

    void DestroyImageData(uint8_t* pImage) NN_NOEXCEPT
    {
        free(pImage);
    }
};

AlbumScene::AlbumScene() NN_NOEXCEPT
    : m_State(State_None), m_SelectedLauchType(LaunchType_None),
    m_CurrentSelectPos(0), m_CurrentMaxPosNum(MaxItemCount - 1)
{
}

void AlbumScene::InternalSetup() NN_NOEXCEPT
{
    // 表示するラベルと対応する呼び出し API の enum 値をまとめて設定
    {
        struct labelPair
        {
            const char* label;
            LaunchType  type;
        } labelList[MaxItemCount] = {
            { "Save Screenshot", LaunchType_SaveScreenshot },
            { "Save And Share Screenshot ", LaunchType_SaveAndShareScreenshot },
            { "Show Album", LaunchType_ShowAlbum },
            //(SIGLO-77517)動画を撮影する機能で実装予定(MaxItemCountも増やす)
            //{ "Record Movie", LaunchType_RecordMovie },
        };

        for (int i = 0; i < MaxItemCount; ++i)
        {
            m_LaunchRangeList[i].range.pos.x = Label_X_Pos;
            m_LaunchRangeList[i].range.pos.y = Label_Y_Pos + (Label_Y_Width * i);
            m_LaunchRangeList[i].range.labelStr = labelList[i].label;
            m_LaunchRangeList[i].type = labelList[i].type;

            if (labelList[i].type == LaunchType_SaveScreenshot)
            {
                m_SaveScreenshotListIndex = i;
            }
            else if (labelList[i].type == LaunchType_SaveAndShareScreenshot)
            {
                m_SaveAndShareScreenshotListIndex = i;
            }
        }
    }

    m_IsOnceSaveScreenshot = false;
    m_IsOnceSaveAndShareScreenshot = false;
    // 念のため Success を入れておく
    m_SaveScreenshotResult = nn::ResultSuccess();
    m_SaveAndShareScreenshotResult = nn::ResultSuccess();

    // 経過時間測定のための変数初期化
    m_SaveScreenshotFrame = 0;
    m_SaveAndShareScreenshotFrame = 0;
}

void AlbumScene::InternalHandleNPad() NN_NOEXCEPT
{
    if (m_State != State_None)
    {
        // State_None 以外は受け付けないようにしておく
        return;
    }

    if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
    {
        m_State = State_Launching;
        m_SelectedLauchType = m_LaunchRangeList[m_CurrentSelectPos].type;
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Down::Mask)
        || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLDown::Mask))
    {
        if (m_CurrentSelectPos < m_CurrentMaxPosNum)
        {
            ++m_CurrentSelectPos;
        }
        else
        {
            m_CurrentSelectPos = 0;
        }
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Up::Mask)
        || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLUp::Mask))
    {
        if (m_CurrentSelectPos > 0)
        {
            --m_CurrentSelectPos;
        }
        else
        {
            m_CurrentSelectPos = m_CurrentMaxPosNum;
        }
    }
}

void AlbumScene::InternalHandleTouchScreen() NN_NOEXCEPT
{
    if (m_State != State_None)
    {
        // State_None 以外は受け付けないようにしておく
        return;
    }

    int count = 0;
    for (auto& lr : m_LaunchRangeList)
    {
        if (lr.range.range.IsInRange(m_PreviousTouch) == true)
        {
            m_State = State_Launching;
            m_SelectedLauchType = lr.type;
            m_CurrentSelectPos = count;
            break;
        }
        ++count;
    }
}

void AlbumScene::InternalProcess() NN_NOEXCEPT
{
    if (m_State == State_Launching)
    {
        // nn::album 系モジュールは Generic では扱わない
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        switch (m_SelectedLauchType)
        {
        case LaunchType_SaveScreenshot:
            {
                m_SaveScreenshotResult = ExecuteSaveScreenshot(false);
                m_IsOnceSaveScreenshot = true;

                m_SaveScreenshotFrame = 0;
            }
            break;
        case LaunchType_SaveAndShareScreenshot:
            {
                m_SaveAndShareScreenshotResult = ExecuteSaveScreenshot(true);
                m_IsOnceSaveAndShareScreenshot = true;

                m_SaveAndShareScreenshotFrame = 0;
            }
            break;
        case LaunchType_ShowAlbum:
            {
                nn::album::ShowAlbumFiles();
            }
            break;
        case LaunchType_RecordMovie:
            {

            }
            break;
        case LaunchType_None:
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        m_State = State_None;
    }
}

void AlbumScene::InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetTextColor(White);
    writer->SetCursor(40.0f, 75.0f);
    writer->Print("[Album]");

    writer->SetScale(1.8f, 1.8f);
    {
        writer->SetTextColor(White);
        writer->SetCursor(Label_X_Pos - 30.0f, Label_Y_Pos + (Label_Y_Width * (m_CurrentSelectPos)));
        writer->Print("*");
    }

    for (auto& lr : m_LaunchRangeList)
    {
        this->WriteTouchRange(writer, &lr.range);
    }

    if (m_State == State_None)
    {
        writer->SetTextColor(White);
        writer->SetCursor(100.0f, 600.0f);
        writer->Print("A: Execute");

        // (SIGLO-77516)スクリーンショット撮影した際のresult値の表示
        auto getResultString = [&](nn::Result result, bool isExecuted, int elapsedFrame) {
            std::string resultStr = "";
            if (isExecuted == false)
            {
                writer->SetTextColor(White);
                resultStr = "Result : ---------- ";
            }
            else
            {
                // 同結果の場合result表示部が変わらず処理が行われたか分かり辛いので疑似的に"Processing..."の表記を行う
                if (elapsedFrame >= 60)
                {
                    if (result.IsFailure() == false)
                    {
                        writer->SetTextColor(Green);
                        resultStr = "Result : 0x%08x (Success)";
                    }
                    else
                    {
                        writer->SetTextColor(Red);
                        resultStr = "Result : 0x%08x (Failure)";
                    }
                }
                else
                {
                    writer->SetTextColor(Yellow);
                    resultStr = "Processing...";
                }
            }
            return resultStr;
        };

        // "Save Screenshot"のreult表示
        std::string resultString = getResultString(m_SaveScreenshotResult, m_IsOnceSaveScreenshot, m_SaveScreenshotFrame++);
        writer->SetCursor(640.0f, m_LaunchRangeList[m_SaveScreenshotListIndex].range.pos.y);
        writer->Print(resultString.c_str(), m_SaveScreenshotResult.GetInnerValueForDebug());

        // "Save And Share Screenshot"のreult表示
        resultString = getResultString(m_SaveAndShareScreenshotResult, m_IsOnceSaveAndShareScreenshot, m_SaveAndShareScreenshotFrame++);
        writer->SetCursor(640.0f, m_LaunchRangeList[m_SaveAndShareScreenshotListIndex].range.pos.y);
        writer->Print(resultString.c_str(), m_SaveAndShareScreenshotResult.GetInnerValueForDebug());
    }
}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
nn::Result AlbumScene::ExecuteSaveScreenshot(bool isShare) NN_NOEXCEPT
{
    // 保存する画像データを生成します。
    size_t imageDataSize = 0;
    nn::album::Initialize();
    auto imageWidth = nn::album::GetImageWidth(ScreenShotSize);
    auto imageHeight = nn::album::GetImageHeight(ScreenShotSize);
    auto pImageData = CreateImageData(&imageDataSize, imageWidth, imageHeight);

    // 画像への表題追記
    const int TextOffsetX = 32;
    const int TextOffsetY = 32;
    const int LinesHeight = 64;
    int lineNumber = 0;
    DrawText(
        pImageData, imageWidth, imageHeight, PixelSize,
        "MakeTestApplication", TextOffsetX, TextOffsetY + LinesHeight * lineNumber++
    );

    // 画像へのタイムスタンプ追記
    nn::time::CalendarTime calendarTime = TimeStamp::GetCalendarTime();
    std::string timeString = "";
    timeString += Convert::ToStringFillZero(calendarTime.year, 4);
    timeString += "/";
    timeString += Convert::ToStringFillZero(calendarTime.month, 2);
    timeString += "/";
    timeString += Convert::ToStringFillZero(calendarTime.day, 2);
    timeString += " ";
    timeString += Convert::ToStringFillZero(calendarTime.hour, 2);
    timeString += ":";
    timeString += Convert::ToStringFillZero(calendarTime.minute, 2);
    timeString += ":";
    timeString += Convert::ToStringFillZero(calendarTime.second, 2);
    DrawText(
        pImageData, imageWidth, imageHeight, PixelSize,
        timeString, TextOffsetX, TextOffsetY + LinesHeight * lineNumber++
    );

    // 画像へのApplicationId追記
    std::string applicationIdString = "ApplicationId : ";
    applicationIdString += m_ApplicationInfo.id;
    DrawText(
        pImageData, imageWidth, imageHeight, PixelSize,
        applicationIdString, TextOffsetX, TextOffsetY + LinesHeight * lineNumber++
    );

    // 画像へのReleaseVersion追記
    std::string releaseVersionString = "ReleaseVersion : ";
    releaseVersionString += std::to_string(m_ApplicationInfo.ver.releaseVer);
    DrawText(
        pImageData, imageWidth, imageHeight, PixelSize,
        releaseVersionString, TextOffsetX, TextOffsetY + LinesHeight * lineNumber++
    );

    // スクリーンショットの保存
    nn::Result result;
    if (isShare == false)
    {
        result = nn::album::SaveScreenshot(
            pImageData,
            imageDataSize,
            ScreenShotSize,
            ReportOption
        );
    }
    else
    {
        result = nn::album::SaveAndShareScreenshot(
            pImageData,
            imageDataSize,
            ScreenShotSize,
            ReportOption,
            nn::account::InvalidUid
        );
    }

    // API 呼出後、保存した画像データは破棄できます。
    DestroyImageData(pImageData);

    // アルバムライブラリを終了します。
    nn::album::Finalize();

    // 表示検証用・意図的なresult値を返す
    //result = nn::result::detail::ResultInternalAccessor::ConstructResult(1);

    return result;
}
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
