﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>

#include "TestAppSimple_FsUtilities.h"
#include "TestAppSimple_AbortDebugScene.h"

#if defined(NN_BUILD_CONFIG_OS_WIN32)
// スタックオーバーフローのコンパイラ警告を回避するための措置
#pragma warning ( disable : 4717 )
#endif // NN_BUILD_CONFIG_OS_WIN32

namespace {
    const char* const AbortFilePath = "Contents:/AbortResult.txt";

    const char* const NullPtrAcceseStr = "NullPointerAccess";
    const char* const StackOverflowStr = "StackOverflow";
    const char* const QuickExitStr = "Call std::quick_exit(0)";

    const float Label_X_Pos = 60.0f;
    const float Label_Y_Pos = 130.0f;
    const float Label_Y_Width = 90.0f;
}

AbortDebugScene::AbortDebugScene() NN_NOEXCEPT
    : m_State(State_None), m_CurrentSelectPos(0), m_CurrentMaxPosNum(1), m_SelectedRangePtr(nullptr)
{
}

void AbortDebugScene::InternalSetup() NN_NOEXCEPT
{
    int ypos = 0;
    {
        m_NullPointerAccessRange.pos.x = Label_X_Pos;
        m_NullPointerAccessRange.pos.y = Label_Y_Pos + (Label_Y_Width * ypos++);
        m_NullPointerAccessRange.labelStr = NullPtrAcceseStr;

        m_DisplayRangePtrList.push_back(&m_NullPointerAccessRange);
    }
    {
        m_StackOverflowRange.pos.x = Label_X_Pos;
        m_StackOverflowRange.pos.y = Label_Y_Pos + (Label_Y_Width * ypos++);
        m_StackOverflowRange.labelStr = StackOverflowStr;

        m_DisplayRangePtrList.push_back(&m_StackOverflowRange);
    }
    {
        m_QuickExitRange.pos.x = Label_X_Pos;
        m_QuickExitRange.pos.y = Label_Y_Pos + (Label_Y_Width * ypos++);
        m_QuickExitRange.labelStr = QuickExitStr;

        m_DisplayRangePtrList.push_back(&m_QuickExitRange);
    }

    if (fsutil::IsExistPath(AbortFilePath) == true)
    {
        // Abort指定のResult値を読み込んでおく
        auto result = this->ReadAbortResultValue();
        // Abort実行処理開始タッチ位置の定義
        if (result.IsSuccess())
        {
            m_DoAbortRange.pos.x = Label_X_Pos;
            m_DoAbortRange.pos.y = Label_Y_Pos + (Label_Y_Width * ypos++);
            m_DoAbortRange.labelStr = "Abort [" + m_AbortResultValue + "]";

            m_DisplayRangePtrList.push_back(&m_DoAbortRange);
        }
    }

    m_CurrentMaxPosNum = static_cast<int>(m_DisplayRangePtrList.size() - 1);
    m_SelectedRangePtr = m_DisplayRangePtrList[m_CurrentSelectPos];

    m_WarningView.SetTitle("!! WARNING !!");
    m_WarningView.PushMessage("Do you really want to execute \"", White);
    m_AbortMsgIdx = m_WarningView.PushMessage(m_SelectedRangePtr->labelStr.c_str(), Coral);
    m_WarningView.PushMessage("\" ?", White);
}

nn::Result AbortDebugScene::ReadAbortResultValue() NN_NOEXCEPT
{
    fsutil::File file;

    auto result = file.Open(AbortFilePath, nn::fs::OpenMode_Read);
    if (result.IsFailure())
    {
        NN_LOG("[Error] ReadAbortResultValue()  FileOpen(): result = 0x%08x\n", result.GetInnerValueForDebug());
        return result;
    }

    const auto fileSize = file.GetSize();
    if (fileSize <= 0)
    {
        NN_LOG("[Error] ReadAbortResultValue() : fileSize = %lld\n", fileSize);
        return result;
    }

    const size_t bufSize = static_cast<size_t>(fileSize - 3) + 1;
    std::unique_ptr<char[]> buf(new char[bufSize]);
    *(buf.get() + (bufSize - 1)) = '\0';

    size_t outSize = 0;
    size_t readSize = bufSize - 1;
    // UTF8のBOM付きファイルであることを想定(オフセットの3バイトはそのため)
    result = file.Read(&outSize, 3, buf.get(), readSize);
    if (result.IsFailure())
    {
        // ファイル読み込み処理で失敗
        NN_LOG("[Error] ReadAbortResultValue()  FileRead(): result = 0x%08x\n", result.GetInnerValueForDebug());
        return result;
    }

    // 読み込んだ Reault 値を保持しておく
    m_AbortResultValue = buf.get();

    return result;
}

void AbortDebugScene::InternalHandleNPad() NN_NOEXCEPT
{
    if (m_State == State_None)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
        {
            m_State = State_AbortReady;
            m_SelectedRangePtr = m_DisplayRangePtrList[m_CurrentSelectPos];
            m_WarningView.UpdateMessage(m_AbortMsgIdx, m_SelectedRangePtr->labelStr.c_str());
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Down::Mask)
            || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLDown::Mask))
        {
            if (m_CurrentSelectPos < m_CurrentMaxPosNum)
            {
                ++m_CurrentSelectPos;
            }
            else
            {
                m_CurrentSelectPos = 0;
            }
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Up::Mask)
            || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLUp::Mask))
        {
            if (m_CurrentSelectPos > 0)
            {
                --m_CurrentSelectPos;
            }
            else
            {
                m_CurrentSelectPos = m_CurrentMaxPosNum;
            }
        }
    }
    else if (m_State == State_AbortReady)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
        {
            m_State = State_Aborting;
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask))
        {
            m_State = State_None;
        }
    }
}

void AbortDebugScene::InternalHandleTouchScreen() NN_NOEXCEPT
{
    if (m_State == State_None)
    {
        int count = 0;
        for (auto ptr : m_DisplayRangePtrList)
        {
            if (ptr->range.IsInRange(m_PreviousTouch) == true)
            {
                m_State = State_AbortReady;
                m_SelectedRangePtr = ptr;
                m_WarningView.UpdateMessage(m_AbortMsgIdx, m_SelectedRangePtr->labelStr.c_str());

                m_CurrentSelectPos = count;

                break;
            }
            ++count;
        }
    }
    else if (m_State == State_AbortReady)
    {
        if (m_ConfirmOkRange.range.IsInRange(m_PreviousTouch))
        {
            m_State = State_Aborting;
        }
        else if (m_ConfirmCancelRange.range.IsInRange(m_PreviousTouch))
        {
            m_State = State_None;
        }
    }
}

void AbortDebugScene::InternalProcess() NN_NOEXCEPT
{
    if (m_State == State_Aborting)
    {
        if (m_SelectedRangePtr->labelStr == NullPtrAcceseStr)
        {
            this->ExecuteNullPointerAccess();
        }
        else if (m_SelectedRangePtr->labelStr == StackOverflowStr)
        {
            this->ExecuteStackOverflow();
        }
        else if (m_SelectedRangePtr->labelStr.find("Abort [") != std::string::npos)
        {
            this->ExecuteAbortResult();
        }
        else if (m_SelectedRangePtr->labelStr == QuickExitStr)
        {
            this->ExecuteQuickExit();
        }

        // 実質的にこの処理が実行されることは無いと思うが・・
        m_State = State_None;
    }
}

void AbortDebugScene::ExecuteNullPointerAccess() NN_NOEXCEPT
{
    NN_LOG("[Trace] Execute NullPointerAccess\n");
    // わざとNULLアクセスを発生させる
    int* ptr = nullptr;
    *ptr = 1;
}

int recursiveCallB() NN_NOEXCEPT;
int recursiveCallA() NN_NOEXCEPT
{
    NN_LOG("[Trace] Call recursiveCallA\n");
    return recursiveCallB();
}

int recursiveCallB() NN_NOEXCEPT
{
    NN_LOG("[Trace] Call recursiveCallB\n");
    return recursiveCallA();
}

void AbortDebugScene::ExecuteStackOverflow() NN_NOEXCEPT
{
    NN_LOG("[Trace] Execute StackOverflow 01\n");
    // わざとスタックオーバーフローを発生させる
    // ひとまず無限再帰で実現させる
    recursiveCallA();

    NN_LOG("[Trace] Execute StackOverflow 02\n");
}

void AbortDebugScene::ExecuteAbortResult() NN_NOEXCEPT
{
    NN_LOG("[Trace] Execute AbortResult\n");
    // わざと Abort を発生させる
    auto result = nn::result::detail::ResultInternalAccessor::ConstructResult(
        Convert::ToUInt32(m_AbortResultValue));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void AbortDebugScene::ExecuteQuickExit() NN_NOEXCEPT
{
    NN_LOG("[Trace] Execute QuickExit\n");
    // (SIGLO-62328) 明示的に std::quick_exit(0) を呼び出す
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    // VS2013 でのビルドを考慮して Generic 版では _exit(0) を呼び出す形にする
    _exit(0);
#else // defined(NN_BUILD_CONFIG_OS_WIN32)
    // Generic 版以外では std::quick_exit(0) を呼び出す
    std::quick_exit(0);
#endif // defined(NN_BUILD_CONFIG_OS_WIN32)
}

void AbortDebugScene::InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetTextColor(White);
    writer->SetCursor(40.0f, 75.0f);
    writer->Print("[AbortDebug]");

    if (m_State == State_AbortReady)
    {
        // 警告画面の描画
        this->DrawWarningMessage(writer, &m_WarningView);

        // 以降の処理は書き出さない
        return;
    }

    writer->SetScale(1.8f, 1.8f);
    {
        writer->SetTextColor(White);
        writer->SetCursor(Label_X_Pos - 30.0f, Label_Y_Pos + (Label_Y_Width * (m_CurrentSelectPos)));
        writer->Print("*");
    }

    for (auto ptr : m_DisplayRangePtrList)
    {
        this->WriteTouchRange(writer, ptr);
    }

    writer->SetTextColor(White);
    writer->SetCursor(100.0f, 600.0f);
    writer->Print("A: Execute");
}
