﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

namespace MakeTestApplication
{
    internal class TestAppBuilder
    {
        private const string TestAppSrcMiddleDirPath = @"Tests\Sources\Tools\MakeTestApplication\TestApplicationSimple\";

        private bool isVerbose = false;
        public void SetVerboseMode()
        {
            isVerbose = true;
        }

        /// <summary>
        /// code ディレクトリの存在確認などのチェック処理を実施
        /// エラーの場合は例外を投げる
        /// </summary>
        /// <param name="inCodeDir">チェック対象のディレクトリパス</param>
        public void CheckExistCodeData(string inCodeDir)
        {
            // code ディレクトリの存在確認のみを実施する
            if (Directory.Exists(inCodeDir) == false)
            {
                throw new Exception("テストアプリケーションのコードディレクトリが存在しません");
            }
        }

        /// <summary>
        /// テストアプリケーションが参照するmeta情報一覧ファイルやダミーデータハッシュ情報をファイルに書き出す
        /// </summary>
        /// <param name="inParam">ファイル書き込み用の入力情報</param>
        /// <param name="outputDirPath">ファイルを書き出すディレクトリパス</param>
        public void CreateSettingParamFile(TestAppSettingParam inParam, string outputDirPath)
        {
            // SIGLO-27640 リビルドを行わない対応
            try
            {
                // ハッシュ文字列をファイルに書き込み (UTF8のBOM付き形式で保存)
                FsFacade.CreateHashValueFile(inParam.SHA1HashString, outputDirPath);

                {
                    // メタ情報一覧のファイル書き込み用の文字列を作成する
                    var sb = new StringBuilder();
                    foreach (var pr in inParam.MetaSetList)
                    {
                        sb.Append(string.Format("\"{0}\",\"{1}\"\r\n", pr.Item, pr.Value));
                    }

                    // メタ情報一覧をファイルに書き込み (UTF8のBOM付き形式で保存)
                    var metaListFilePath = Path.Combine(outputDirPath, @"MetaList.txt");
                    FsFacade.CreateRecordFile(metaListFilePath, sb.ToString());
                }

                // OfflineHtmlPathList が設定されていればファイル作成処理を実施
                if (inParam.OfflineHtmlPathList != null && inParam.OfflineHtmlPathList.Count > 0)
                {
                    var sb = new StringBuilder();
                    foreach (var path in inParam.OfflineHtmlPathList)
                    {
                        sb.Append(string.Format("{0}\r\n", path));
                    }

                    // オフラインWebファイルパスをファイルに書き込み (UTF8のBOM付き形式で保存)
                    var htmlFilePath = Path.Combine(outputDirPath, @"HtmlPath.txt");
                    FsFacade.CreateRecordFile(htmlFilePath, sb.ToString());
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }
        }

        private const string DotNetDirPath = @"C:\WINDOWS\Microsoft.NET\Framework\";
        private const string DotNetVer4DirName = @"v4.0.30319";
        private const string DotNetVer35DirName = @"v3.5";
        private const string MSBuildExeName = @"MSBuild.exe";

        private void Process_DataReceived(object sender, DataReceivedEventArgs e)
        {
            Console.WriteLine(e.Data);
        }

        private void Process_ErrorDataReceived(object sender, DataReceivedEventArgs e)
        {
            Console.Error.WriteLine(e.Data);
        }

        /// <summary>
        /// MSBuild.exe を実行する処理
        /// </summary>
        /// <param name="outputDirPath">出力フォルダのパス</param>
        /// <param name="intermediatesDirPath">中間フォルダのパス</param>
        /// <param name="isCleanOnly">クリーンのみの実行フラグ</param>
        /// <returns></returns>
        public int ExecuteMSBuild(string outputDirPath, string intermediatesDirPath, bool isCleanOnly = false)
        {
            if (isVerbose == true)
            {
                var outputEvent = new DataReceivedEventHandler(Process_DataReceived);
                var outputErrorEvent = new DataReceivedEventHandler(Process_ErrorDataReceived);
                return ExecuteMSBuild(outputDirPath, intermediatesDirPath, outputEvent, outputErrorEvent, isCleanOnly);
            }
            else
            {
                return ExecuteMSBuild(outputDirPath, intermediatesDirPath, null, null, isCleanOnly);
            }
        }

        private int ExecuteMSBuild(string outputDirPath,
                           string intermediatesDirPath,
                           DataReceivedEventHandler outputDataReceivedEventHandler,
                           DataReceivedEventHandler errorDataReceivedEventHandler,
                           bool isCleanOnly)
        {
            var msBuildPath = string.Empty;
#if false
            try
            {
                // .Net4 系の存在確認
                var dotNetPath = DotNetDirPath + DotNetVer4DirName;
                if (Directory.Exists(dotNetPath) == false)
                {
                    // .Net3.5 系の存在確認
                    dotNetPath = DotNetDirPath + DotNetVer35DirName;
                    if (Directory.Exists(dotNetPath) == false)
                    {
                        // 通常の開発環境ではここに来ることはまずありえないと思うが・・
                        Console.Error.WriteLine(".NetFramework 3.5 以上をインストールしてください");
                        return -1;
                    }
                }

                msBuildPath = dotNetPath + "\\" + MSBuildExeName;
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                return -1;
            }
#else
            msBuildPath = @"C:\Program Files (x86)\MSBuild\14.0\Bin\" + MSBuildExeName;
#endif

            var projectFilePath = CommonParams.GetSigloRoot() + TestAppSrcMiddleDirPath + @"TestAppSimple-spec.NX.vcxproj";
            using (var prc = new Process())
            {
                prc.StartInfo.CreateNoWindow = true;
                prc.StartInfo.UseShellExecute = false;
                prc.StartInfo.RedirectStandardError = true;
                //prc.StartInfo.StandardErrorEncoding = System.Text.Encoding.UTF8;
                prc.StartInfo.FileName = msBuildPath;

                var args = new StringBuilder();
                if (isCleanOnly)
                {
                    args.Append(" /t:Clean");
                }
                else
                {
                    args.Append(" /t:Rebuild");
                }
                // VS2015対応
                //args.Append(" /p:TargetFrameworkVersion=v4.5.1");
                args.Append(" /p:PlatformToolset=v140");
                args.Append(" /p:VisualStudioVersion=14.0");
                args.Append(" /toolsversion:14.0");
                //args.Append(" /verbosity:diagnostic");
                args.AppendFormat(" /p:Platform={0}", "NX64");  // プラットフォーム
                args.AppendFormat(" /p:Configuration={0}", "NX_Release");  // ビルド構成
                args.AppendFormat(" /p:OutDir={0}", outputDirPath);        // 出力フォルダ
                args.AppendFormat(" /p:IntDir={0}", intermediatesDirPath);        // 中間フォルダ
                args.Append(" /p:BuildProjectReferences=false");           // プロジェクト参照をビルドするか
                args.AppendFormat(" /p:DebugType={0}", "none");          // デバッグ情報の出力
                args.AppendFormat(" /p:DebugSymbols={0}", "false");      // pdbファイルを生成するか
                args.AppendFormat(" /p:ApplicationProgramFormat={0}", "Raw");  // 出力フォーマット(NX用)
                //args.AppendFormat(" /p:DocumentationFile={0}", isDebug ? string.Format("{0}.xml", GetAssemblyName(projectFilePath)) : " "); // XMLドキュメントファイル
                args.AppendFormat(" \"{0}\"", projectFilePath);  // プロジェクトファイル
                prc.StartInfo.Arguments = args.ToString();

                if (isVerbose == true)
                {
                    Console.WriteLine("ExecuteMSBuild arg : {0}", prc.StartInfo.Arguments);
                }

                // 出力をハンドリングする場合
                if (outputDataReceivedEventHandler != null || errorDataReceivedEventHandler != null)
                {
                    prc.StartInfo.WindowStyle = ProcessWindowStyle.Hidden;

                    if (outputDataReceivedEventHandler != null)
                    {
                        prc.StartInfo.RedirectStandardOutput = true;
                        prc.OutputDataReceived += outputDataReceivedEventHandler;
                    }

                    if (errorDataReceivedEventHandler != null)
                    {
                        prc.StartInfo.RedirectStandardError = true;
                        prc.ErrorDataReceived += errorDataReceivedEventHandler;
                    }
                }

                // MSBuild実行
                prc.Start();

                // 出力の非同期読み取り
                if (outputDataReceivedEventHandler != null)
                {
                    prc.BeginOutputReadLine();
                }

                // エラー出力の非同期読み取り
                if (errorDataReceivedEventHandler != null)
                {
                    prc.BeginErrorReadLine();
                }

                // MSBuild実行完了まで待機
                prc.WaitForExit();

                // MSBuildの戻り値を返す
                return prc.ExitCode;
            }
        }
    }
}
