﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Reflection;

namespace PCToolTester
{
    public static class PathUtility
    {
        /// <summary>
        /// 相対パスを処理します。
        /// </summary>
        public static string GetFullFilePath(string basePath, string filePath)
        {
            Debug.Assert(string.IsNullOrEmpty(basePath) == false, "basePath is null or Empty!");

            basePath = EnvironmentUtility.ExpandEnvironmentVariables(basePath);
            filePath = EnvironmentUtility.ExpandEnvironmentVariables(filePath);

            if (string.IsNullOrEmpty(filePath) == true || Path.IsPathRooted(filePath) == true)
            {
                return PathUtility.GetFullPath(filePath); // null や Empty や絶対パスならそのまま返す。
            }
            else
            {
                return PathUtility.GetFullPath(Path.Combine(basePath, filePath)); // 相対パスは basePath と結合する。
            }
        }

        private static string GetFullPath(this string path)
        {
            string sourcePath = path;

            // Path.GetFullPath() を直接使用すると
            // ../ 等を含む冗長なパス長に制約がかかるため、
            // Uri.LocalPath で先に最短パスに変換してから、
            // Path.GetFullPath() にてパス長を検証しています。
            // ただし、path が相対パスの場合は Uri で例外が発生するため、
            // path が絶対パスの場合にのみ実行します。
            if (Path.IsPathRooted(path))
            {
                sourcePath = new Uri(path).LocalPath;
            }

            return Path.GetFullPath(sourcePath);
        }

        /// <summary>
        /// 指定パスの先祖ディレクトリが１つでも存在するかを調べます。
        /// </summary>
        /// <param name="fullPath"></param>
        /// <returns></returns>
        public static bool IsAnyAnscestorDirectoryExist(string fullPath)
        {
            if (!Path.IsPathRooted(fullPath))
            {
                throw new ArgumentException("Full path is required.", "fullPath");
            }

            var directoryPath = Path.GetDirectoryName(fullPath);
            while (!string.IsNullOrEmpty(directoryPath))
            {
                if (directoryPath[directoryPath.Length - 1] == Path.DirectorySeparatorChar)
                {
                    // パスルート (C:\) に達した。
                    return false;
                }

                if (Directory.Exists(directoryPath))
                {
                    return true;
                }

                directoryPath = Path.GetDirectoryName(directoryPath);
            }

            return false;
        }
    }
}
