﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using PCToolTester.Resources;
using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;

namespace PCToolTester
{
    public class ConstantDictionary : Dictionary<string, string>
    {
        private const string CircularReferenceMark = "\0CircularReference\0";

        private readonly Dictionary<string, string> _cache = new Dictionary<string, string>();

        public ConstantDictionary(ConstantDictionary fallback = null)
        {
            this.Fallback = fallback;
        }

        public ConstantDictionary Fallback { get; private set; }

        public void Add(Constant[] constants)
        {
            foreach (var c in constants)
            {
                string nameErrorMessage = null;
                if (c.Name == null)
                {
                    nameErrorMessage = MessageResource.Message_Error_Constants_Undefined;
                }
                else if (string.IsNullOrEmpty(c.Name))
                {
                    nameErrorMessage = MessageResource.Message_Error_Constants_Empty;
                }
                else if (this.ContainsKey(c.Name))
                {
                    nameErrorMessage = string.Format(
                        MessageResource.Message_Error_Constants_Duplicate,
                        c.Name);
                }

                string valueErrorMessage = null;
                if (c.Value == null)
                {
                    valueErrorMessage = MessageResource.Message_Error_Constants_Undefined;
                }

                if (nameErrorMessage != null || valueErrorMessage != null)
                {
                    throw new ApplicationException(
                        string.Format(
                            MessageResource.Message_Error_Constants_InvalidDefinition,
                            nameErrorMessage ?? GetQuotedString(c.Name),
                            valueErrorMessage ?? GetQuotedString(c.Value)));
                }

                this.Add(c.Name, c.Value);
            }
        }

        private static string GetQuotedString(string str)
        {
            return '"' + str + '"';
        }

        /// <summary>
        /// 指定した文字列に含まれる定数参照 $(xxx) を定数値に展開します。
        /// </summary>
        /// <param name="sourceStr"></param>
        /// <returns></returns>
        /// <seealso cref="GetExpandedValue"/>
        /// <exception cref="ApplicationException">定数展開できなかった場合。</exception>
        public string ExpandConstants(string sourceStr)
        {
            if (string.IsNullOrEmpty(sourceStr))
            {
                return string.Empty;
            }

            return Regex.Replace(sourceStr, @"\$\(([^)]+)\)", (match) =>
                {
                    var name = match.Groups[1].Value;
                    var value = this.GetExpandedValue(name);
                    if (value != null)
                    {
                        return value;
                    }
                    else
                    {
                        throw new ApplicationException(
                            string.Format(
                                MessageResource.Message_Error_Constants_FailedExpand,
                                name));
                    }
                });
        }

        /// <summary>
        /// 指定した定数の、定数展開された値を得ます。
        /// <para>
        /// 定数値は this、<see cref="Fallabck"/>、環境変数の順で検索されます。
        /// 取得した値に対して <see cref="ExpandConstants()"/> による定数展開が行われます。
        /// </para>
        /// </summary>
        /// <param name="name"></param>
        /// <returns>指定された名前の定数が定義されていない場合は null を返します。</returns>
        /// <exception cref="ApplicationException">定数展開できなかった場合。</exception>
        public string GetExpandedValue(string name, bool allowExpandFail = false)
        {
            string value;

            if (_cache.TryGetValue(name, out value))
            {
                if (value == CircularReferenceMark)
                {
                    throw new ApplicationException(
                        string.Format(
                            MessageResource.Message_Error_Constants_CircularReference,
                            name));
                }

                return value;
            }

            _cache[name] = CircularReferenceMark;

            var dictionary = this;
            while (dictionary != null)
            {
                if (dictionary.TryGetValue(name, out value))
                {
                    if (string.IsNullOrEmpty(value))
                    {
                        value = string.Empty;
                    }
                    break;
                }

                dictionary = dictionary.Fallback;
            }

            if (value == null)
            {
                value = EnvironmentUtility.GetEnvironmentVariable(name);
            }

            if (value != null)
            {
                value = this.ExpandConstants(value);
            }

            _cache[name] = value;
            return value;
        }
    }
}
